<?php

class Crossroads_API_CategoriesController extends Crossroads_API_Controller_Resource
{
    /**
     * @api {get} /categories Get all categories
     * @apiName getCategories
     * @apiGroup Categories
     *
     * @apiParam {Boolean} includeImages  If set to "true" it will include images.
     *
     * @apiSuccess {Object[]} categories Array of the categories
     * @apiSuccess {String} categories.id Category ID
     * @apiSuccess {String} categories.name Name of the category
     * @apiSuccess {String} categories.description Category description
     * @apiSuccess {String} categories.metaDescription
     * @apiSuccess {String} categories.urlKey
     * @apiSuccess {Integer} categories.position
     * @apiSuccess {Integer} categories.level
     * @apiSuccess {Boolean} categories.isAnchor
     * @apiSuccess {String} categories.path
     * @apiSuccess {Integer} categories.parentId
     * @apiSuccess {Boolean} categories.includeInMenu
     */
    public function getAll()
    {
        $req           = $this->getRequest();
        $store_id      = Mage::app()->getStore()->getStoreId();
        $root_id       = Mage::app()->getStore($store_id)->getRootCategoryId();
        $includeImages = $req->has("includeImages") ? strtolower($req->get("includeImages")) === "true" : false;

        $categories = Mage::getModel('catalog/category')
            ->getCollection()
            ->setStoreId($store_id)
            ->addFieldToFilter('is_active', 1)
            ->addAttributeToFilter('path', ['like' => "1/{$root_id}/%"])
            ->addAttributeToSelect('*')
            ->addAttributeToSort("position");
        $helper = Mage::helper("API/category");

        return [200, array_values(array_map(function($c) use($helper, $includeImages) {
            return $helper->prepareCategory($c, $includeImages);
        }, $categories->getItems()))];
    }

    /**
     * @api {get} /categories/:id Get category
     * @apiName getCategory
     * @apiGroup Categories
     *
     * @apiParam {Integer} id Category ID
     *
     * @apiSuccess {Object} category Category
     * @apiSuccess {String} category.id Category ID
     * @apiSuccess {String} category.name Name of the category
     * @apiSuccess {String} category.description Category description
     * @apiSuccess {String} category.metaDescription
     * @apiSuccess {String} category.urlKey
     * @apiSuccess {String} category.metaTitle
     * @apiSuccess {Integer} category.position
     * @apiSuccess {Array} category.images
     * @apiSuccess {Object} category.images.small
     * @apiSuccess {String} category.images.small.url
     * @apiSuccess {String} category.images.small.dataUrl
     * @apiSuccess {Object} category.images.medium
     * @apiSuccess {String} category.images.medium.url
     * @apiSuccess {String} category.images.medium.dataUrl
     * @apiSuccess {Object} category.images.large
     * @apiSuccess {String} category.images.large.url
     * @apiSuccess {String} category.images.large.dataUrl
     * @apiSuccess {Integer} category.level
     * @apiSuccess {Boolean} category.isAnchor
     * @apiSuccess {String} category.path
     * @apiSuccess {Integer} category.parentId
     * @apiSuccess {Boolean} category.includeInMenu
     *
     * @apiSuccess {Object} category.hero
     * @apiSuccess {String} category.hero.heading
     * @apiSuccess {String} category.hero.description
     * @apiSuccess {String} category.hero.offerIcon
     * @apiSuccess {Object} category.hero.button
     * @apiSuccess {String} category.hero.button.text
     * @apiSuccess {String} category.hero.button.link
     *
     * @apiSuccess {Object[]} products Array of the products in the category
     * @apiSuccess {Integer}  products.id Product id
     * @apiSuccess {String}   products.type Product type
     * @apiSuccess {String}   products.sku
     * @apiSuccess {String}   products.urlKey
     * @apiSuccess {Double}   products.price
     * @apiSuccess {Double}   products.specialPrice
     * @apiSuccess {Double}   products.msrp
     * @apiSuccess {Double}   products.discountPercent
     * @apiSuccess {String}   products.shortDescription
     * @apiSuccess {Object}   products.attributes              Key value pairs of the selected attributes in API_section/product_list/attributes
     * @apiSuccess {String}   products.smallImage
     * @apiSuccess {Boolean}  products.isSalable
     * @apiSuccess {Object[]} products.options  Null if simple product, object if complex, can be turned off in list view through System Configuration
     */
    public function getItem($id)
    {
        $category = Mage::getModel('catalog/category')->load($id);

        if( ! $category || !$category->getId() || !$category->getIsActive()) {
            return [404];
        }

        $collection = Mage::getModel('catalog/product')
            ->getCollection()
            ->addCategoryFilter($category)
            ->addAttributeToSelect("*")
            ->addMinimalPrice()
            ->addAttributeToFilter(
                'visibility',
                ['neq' => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE]
            )
            ->load();

        $helper = Mage::helper("API/product");

        return [200, [
            "category" => Mage::helper("API/category")->prepareCategory($category, true),
            "products" => array_values(array_map(function($p) use($helper) {
                return $helper->prepareListProduct($p);
            }, $collection->getItems()))
        ]];
    }
}
