<?php

class Crossroads_API_Helper_Cart
{
    /**
     * Executed after the basic quote associated array has been prepared. This event allows
     * modification of the returned quote associated array.
     *
     * Params:
     *  * quote           The quote object
     *  * totals          The quote totals map, from Mage_Sales_Model_Quote::getTotals()
     *  * prepared_data   The associated array as a varien object, note that the data is in
     *                    camel-case, so use setData and getData to modify.
     */
    const EVENT_QUOTE_POST_DATA_PREPARE = "crossroads_api_quote_post_data_prepare";

    public function getCouponData() {
        $quote  = Mage::getModel('checkout/session')->getQuote();

        if( ! $quote->getCouponCode()) {
            return null;
        }

        $coupon = Mage::getModel("salesrule/coupon")->load($quote->getCouponCode(), "code");

        if(!$coupon || !$coupon->getRuleId()) {
            return null;
        }

        $rule = Mage::getModel("salesrule/rule")->load($coupon->getRuleId());

        return [
            "couponCode" => $quote->getCouponCode(),
            "ruleName"   => $rule->getName(),
        ];
    }

    public function getCart()
    {
        $quote = Mage::getModel('checkout/session')->getQuote();

        $quote->collectTotals();

        return $this->formatQuote($quote);
    }

    public function formatQuote($quote) {
        $items    = $quote->getAllVisibleItems();
        $qty      = $quote->getQty();
        $totals   = $quote->getTotals();
        $shipping = $quote->getShippingAddress();

        foreach ($items as $item) {
            $qty += $item->getQty();
        }

        $quoteData = new Varien_Object([
            "items"      => array_map(function($p) {
                return Mage::helper("API/Product")->prepareCartProduct($p);
            }, $items),
            "summary"    => [
                "subTotal"              => array_key_exists("subtotal", $totals) ? (double) $totals["subtotal"]->getValueInclTax() : null,
                "subTotalExclTax"       => array_key_exists("subtotal", $totals) ? (double) $totals["subtotal"]->getValueExclTax() : null,
                "grandTotal"            => (double)$quote->getGrandTotal(),
                "grandTotalExclTax"     => (double)$quote->getGrandTotal() - (double)$quote->getShippingAddress()->getTaxAmount(),
                "tax"                   => array_key_exists("tax", $totals) ? (double) $totals["tax"]->getValue() : null,
                "discount"              => array_key_exists("discount", $totals) ? (double) $totals["discount"]->getValue() : null,
                "shippingAmount"        => (double)$shipping->getShippingInclTax() ?: $shipping->getShippingAmount(),
                "shippingAmountExclTax" => (double)$shipping->getShippingAmount(),
                "quoteCurrencyCode"     => $quote->getQuoteCurrencyCode(),
                "qty"                   => $qty,
                "coupon"                => $this->getCouponData(),
                "virtual"               => $quote->isVirtual()
            ]
        ]);

        Mage::dispatchEvent(self::EVENT_QUOTE_POST_DATA_PREPARE, [
            "quote"         => $quote,
            "totals"        => $totals,
            "prepared_data" => $quoteData
        ]);

        return $quoteData->getData();
    }

    public function updateItem($item)
    {
        $cart       = Mage::getSingleton('checkout/cart');
        $attributes = null;

        if(!is_array($item)) {
            throw Crossroads_API_ResponseException::create(400, "item is not an associative array, got '".gettype($item)."'.", null, 2008);
        }

        // map attributes
        if (array_key_exists("attributes", $item) && $item["attributes"] !== null) {
            if(!is_array($item["attributes"])) {
                throw Crossroads_API_ResponseException::create(400, "item.attributes should be a map, got '".gettype($item["attributes"])."'.", null, 2003);
            }

            if( ! empty($item["attributes"])) {
                $attributes = [];

                foreach($item["attributes"] as $k => $v) {
                    $attributes[(int)$k] = (int)$v;
                }
            }
        }

        // Null-check required on $item
        if (empty($item) || empty($item['qty']) || empty($item['product'])) {
            throw Crossroads_API_ResponseException::create(400, "item is missing qty and/or product parameters.", null, 2004);
        }

        if( ! is_array($item["product"]) || !array_key_exists("id", $item["product"])) {
            throw Crossroads_API_ResponseException::create(400, "item is missing product key, or product key does not contain an object with an id property.", null, 2005);
        }

        $product = Mage::getModel('catalog/product')
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load((int)$item["product"]["id"]);

        if (!$product || !$product->getId()) {
            throw Crossroads_API_ResponseException::create(404, "Product not found.", null, 2006);
        }

        if(!empty($item["id"])) {
            return $this->updateProductInCart((int)$item["id"], $product, $item["qty"], $attributes);
        }

        return $this->addProductToCart($product, $item["qty"], $attributes);
    }

    /**
     * Updates a product in the cart.
     *
     * @param  int      The row id of the cart row
     * @param  Product  The product to update
     * @param  int      The new quantity
     * @param  mixed    Any product attributes (super_attributes)
     */
    protected function updateProductInCart($rowId, $product, $qty, $attributes) {
        $cart = Mage::getSingleton('checkout/cart');

        try {
            $cart->updateItem($rowId, [
                "product"         => $product->getId(),
                "qty"             => $qty,
                "super_attribute" => $attributes,
                // Needed to prevent magento from attempting to fetch the original qty + the new qty
                // if we do not have id here the old quote item is considered different and will be
                // first summed with this request and then deleted after the validation has been performed
                // It also needs to be a string to pass === in Mage_Sales_Model_Quote
                // If it is a different item,
                "id"              => (string)$rowId
            ]);
        }
        catch(Exception $e) {
            // Can only happen in update product
            if($e->getMessage() === Mage::helper('checkout')->__('Quote item does not exist.')) {
                throw Crossroads_API_ResponseException::create(400, "Cart item with id '$rowId' was not found when attempting to update cart.", null, 2007);
            }

            // Determnine if we return an error to the user or rethrow
            $this->handleProductException($e, $product);
        }

        $this->saveCart($cart, $product, 'checkout_cart_update_item_complete');
    }

    /**
     * Adds a new product to the cart.
     *
     * @param  Product  The product to update
     * @param  int      The new quantity
     * @param  mixed    Any product attributes (super_attributes)
     */
    protected function addProductToCart($product, $qty, $attributes) {
        $cart = Mage::getSingleton('checkout/cart');

        try {
            $cart->addProduct($product, [
                "product"         => $product->getId(),
                "qty"             => $qty,
                "super_attribute" => $attributes
            ]);
        } catch (Exception $e) {
            // Determnine if we return an error to the user or rethrow
            $this->handleProductException($e, $product);
        }

        $this->saveCart($cart, $product, 'checkout_cart_add_product_complete');
    }

    /**
     * Saves the cart, notifies session that cart was updated and fires off the supplied event
     * with product, request and response.
     *
     * @param  Cart
     * @param  Product
     * @param  string
     */
    protected function saveCart($cart, $product, $event) {
        $session  = Mage::getSingleton("checkout/session");
        $request  = Mage::app()->getRequest();
        $response = Mage::app()->getResponse();

        $cart->save();
        $session->setCartWasUpdated(true);

        Mage::dispatchEvent($event, [
            'product'  => $product,
            'request'  => $request,
            'response' => $response
        ]);
    }

    /**
     * Handles exceptions caused in $cart->addProduct or $cart->updateItem gracefully, produces a
     * Crossroads_API_ResponseException error message * with the correct HTTP-status and an errorCode if
     * the error is caused by user input. Will rethrow if it is a system error.
     *
     * @param  Exception
     * @param  Product  The product used when encountering the exception
     *
     * @throws Crossroads_API_ResponseException
     */
    protected function handleProductException($e, $product) {
        $msg = $e->getMessage();
        // Magento is not sane, it is using stringly TRANSLATED typed exceptions
        if($msg === Mage::helper('catalog')->__('Please specify the product\'s option(s).') ||
           $msg === Mage::helper('catalog')->__('Please specify the product\'s required option(s).'))  {
            throw Crossroads_API_ResponseException::create(400, "Please specify the product's option(s).", null, 2002);
        }

        if($msg === Mage::helper('checkout')->__('The product could not be found.')) {
            throw Crossroads_API_ResponseException::create(400, "Product coult not be found", null, 2009);
        }

        if($msg === Mage::helper('cataloginventory')->__('This product is currently out of stock.')) {
            throw Crossroads_API_ResponseException::create(400, "Product is currently out of stock", null, 2000);
        }

        if(stripos($msg, Mage::helper('sales')->__('Item qty declaration error.')) !== false ||
           stripos($msg, Mage::helper('cataloginventory')->__('Item qty declaration error.\nThe requested quantity for \"%s\" is not available.', $product->getName())) !== false ||
           stripos($msg, Mage::helper('cataloginventory')->__('The requested quantity for "%s" is not available.', $product->getName())) !== false) {
            throw Crossroads_API_ResponseException::create(400, "Product quantity is not available.", null, 2001);
        }

        if(stripos($msg, strstr(Mage::helper('cataloginventory')->__('The maximum quantity allowed for purchase is %s.', "DEADBEEF"), "DEADBEEF", true)) !== false) {
            throw Crossroads_API_ResponseException::create(400, "Maximum quantity of requested product exceeded.", null, 20010);
        }

        throw $e;
    }
}