<?php

class Crossroads_API_CheckoutController extends Crossroads_API_Controller_Resource
{
    // TODO: More events

    /**
     * Event triggered just after the quote and customer has been obtained in preparation of
     * a checkout, after the checkout data has been merged.
     *
     * Params:
     *  * quote
     *  * customer
     */
    const EVENT_PRE_CREATE = "crossroads_api_checkout_pre_create";

    /**
     * Event triggered just after the quote has been loaded and the input has been determined to
     * not be empty, before any attempts have been made to merge any of the data into the quote.
     * It is possible to modify the input params by modifying the `params` object.
     *
     * Params:
     *  * quote
     *  * params
     *    The actual deserialized input data as a Varien_Object
     */
    const EVENT_PRE_MERGE = "crossroads_api_checkout_pre_merge";

    /**
     * Event triggered after checkout data has been merged in the quote, just before totals
     * will be collected. Any modifications to params will not be propagated.
     *
     * Params:
     *  * quote
     *  * params
     */
    const EVENT_POST_MERGE = "crossroads_api_checkout_post_merge";

    /**
     * Event triggered before the quote is verified.
     *
     * Params:
     *  * quote
     */
    const EVENT_PRE_VERIFY_QUOTE = "crossroads_api_checkout_pre_verify_quote";

    /**
     * Event triggered after quote has been verified but before shipping and totals have been collected.
     * By setting the response object's `code` to a value greater than 0 (eg. 403) the
     * observer is able to abort the checkout process before the order is created.
     *
     * Params:
     *  * quote
     *  * response
     *      code:      HTTP Response code, int, default 0
     *      errorCode: `errorCode` response value, int, default 0
     *      message:   extra `message` property present in `data`, string, default ""
     */
    const EVENT_POST_VERIFY_QUOTE = "crossroads_api_checkout_post_verify_quote";

    /**
     * Event triggered after `POST /checkout` finishes validation, before the order is placed.
     * By setting the response object's `code` to a value greater than 0 (eg. 403) the
     * observer is able to abort the checkout process before the order is created.
     *
     * Params:
     *  * quote
     *  * response
     *      code:      HTTP Response code, int, default 0
     *      errorCode: `errorCode` response value, int, default 0
     *      message:   extra `message` property present in `data`, string, default ""
     */
    const EVENT_POST_CREATE_VALIDATION = "crossroads_api_checkout_post_create_validation";

    /**
     * Duration the order will be visible for in GET /checkout/:incrementId even if the cookie
     * does not match. Seconds.
     *
     * This is to make sure that even if cookies are cleared when the client visits an external
     * partner they will still see their order status in the end.
     */
    const ORDER_VISIBLE_WINDOW = 900;

    /**
     * Session storage key order incrementIds will be stored in. This key is used to limit
     * access to `/checkout/:incrementId`.
     */
    const ORDER_INCREMENT_ID_STORAGE_KEY = "crossroads_api_checkout_visible_orders";

    /**
     * List of valid payment methods and their classes.
     */
    protected $PAYMENT_METHODS = [
        "Dibspw" => "Crossroads_API_Payment_Dibspw",
    ];

    /**
     * Billing address properties that must be always copied to shipping address.
     */
    protected $REQUIRED_BILLING_ATTRIBUTES = ['customer_address_id'];

    /**
     * Adds the supplied order incrementId to the list of visible orders for the current user (based on session).
     *
     * @param  string
     * @return void
     */
    protected function setOrderIncrementIdVisibleForUser($incrementId) {
        $orderIds = $this->getVisibleOrderIncrementIdsForUser();

        $orderIds[] = (string) $incrementId;

        Mage::getSingleton("core/session")->setData(self::ORDER_INCREMENT_ID_STORAGE_KEY, $orderIds);
    }

    /**
     * Retrieves the list of visible order ids for the current user (based on session).
     *
     * @return Array<string>
     */
    protected function getVisibleOrderIncrementIdsForUser() {
        $orderIds = Mage::getSingleton("core/session")->getData(self::ORDER_INCREMENT_ID_STORAGE_KEY);

        if(!is_array($orderIds)) {
            $orderIds = [];
        }

        return $orderIds;
    }

    /**
     * Returns true if the supplied order is visible for the current user (based on session).
     *
     * @param  Mage_Sales_Model_Order
     * @return boolean
     */
    protected function isOrderVisibleForUser($order) {
        $orderIds = $this->getVisibleOrderIncrementIdsForUser();

        return in_array((string)$order->getIncrementId(), $orderIds, true) || strtotime($order->getCreatedAt()) + self::ORDER_VISIBLE_WINDOW > time();
    }

    /**
     * Returns the input key for the specific address type.
     *
     * @param  string One of the Mage_Sales_Model_Quote_Address::Type_* constants.
     * @return string|false
     */
    protected static function addressKey($type) {
        switch($type) {
        case Mage_Sales_Model_Quote_Address::TYPE_SHIPPING:
            return "shippingAddress";
        case Mage_Sales_Model_Quote_Address::TYPE_BILLING:
            return "billingAddress";
        }

        return false;
    }

    /**
     * Attempts to load the associated payment model for the payment identifier. Throws an exception
     * if the payment method is not supported.
     *
     * @param  string
     * @return Crossroads_API_Payment_Abstract
     */
    protected function paymentModel($paymentMethod) {
        if(!array_key_exists($paymentMethod, $this->PAYMENT_METHODS)) {
            throw new Exception("Crossroads API: Invalid payment method '$paymentMethod'.");
        }

        $klass = $this->PAYMENT_METHODS[$paymentMethod];

        return new $klass();
    }

    /**
     * Returns a list of supported AND enabled payment methods.
     *
     * @return array
     */
    protected function paymentMethods() {
        $paymentCodes  = array_keys(Mage::getSingleton('payment/config')->getActiveMethods());
        $supported     = array_keys($this->PAYMENT_METHODS);

        return array_map(function($paymentCode) {
            return [
                'code'  => $paymentCode,
                'label' => Mage::getStoreConfig('payment/' . $paymentCode . '/title')
            ];
        }, array_values(array_filter($paymentCodes, function($c) use($supported) {
            return in_array($c, $supported);
        })));
    }

    /**
     * @apiDefine orderStatus
     *
     * @apiSuccess {String} id  Order id
     * @apiSuccess {String} paymentMethod
     * @apiSuccess {String} paymentStatus  `IN_PROGRESS`, `COMPLETE` or `FAILED`
     * @apiSuccess {Mixed}  paymentData    Data for the payment method
     *
     * @apiSuccess {string}   email
     * @apiSuccess {string}   createdAt ISO 8601 UTC
     *
     * @apiParam {String} shippingMethod
     *
     * @apiSuccess {Object}   billingAddress
     * @apiSuccess {String}   billingAddress.prefix
     * @apiSuccess {String}   billingAddress.firstname
     * @apiSuccess {String}   billingAddress.middlename
     * @apiSuccess {String}   billingAddress.lastname
     * @apiSuccess {String}   billingAddress.suffix
     * @apiSuccess {String}   billingAddress.company
     * @apiSuccess {String[]} billingAddress.street
     * @apiSuccess {String}   billingAddress.postcode
     * @apiSuccess {String}   billingAddress.city
     * @apiSuccess {String}   billingAddress.regionId
     * @apiSuccess {String}   billingAddress.countryId (SE, NO, DK)
     * @apiSuccess {String}   billingAddress.telephone
     * @apiSuccess {String}   billingAddress.fax
     * @apiSuccess {bool}     billingAddress.useAsShippingAddress
     *
     * @apiSuccess {Object}   shippingAddress
     * @apiSuccess {String}   shippingAddress.prefix
     * @apiSuccess {String}   shippingAddress.firstname
     * @apiSuccess {String}   shippingAddress.middlename
     * @apiSuccess {String}   shippingAddress.lastname
     * @apiSuccess {String}   shippingAddress.suffix
     * @apiSuccess {String}   shippingAddress.company
     * @apiSuccess {String[]} shippingAddress.street
     * @apiSuccess {String}   shippingAddress.postcode
     * @apiSuccess {String}   shippingAddress.city
     * @apiSuccess {String}   shippingAddress.regionId
     * @apiSuccess {String}   shippingAddress.countryId (SE, NO, DK)
     * @apiSuccess {String}   shippingAddress.telephone
     * @apiSuccess {String}   shippingAddress.fax
     *
     * @apiSuccess {Quote}    quote  See the cart object from `/cart`
     *
     * @apiSuccessExample {json} Example Complete
     * { "id": "A-100066", "paymentMethod": "Dibspw", "status": "COMPLETE", "data": null }
     * @apiSuccessExample {json} Example Dibs
     * { "id": "A-100066", "paymentMethod": "Dibspw", "status": "IN_PROGRESS", "data": { "action": "SUBMIT_FORM_DATA", "url": "https://example.com/entrypoint", "fields": { "acceptReturnUrl": "http://example.com/accept/A-100066", "cancelReturnUrl": "http://example.com/cancel/A-100066", "callbackUrl": "http://example.com/system/A-100066", "amount": 80000, "currency": "SEK", "language": "sv_SE", "merchant": "0000000", "orderId": "A-100066", "billingFirstName": "Testperson", "billingLastName": "Testsson", "billingAddress": "SE", "billingAddress2": "Testvägen 123", "billingPostalCode": "000 00", "billingPostalPlace": "Teststaden", "billingEmail": "test@example.com", "shippingFirstName": "Testperson", "shippingLastName": "Testsson", "shippingAddress": "SE", "shippingAddress2": "Testvägen 123", "shippingPostalCode": "000 00", "shippingPostalPlace": "Teststaden", "captureNow": 1, "yourRef": "A-100066", "payType": "VISA,MC", "MAC": "0000000000000000000000000000000000000000000000000000000000000000" } } }
     */

    /**
     * @apiDefine checkoutData
     *
     * @apiParam {String} paymentMethod
     * @apiParam {String} shippingMethod
     * @apiParam {String} email Order email
     *
     * @apiSuccess {Object[]} paymentMethods  Available paymentMethods, key-value map with the key being the payment-method identifier
     * @apiSuccess {String}   paymentMethods.code
     * @apiSuccess {String}   paymentMethods.label
     *
     * @apiSuccess {Object[]} shippingMethods  Available shippingMethods
     * @apiSuccess {String}   shippingMethods.code
     * @apiSuccess {String}   shippingMethods.title
     * @apiSuccess {String}   shippingMethods.label
     * @apiSuccess {String}   shippingMethods.description
     *
     * @apiSuccess {String}   shippingMethod  Selected paymentMethod
     *
     * @apiSuccess {String}   paymentMethod  Selected paymentMethod
     * @apiSuccess {Object}   paymentMethodData Data specific to the payment method
     *
     * @apiSuccess {Object}   billingAddress
     * @apiSuccess {String}   billingAddress.prefix
     * @apiSuccess {String}   billingAddress.firstname
     * @apiSuccess {String}   billingAddress.middlename
     * @apiSuccess {String}   billingAddress.lastname
     * @apiSuccess {String}   billingAddress.suffix
     * @apiSuccess {String}   billingAddress.company
     * @apiSuccess {String[]} billingAddress.street
     * @apiSuccess {String}   billingAddress.postcode
     * @apiSuccess {String}   billingAddress.city
     * @apiSuccess {String}   billingAddress.regionId
     * @apiSuccess {String}   billingAddress.countryId (SE, NO, DK)
     * @apiSuccess {String}   billingAddress.telephone
     * @apiSuccess {String}   billingAddress.fax
     * @apiSuccess {bool}     billingAddress.useAsShippingAddress
     *
     * @apiSuccess {Object}   shippingAddress
     * @apiSuccess {String}   shippingAddress.prefix
     * @apiSuccess {String}   shippingAddress.firstname
     * @apiSuccess {String}   shippingAddress.middlename
     * @apiSuccess {String}   shippingAddress.lastname
     * @apiSuccess {String}   shippingAddress.suffix
     * @apiSuccess {String}   shippingAddress.company
     * @apiSuccess {String[]} shippingAddress.street
     * @apiSuccess {String}   shippingAddress.postcode
     * @apiSuccess {String}   shippingAddress.city
     * @apiSuccess {String}   shippingAddress.regionId
     * @apiSuccess {String}   shippingAddress.countryId (SE, NO, DK)
     * @apiSuccess {String}   shippingAddress.telephone
     * @apiSuccess {String}   shippingAddress.fax
     *
     * @apiSuccess {Cart}     cart  See the cart object
     *
     * @apiSuccessExample {json} Empty checkout
     * { "paymentMethods": [ { "code": "Dibspw", "label": "Mastercard / VISA" } ], "shippingMethods": [ { "code": "flatrate_flatrate", "title": "Fraktfritt", "label": "0:-" }, { "code": "freeshipping_freeshipping", "title": "Frakten ingår", "label": "Ingen extra avgift" }, { "code": "tablerate_bestway", "title": "Best Way", "label": "Table Rate" } ], "paymentMethod": null, "shippingMethod": null, "billingAddress": { "prefix": null, "firstname": null, "middlename": null, "lastname": null, "suffix": null, "company": null, "street": [ "" ], "postcode": null, "city": null, "regionId": null, "countryId": null, "telephone": null, "fax": null, "useAsShippingAddress": false }, "shippingAddress": { "prefix": null, "firstname": null, "middlename": null, "lastname": null, "suffix": null, "company": null, "street": [ "" ], "postcode": null, "city": null, "regionId": null, "countryId": null, "telephone": null, "fax": null }, "cart": { "items": [], "summary": { "subTotal": 0, "subTotalExclTax": 0, "grandTotal": 0, "grandTotalExclTax": 0, "tax": null, "discount": null, "shippingAmount": 0, "quoteCurrencyCode": "SEK", "qty": null, "couponCode": null }, "virtual": false }, "email": null }
     *
     * @apiSuccessExample {json} Fully filled out checkout
     * { "paymentMethods": [ { "code": "Dibspw", "label": "Mastercard / VISA" } ], "shippingMethods": [ { "code": "flatrate_flatrate", "title": "Fraktfritt", "label": "0:-" }, { "code": "freeshipping_freeshipping", "title": "Frakten ingår", "label": "Ingen extra avgift" }, { "code": "tablerate_bestway", "title": "Best Way", "label": "Table Rate" } ], "paymentMethod": "Dibspw", "shippingMethod": "flatrate_flatrate", "billingAddress": { "prefix": null, "firstname": "Test", "middlename": null, "lastname": "Testsson", "suffix": null, "company": "Testföretaget", "street": [ "Testvägen 123" ], "postcode": "000 00", "city": "Teststaden", "regionId": null, "countryId": "SWE", "telephone": "000-0000000", "fax": null, "useAsShippingAddress": true }, "shippingAddress": { "prefix": null, "firstname": "Test", "middlename": null, "lastname": "Testsson", "suffix": null, "company": "Testföretaget", "street": [ "Testvägen 123" ], "postcode": "000 00", "city": "Teststaden", "regionId": null, "countryId": "SWE", "telephone": "000-0000000", "fax": null }, "cart": { "items": [ { "id": 1072, "product": { "id": 17, "name": "Biobiljett SF Bio", "urlKey": "biobiljett-sf-bio", "price": 93.4, "stockQty": 500, "manufacturer": "Presentkort", "thumbnail": "http://example.com/media/thumbnail.jpg" }, "qty": 1, "rowTotal": 93.4, "rowTax": 0, "attributes": null, "options": null }, { "id": 1073, "product": { "id": 38, "name": "Testprodukt, config", "urlKey": "testprodukt-config", "price": 800, "stockQty": 0, "manufacturer": "Design Test", "thumbnail": "http://tui.dev/thumbnail.jpg" }, "qty": 1, "rowTotal": 800, "rowTax": 0, "attributes": { "92": 23 }, "options": [ { "id": 92, "code": "color", "title": "Color", "useAsDefault": false, "position": 0, "value": { "id": 23, "label": "Gul", "isPercent": false, "price": null } } ] } ], "summary": { "subTotal": 893.4, "subTotalExclTax": 893.4, "grandTotal": 893.4, "grandTotalExclTax": 893.4, "tax": null, "discount": null, "shippingAmount": 0, "quoteCurrencyCode": "SEK", "qty": 2, "couponCode": null }, "virtual": false }, "email": "test@example.com" }
     */

    /**
     * Adds parameters to the checkout data.
     *
     * @apiDefine checkoutInput
     *
     * @apiParam {Object}   billingAddress
     * @apiParam {String}   billingAddress.prefix
     * @apiParam {String}   billingAddress.firstname
     * @apiParam {String}   billingAddress.middlename
     * @apiParam {String}   billingAddress.lastname
     * @apiParam {String}   billingAddress.suffix
     * @apiParam {String}   billingAddress.company
     * @apiParam {String[]} billingAddress.street
     * @apiParam {String}   billingAddress.postcode
     * @apiParam {String}   billingAddress.city
     * @apiParam {String}   billingAddress.regionId
     * @apiParam {String}   billingAddress.countryId (SE, NO, DK)
     * @apiParam {String}   billingAddress.telephone
     * @apiParam {String}   billingAddress.fax
     * @apiParam {bool}     billingAddress.useAsShippingAddress
     *
     * @apiParam {Object}   shippingAddress
     * @apiParam {String}   shippingAddress.prefix
     * @apiParam {String}   shippingAddress.firstname
     * @apiParam {String}   shippingAddress.middlename
     * @apiParam {String}   shippingAddress.lastname
     * @apiParam {String}   shippingAddress.suffix
     * @apiParam {String}   shippingAddress.company
     * @apiParam {String[]} shippingAddress.street
     * @apiParam {String}   shippingAddress.postcode
     * @apiParam {String}   shippingAddress.city
     * @apiParam {String}   shippingAddress.regionId
     * @apiParam {String}   shippingAddress.countryId (SE, NO, DK)
     * @apiParam {String}   shippingAddress.telephone
     * @apiParam {String}   shippingAddress.fax
     *
     * @return Array|null Null if no issues, array as a response if the user needs to be informed
     */
    protected function mergeCheckoutData() {
        /*
         * Rules:
         *
         * - Virtual quotes do not have a shippingAddress
         *
         * - You should not be able to pass a custom email on a virtual quote when you're logged in
         */

        $params = $this->requestData();

        if(empty($params)) {
            return;
        }

        if(!is_array($params)) {
            return self::formatError(400, "Request body must be an object", 4015);
        }

        $quote = Mage::getSingleton("checkout/cart")->getQuote();
        // We use a Varien_Object to allow the observers to actually modify input data
        $event = new Varien_Object($params);

        Mage::dispatchEvent(self::EVENT_PRE_MERGE, [
            "quote"  => $quote,
            "params" => $event
        ]);

        $params = $event->getData();

        if($r = $this->mergeCart($params)) {
            return $r;
        }

        if (array_key_exists("email", $params)) {
            $quote->setCustomerEmail($params["email"]);
        }

        if (array_key_exists("billingAddress", $params)) {
            if($r = $this->updateAddress($params["billingAddress"], $quote->getBillingAddress())) {
                return $r;
            }
        }

        if (array_key_exists("shippingAddress", $params)) {
            if($r = $this->updateAddress($params["shippingAddress"], $quote->getShippingAddress())) {
                return $r;
            }

            // We collect totals here to make sure we can validate shipping properly later
            $quote->setTotalsCollectedFlag(false)->collectTotals()->save();
        }

        if($r = $this->setUseAsShippingAddress($quote, $params)) {
            return $r;
        }

        if (array_key_exists("paymentMethod", $params)) {
            $method = $params["paymentMethod"];

            if(!is_string($method)) {
                return self::formatError(400, "paymentMethod value must be a string", null, 4000);
            }

            if ($quote->isVirtual()) {
                $quote->getBillingAddress()->setPaymentMethod($method);
            } else {
                $quote->getShippingAddress()->setPaymentMethod($method);
            }

            $quote->getPayment()->importData(array('method' => $method));
        }

        if (array_key_exists("shippingMethod", $params)) {
            if(!$quote->isVirtual()) {
                $method = $params["shippingMethod"];

                if(!is_string($method)) {
                    return self::formatError(400, "shippingMethod value must be a string", null, 4002);
                }

                $rate = $quote->getShippingAddress()->getShippingRateByCode($method);
                if(!$rate) {
                    return self::formatError(400, "Invalid shipping method", null, 4003);
                }

                // Make sure we recalculate the shipping rates
                $quote->getShippingAddress()->setShippingMethod($method)->setCollectShippingRates(true);
            }

        }

        Mage::dispatchEvent(self::EVENT_POST_MERGE, [
            "quote"  => $quote,
            "params" => $event
        ]);

        // Always collect totals here because we have updated values
        $quote->setTotalsCollectedFlag(false)->collectTotals()->save();
    }

    /**
     * Merges the cart data to enable updates, deletes and additions to the cart as well as complete
     * orders witout prior order data through POST /checkout.
     */
    protected function mergeCart($params) {
        $cart  = Mage::getSingleton("checkout/cart");

        if (!array_key_exists("cart", $params) || !$params["cart"]) {
            return;
        }

        if (!is_array($params["cart"]) ||
            !array_key_exists("items", $params["cart"]) ||
            !is_array($params["cart"]["items"])) {
            return self::formatError(400, "cart.items is missing", null, 4004);
        }

        foreach($cart->getQuote()->getAllVisibleItems() as $cartItem) {
            $exists = false;

            foreach($params["cart"]["items"] as $item) {
                if(array_key_exists("id", $item) && (int)$item["id"] === (int)$cartItem->getItemId()) {
                    $exists = true;

                    break;
                }
            }

            if(!$exists) {
                $cart->removeItem($cartItem->getItemId());
            }
        }

        foreach($params["cart"]["items"] as $item) {
            if($r = Mage::helper("API/cart")->updateItem($item)) {
                return $r;
            }
        }

        $cart->save();
    }

    protected function updateAddress($data, $address) {
        // TODO: Support for customer address id (ie. load saved address)
        if(!$data) {
            return;
        }

        if(!is_array($data)) {
            return self::formatError(400, self::addressKey($address->getAddressType())." must be a map, '".gettype($data)."' received.", null, 4005);
        }

        if(array_key_exists("prefix", $data))     { $address->setPrefix($data["prefix"]); }
        if(array_key_exists("firstname", $data))  { $address->setFirstname($data["firstname"]); }
        if(array_key_exists("middlename", $data)) { $address->setMiddlename($data["middlename"]); }
        if(array_key_exists("lastname", $data))   { $address->setLastname($data["lastname"]); }
        if(array_key_exists("suffix", $data))     { $address->setSuffix($data["suffix"]); }
        if(array_key_exists("company", $data))    { $address->setCompany($data["company"]); }
        if(array_key_exists("street", $data))     { $address->setStreet($data["street"]); }
        if(array_key_exists("city", $data))       { $address->setCity($data["city"]); }
        if(array_key_exists("regionId", $data))   { $address->setRegionId($data["regionId"]); }
        if(array_key_exists("postcode", $data))   { $address->setPostcode($data["postcode"]); }
        if(array_key_exists("countryId", $data))  { $address->setCountryId($data["countryId"]); }
        if(array_key_exists("telephone", $data))  { $address->setTelephone($data["telephone"]); }
        if(array_key_exists("fax", $data))        { $address->setFax($data["fax"]); }

        // If we're saving this it is not the same as billing
        if($address->getAddressType() === Mage_Sales_Model_Quote_Address::TYPE_SHIPPING) {
            $address->setSameAsBilling(0);
        }

        // Modified address, shipping rates needs updates
        $address->setCollectShippingRates(true);
    }

    /**
     * If billingAddress.useAsShippingAddress is set to true, duplicate the billingAddress to the shippingAddress.
     */
    protected function setUseAsShippingAddress($quote, $params) {
        $shippingAddress = $quote->getShippingAddress();
        $billingAddress  = $quote->getBillingAddress();

        if(!array_key_exists("billingAddress", $params) ||
            !is_array($params["billingAddress"]) ||
            !array_key_exists("useAsShippingAddress", $params["billingAddress"])) {
            return;
        }

        $useAsShippingAddress = $params["billingAddress"]["useAsShippingAddress"];

        if(!is_bool($useAsShippingAddress)) {
            return self::formatError(400, "billingAddress.useAsShippingAddress must be boolean.", null, 4007);
        }

        $quote = Mage::getSingleton("checkout/cart")->getQuote();

        if($quote->isVirtual() && $useAsShippingAddress) {
            // Intended, we ignore the useAsShippingAddress parameter
            return;
        }

        if(!$useAsShippingAddress) {
            $shipping = $this->getQuote()->getShippingAddress();
            $shipping->setSameAsBilling(0);
        }

        $shippingMethod = $shippingAddress->getShippingMethod();
        $newShipping    = clone $billingAddress;

        $newShipping->unsAddressId()->unsAddressType();

        foreach($shippingAddress->getData() as $key => $value) {
            if(!is_null($value) && !is_null($newShipping->getData($key)) &&
               !isset($params["billingAddress"][$key]) && !in_array($key, $this->REQUIRED_BILLING_ATTRIBUTES)) {
                $newShipping->unsetData($key);
            }
        }

        // CollectShippingRates = true here to update shipping properly
        $shippingAddress->addData($newShipping->getData())
            ->setSameAsBilling(1)
            ->setSaveInAddressBook(0)
            ->setShippingMethod($shippingMethod)
            ->setCollectShippingRates(true);

        // We collect totals here to make sure we can validate shipping properly
        $quote->setTotalsCollectedFlag(false)->collectTotals()->save();
    }

    /**
     * @api {get} /checkout  Fetch current checkout data
     * @apiName getCheckoutData
     * @apiGroup Checkout
     *
     * @apiUse checkoutData
     */
    public function getAll()
    {
        $cart            = Mage::helper("API/Cart")->getCart();
        $quote           = Mage::getSingleton('checkout/session')->getQuote();
        $carriers        = Mage::getSingleton('shipping/config')->getActiveCarriers();
        $shippingMethods = [];

        if($quote->hasItems()) {
            // Magento will crash (unless warnings and notices are off) if we call
            // collectShippingRates() without anything in the cart
            $carriers = $quote->getShippingAddress()->collectShippingRates()
                     ->getGroupedAllShippingRates();

            foreach($carriers as $_carrier_name => $methods) {
                foreach($methods as $method) {
                    $shippingMethods[] =  [
                        "code"        => $method->getCode(),
                        "title"       => $method->getCarrierTitle(),
                        "label"       => $method->getMethodTitle(),
                        "description" => $method->getMethodDescription()
                    ];
                }
            }
        }

        return [200, [
            "paymentMethods"  => $this->paymentMethods(),
            "shippingMethods" => $shippingMethods,
            "paymentMethod"   => $quote->getPayment()->getMethod(),
            "shippingMethod"  => $quote->getShippingAddress()->getShippingMethod(),
            "billingAddress"  => $this->prepareAddress($quote->getBillingAddress()),
            "shippingAddress" => $this->prepareAddress($quote->getShippingAddress()),
            "cart"            => $cart,
            "email"           => $quote->getCustomerEmail() ?: null
        ]];
    }

    protected function prepareAddress($address) {
        $data = [
            "prefix"     => $address->getPrefix(),
            "firstname"  => $address->getFirstname(),
            "middlename" => $address->getMiddlename(),
            "lastname"   => $address->getLastname(),
            "suffix"     => $address->getSuffix(),
            "company"    => $address->getCompany(),
            "street"     => $address->getStreet(),
            "postcode"   => $address->getPostcode(),
            "city"       => $address->getCity(),
            "regionId"   => $address->getRegionId(),
            "countryId"  => $address->getCountryId(),
            "telephone"  => $address->getTelephone(),
            "fax"        => $address->getFax(),
        ];

        if($address->getAddressType() === Mage_Sales_Model_Quote_Address::TYPE_BILLING) {
            $data["useAsShippingAddress"] = (bool)Mage::getSingleton("checkout/session")
                ->getQuote()
                ->getShippingAddress()
                ->getSameAsBilling();
        }

        return $data;
    }

    /**
     * Verifies a quote to make sure an order can be placed using it, returns a response
     * if it does not verify. Null is returned if it is valid.
     */
    protected function verifyQuote($quote) {
        Mage::dispatchEvent(self::EVENT_PRE_VERIFY_QUOTE, [
            "quote" => $quote,
        ]);

        if(!$quote->hasItems()) {
            return self::formatError(400, "No products present in cart", 4008);
        }

        if(!$quote->isVirtual()) {
            // Physical items require a shipping address and a method
            $shippingAddress    = $quote->getShippingAddress();
            $shippingValidation = $shippingAddress->validate();

            if($shippingValidation !== true) {
                return self::formatError(400, "Shipping address did not validate.", $shippingValidation, 4009);
            }

            $method = $shippingAddress->getShippingMethod();
            $rate   = $shippingAddress->getShippingRateByCode($method);

            if (!$method || !$rate) {
                return self::formatError(400, "Please specify a shipping method.", null, 4010);
            }
        }

        $billingValidation = $quote->getBillingAddress()->validate();

        if($billingValidation !== true) {
            return self::formatError(400, "Billing address did not validate.", $billingValidation, 4011);
        }

        if (!$quote->getPayment()->getMethod()) {
            return self::formatError(400, "Missing payment method", null, 4012);
        }

        if(!array_key_exists($quote->getPayment()->getMethod(), $this->PAYMENT_METHODS)) {
            return self::formatError(400, "Crossroads API: Invalid payment method '".$this->getPayment()->getMethod()."'.", null, 4013);
        }

        $response = new Varien_Object([
            "code"      => 0,
            "errorCode" => 0,
            "message"   => ""
        ]);
        Mage::dispatchEvent(self::EVENT_POST_VERIFY_QUOTE, [
            "quote"    => $quote,
            "response" => $response
        ]);

        // If code > 0 we will return it as the response
        if($response->getCode()) {
            return self::formatError(
                $response->getCode(),
                "Quote verification aborted by observer.",
                $response->getData(),
                $response->getErrorCode()
            );
        }

        return null;
    }

    /**
     * @api {post} /checkout  Perform a checkout
     * @apiName performCheckout
     * @apiGroup Checkout
     * @apiDescription Note that any data present in request-body will be merged with the data in the session (ie. PATCH behaviour).
     * If the cart key is set the session cart will be synchronized to exactly match the cart in the request. This enables a
     * `POST /checkout` to be used without having to add anything to cart or even prefetch checkout data.
     *
     * @apiUse checkoutInput
     * @apiUse orderStatus
     */
    public function createItem()
    {
        if($r = $this->mergeCheckoutData()) {
            return $r;
        }

        $quote    = Mage::getSingleton("checkout/cart")->getQuote();
        $customer = Mage::getSingleton('customer/session')->getCustomer();

        Mage::dispatchEvent(self::EVENT_PRE_CREATE, [
            "quote"    => $quote,
            "customer" => $customer
        ]);

        if($r = $this->verifyQuote($quote)) {
            return $r;
        }

        if (Mage::getSingleton('customer/session')->isLoggedIn()) {
            $quote->assignCustomer($customer);
        }

        // Make sure we update the totals and rates
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false)->collectTotals();
        $quote->save();

        // By wrapping it here we enable the observer to abort the checkout
        $response = new Varien_Object([
            "code"      => 0,
            "errorCode" => 0,
            "message"   => ""
        ]);
        Mage::dispatchEvent(self::EVENT_POST_CREATE_VALIDATION, [
            "quote"    => $quote,
            "response" => $response
        ]);

        // If code > 0 we will return it as the response
        if($response->getCode()) {
            return self::formatError(
                $response->getCode(),
                "Checkout creation aborted by observer.",
                $response->getData(),
                $response->getErrorCode()
            );
        }

        // Common validation done

        $model = $this->paymentModel($quote->getPayment()->getMethod());

        if($r = $model->isQuoteInvalid($quote)) {
            return self::formatError(400, "Failed to validate payment data.", $r, 4014);
        }

        // place order
        $quoteService = Mage::getModel('sales/service_quote', $quote);
        $quoteService->submitAll();

        $order = $quoteService->getOrder();

        // Invalidate quote
        $quote->setIsActive(false)->save();

        $this->setOrderIncrementIdVisibleForUser($order->getIncrementId());

        // Load data and override response code
        $r = $this->getItem($order->getIncrementId());

        if($r[0] !== 200) {
            return $r;
        }

        // 201 is required for the Location header to work
        $this->getResponse()->setRedirect(Mage::getUrl("API/checkout/{$order->getIncrementId()}"), 201);

        return [201, empty($r[1]) ? '' : $r[1]];
    }

    /**
     * @api {delete} /checkout  Reset checkout
     * @apiName resetCheckout
     * @apiGroup Checkout
     * @apiDescription Removes all information from the active quote, resetting all fields to null.
     *
     * @apiUse checkoutData
     */
    public function deleteAll() {
        $quote = Mage::getSingleton("checkout/cart")->getQuote();

        $quote->removeAllItems();
        $quote->removeAllAddresses();
        $quote->removePayment();
        $quote->unsCustomerEmail();

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false)->collectTotals();

        $quote->save();

        return $this->getAll();
    }

    /**
     * @api {put} /checkout  Update checkout
     * @apiName updateCheckout
     * @apiGroup Checkout
     * @apiDescription Conceptually works like a `PATCH`, leaving out keys (ie. the key not being present or null) will preserve the old value.
     *     Setting `null` as values inside addresses (eg. city) will set their corresponding value to `null`. If the cart key is set the session
     *     cart will be synchronized to exactly match the cart in the request. This enables a `POST /checkout` to be used without having to add
     *     anything to cart or even prefetch checkout data.
     *     Adding the `verify` query parameter set to `true` will perform a verification of the qhote before merge,
     *     if this verification succeeds a `POST /checkout` request should also succeed.
     *
     * @apiParam {boolean} verify Optional query parameter, if set to "true" it will verify the quote
     *                            after merging the data
     * @apiUse checkoutInput
     * @apiUse checkoutData
     */
    public function replaceAll()
    {
        $req = $this->getRequest();

        if($r = $this->mergeCheckoutData()) {
            return $r;
        }

        if("true" === strtolower($req->getParam("verify") ?: "")) {
            if($r = $this->verifyQuote(Mage::getSingleton("checkout/cart")->getQuote())) {
                return $r;
            }
        }

        return $this->getAll();
    }

    /**
     * @api {get} /checkout/:incrementId  Payment status for order
     * @apiName getCheckoutPaymentStatus
     * @apiGroup Checkout
     * @apiDescription Only visible for the session which actually created the order.
     *
     * @apiUse orderStatus
     */
    public function getItem($id)
    {
        $order = Mage::getModel("sales/order")->loadByIncrementId($id);

        if(!$order || !$order->getIncrementId() || !$order->getPayment()) {
            return [404];
        }

        if(!$this->isOrderVisibleForUser($order)) {
            return [403];
        }

        $model = $this->paymentModel($order->getPayment()->getMethod());
        $quote = Mage::getModel("sales/quote")->load($order->getQuoteId());

        return [200, [
            "id"              => $order->getIncrementId(),
            "paymentMethod"   => $order->getPayment()->getMethod(),
            "paymentStatus"   => $model->getOrderPaymentStatus($order),
            "paymentData"     => $model->getOrderPaymentData($order),
            "shippingMethod"  => $quote->getShippingAddress()->getShippingMethod(),
            "billingAddress"  => $this->prepareAddress($quote->getBillingAddress()),
            "shippingAddress" => $this->prepareAddress($quote->getShippingAddress()),
            "quote"           => Mage::helper("API/Cart")->formatQuote($quote),
            "email"           => $quote->getCustomerEmail() ?: null,
            "createdAt"       => gmdate("Y-m-d\TH:i:s\Z", strtotime($quote->getCreatedAt())),
        ]];
    }
}
