<?php

class Crossroads_API_CartController extends Crossroads_API_Controller_Resource
{
    protected function getCart()
    {
        return Mage::getSingleton('checkout/cart');
    }

    /**
     * @apiDefine cartItems
     *
     * @apiSuccess {Object[]}  items  The quote list
     * @apiSuccess {Integer} items.id
     * @apiSuccess {Object}  items.product  A lightweight product object
     * @apiSuccess {Integer} items.product.id
     * @apiSuccess {String}  items.product.sku
     * @apiSuccess {String}  items.product.name
     * @apiSuccess {String}  items.product.urlKey
     * @apiSuccess {Double}  items.product.price
     * @apiSuccess {mixed}   items.product.stockQty               Amount in stock, null if information not available
     * @apiSuccess {String}  items.product.manufacturer
     * @apiSuccess {String}  items.product.thumbnail  Complete URL to a thumbnail for the product
     * @apiSuccess {Double}  items.qty
     * @apiSuccess {Double}  items.rowTotal  Total price for the row
     * @apiSuccess {Double}  items.rowTax
     * @apiSuccess {Object}  items.attributes  Map of attribute id => attribute value for configurable products
     * @apiSuccess {Integer} items.attributes[attributeId] Attribute value
     * @apiSuccess {Mixed}   options                   Null if simple product, object if complex
     * @apiSuccess {Integer} options.id
     * @apiSuccess {String}  options.title
     * @apiSuccess {Boolean} options.useAsDefault
     * @apiSuccess {Integer} options.position
     * @apiSuccess {Object}  options.value       The selected value for the complex product attribute
     * @apiSuccess {Integer} options.value.id
     * @apiSuccess {String}  options.value.label
     * @apiSuccess {Boolean} options.value.isPercent
     * @apiSuccess {Mixed}   options.value.price    null if not set
     * @apiSuccess {Object} summary  Cart summary
     * @apiSuccess {Double} summary.subTotal  Always includes tax
     * @apiSuccess {Double} summary.subTotalExclTax
     * @apiSuccess {Double} summary.grandTotal  Always includes tax
     * @apiSuccess {Double} summary.grandTotalExclTax
     * @apiSuccess {Double} summary.discount  Always includes tax, null if no discount available
     * @apiSuccess {Double} summary.tax
     * @apiSuccess {Double} summary.shippingAmount Note that this is affected by magento settings
     * @apiSuccess {Double} summary.shippingAmountExclTax
     * @apiSuccess {String} summary.quoteCurrencyCode
     * @apiSuccess {Double} summary.qty
     * @apiSuccess {Mixed}  summary.coupon Null if no coupon is applied
     * @apiSuccess {String}  summary.coupon.couponCode
     * @apiSuccess {String}  summary.coupon.ruleName
     * @apiSuccess {Boolean} summary.virtual If the quote only contains virtual items
     *
     * @apiSuccessExample {json} Success-Response:
     * { "items": [ { "id": 1071, "product": { "id": 17, "name": "Biobiljett SF Bio", "urlKey": "biobiljett-sf-bio", "price": 99, "stockQty": 500, "manufacturer": "Presentkort", "thumbnail": "http://example.com/media/thumbnail.jpg" }, "qty": 1, "rowTotal": 99, "rowTax": 5.6, "attributes": null, "options": null } ], "summary": { "subTotal": 99, "subTotalExclTax": 93.4, "grandTotal": 99, "grandTotalExclTax": 99, "tax": 5.6, "discount": null, "shippingAmount": 0, "shippingAmountExclTax": 0, "quoteCurrencyCode": "SEK", "qty": 1, "couponCode": null }, "virtual": true }
     */
    public function getCartItems()
    {
        return Mage::helper("API/Cart")->getCart();
    }

    /**
     * @api {post} /cart Add product to cart
     * @apiHeader {String} Content-Type application/json
     * @apiName createCartItem
     * @apiGroup Cart
     *
     * @apiParam {Object} product           The product to add to cart
     * @apiParam {Integer} product.id       The product id
     * @apiParam {Number} qty               Number of items to add
     * @apiParam {Object}  attributes  Map of attribute id => attribute value for configurable products, optional
     * @apiParam {Integer} attributes[attributeId] Attribute value
     *
     * @apiUse cartItems
     */
    public function createItem()
    {
        if($r = Mage::helper("API/cart")->updateItem($this->requestData())) {
            return $r;
        }

        return [200, $this->getCartItems()];
    }

    /**
     * @api {put} /cart/:id Update product cart item
     * @apiHeader {String} Content-Type application/json
     * @apiName replaceCartItem
     * @apiGroup Cart
     * @apiDescription Returns the contents of the whole cart despite just replacing one item, reason is that
     * removing/modifying a product can alter more than just the row itself.
     *
     * @apiParam {Number} id ID of cart item
     * @apiParam {Object} product
     * @apiParam {Integer} product.id
     * @apiParam {Number} qty New item quantity
     * @apiParam {Object}  attributes  Map of attribute id => attribute value for configurable products, optional
     * @apiParam {Integer} attributes[attributeId] Attribute value
     *
     * @apiUse cartItems
     */
    public function replaceItem($id) {
        $params = $this->requestData();

        // Null-check required on $params
        if (empty($params) || empty($params["qty"])) {
            return [400];
        }

        $params["id"] = $id;

        if($r = Mage::helper("API/cart")->updateItem($params)) {
            return $r;
        }

        return [200, $this->getCartItems()];
    }

    /**
     * @api {delete} /cart/:id Remove product from cart
     * @apiName deleteCartItem
     * @apiGroup Cart
     * @apiDescription Returns the contents of the whole cart despite just replacing one item, reason is that
     * removing/modifying a product can alter more than just the row itself.
     *
     * @apiParam {Number} id  ID of cart item
     *
     * @apiUse cartItems
     */
    public function deleteItem($id)
    {
        if (empty($id)) {
            return [404];
        }

        $this->getCart()->removeItem($id)->save();

        return [200, $this->getCartItems()];
    }

    /**
     * @api {get} /cart/:id Get product from cart
     * @apiName getCartItem
     * @apiGroup Cart
     *
     * @apiParam {Number} id  ID of cart item
     *
     * @apiSuccess {Integer} id
     * @apiSuccess {Object}  product  A lightweight product object
     * @apiSuccess {Integer} product.id
     * @apiSuccess {String}  product.sku
     * @apiSuccess {String}  product.name
     * @apiSuccess {String}  product.urlKey
     * @apiSuccess {Double}  product.price
     * @apiSuccess {mixed}   product.stockQty               Amount in stock, null if information not available
     * @apiSuccess {String}  product.manufacturer
     * @apiSuccess {String}  product.thumbnail  Complete URL to a thumbnail for the product
     * @apiSuccess {Double}  qty
     * @apiSuccess {Double}  rowTotal  Total price for the row
     * @apiSuccess {Object}  attributes  Map of attribute id => attribute value for configurable products
     * @apiSuccess {Integer} attributes[attributeId] Attribute value
     */
    public function getItem($id)
    {
        if (empty($id)) {
            return [404];
        }

        $cartItems  = Mage::getModel('checkout/cart')->getQuote()->getAllItems();

        foreach($cartItems as $item) {
            if($item->getId() == $id) {
                return [200, Mage::helper('API/Product')->prepareCartProduct($item)];
            }
        }

        return [404];
    }

    /**
     * @api {get} /cart Get cart
     * @apiName getCartItems
     * @apiGroup Cart
     *
     * @apiUse cartItems
     */
    public function getAll()
    {
      return [200, $this->getCartItems()];
    }

    /**
     * @api {delete} /cart Empty cart
     * @apiName emptyCart
     * @apiGroup Cart
     *
     * @apiSuccessExample {json} Empty-cart Response:
     * { "items": [], "summary": { "subTotal": 0, "subTotalExclTax": 0, "grandTotal": 0, "grandTotalExclTax": 0, "tax": null, "discount": null, "shippingAmount": 0, "quoteCurrencyCode": "SEK", "qty": null, "couponCode": null } }
     * @apiUse cartItems
     */
    public function deleteAll()
    {
        $cart  = $this->getCart();
        $items = $cart->getitems();

        foreach($items as $item) {
            $cart->removeItem($item->getItemId())->save();
        }

        return [200, $this->getCartItems()];
    }
}
