<?php

/**
 *
 * http://tech.dibspayment.com/batch/d2integratedpwhostedinputparametersstandard
 */
class Crossroads_API_Payment_Dibspw extends Crossroads_API_Payment_Abstract {

    /**
     * Instructs the client to take the data found in `fields` in the response
     * and POST that data to the URL found in `url`.
     */
    const ACTION_SUBMIT_FORM_DATA = "SUBMIT_FORM_DATA";

    public function isQuoteInvalid($_quote) {
        // Dibs does not have anything in particular to validate, so never invalid
        return false;
    }

    public function getOrderPaymentData($order) {
        if($order->getStatus() === Mage_Sales_Model_Order::STATE_PENDING_PAYMENT) {
            return $this->paymentFormData($order);
        }

        // Payment done, nothing to do
        return null;
    }

    protected function normalizeStreet($street) {
        return implode(" ", is_array($street) ? $street : [$street]);
    }

    protected function paymentFormData($order) {
        $orderId         = $order->getIncrementId();
        $storeId         = Mage::app()->getStore()->getId();
        $payType         = Mage::getStoreConfig("payment/Dibspw/DIBSPW_paytype", $storeId);
        $quote           = Mage::getModel("sales/quote")->load($order->getQuoteId());
        $billingAddress  = Mage::getModel('sales/order_address')->load($order->getBillingAddressId() ?: $order->getShippingAddressId());
        $shippingAddress = Mage::getModel('sales/order_address')->load($order->getShippingAddressId() ?: $order->getBillingAddressId());

        $data = [
            "acceptReturnUrl"     => $this->getAcceptCallback($orderId),
            "cancelReturnUrl"     => $this->getCancelCallback($orderId),
            "callbackUrl"         => $this->getSystemCallback($orderId),
            "amount"              => intval($order->getBaseGrandTotal() * 100),
            "currency"            => $quote->getQuoteCurrencyCode(),
            "language"            => Mage::app()->getLocale()->getLocaleCode(),
            "merchant"            => Mage::getStoreConfig("payment/Dibspw/DIBSPW_mid", $storeId),
            "orderId"             => $orderId,
            "billingFirstName"    => $billingAddress->getFirstname(),
            "billingLastName"     => $billingAddress->getLastname(),
            // No idea why Dibs wants the countryId on the billingAddress
            "billingAddress"      => $billingAddress->getCountryId(),
            "billingAddress2"     => $this->normalizeStreet($billingAddress->getStreet()),
            "billingPostalCode"   => $billingAddress->getPostcode(),
            "billingPostalPlace"  => $billingAddress->getCity(),
            "billingEmail"        => trim($order->getCustomerEmail()),
            "shippingFirstName"   => $shippingAddress->getFirstname(),
            "shippingLastName"    => $shippingAddress->getLastname(),
            // No idea why Dibs wants the countryId on the shippingAddress
            "shippingAddress"     => $shippingAddress->getCountryId(),
            "shippingAddress2"    => $this->normalizeStreet($shippingAddress->getStreet()),
            "shippingPostalCode"  => $shippingAddress->getPostcode(),
            "shippingPostalPlace" => $shippingAddress->getCity(),
            "captureNow"          => Mage::getStoreConfig("payment/Dibspw/DIBSPW_capturenow", $storeId) == "yes" ? 1 : 0,
            "yourRef"             => $orderId,
        ];

        if($payType) {
            $data["payType"] = $payType;
        }

        if(Mage::getStoreConfig("payment/Dibspw/DIBSPW_testmode") === 'yes') {
            $data["test"] = 1;
        }

        $data["MAC"] = self::encodeMac($data);

        return [
            "action" => self::ACTION_SUBMIT_FORM_DATA,
            "url"    => $this->getDibsAPIUrl(),
            "fields" => $data
        ];
    }

    // TODO: Configurable for test
    protected function getDibsAPIUrl() {
        return "https://sat1.dibspayment.com/dibspaymentwindow/entrypoint";
    }
    protected function getAcceptCallback($orderId) {
        return Mage::getUrl("", [
            "_direct"      => "api/payment/dibs/accept_order/orderId/".urlencode($orderId),
            "_use_rewrite" => false,
        ]);
    }
    protected function getCancelCallback($orderId) {
        return Mage::getUrl("", [
            "_direct"      => "api/payment/dibs/cancel_order/orderId/".urlencode($orderId),
            "_use_rewrite" => false,
        ]);
    }
    protected function getSystemCallback($orderId) {
        return Mage::getUrl("", [
            "_direct"      => "api/payment/dibs/system_order/orderId/".urlencode($orderId),
            "_use_rewrite" => false,
        ]);
    }

    protected static function getMacKey() {
        return pack("H*", Mage::getStoreConfig("payment/Dibspw/DIBSPW_HMAC", Mage::app()->getStore()->getId()));
    }

    public static function encodeMac($msg) {
        ksort($msg, SORT_STRING);

        return hash_hmac("sha256", urldecode(http_build_query($msg)), self::getMacKey());
    }
}
