<?php

class Crossroads_API_Helper_Product {
    protected $smallImageSize;
    protected $largeImageSize;
    protected $thumbImageSize;
    protected $getStockQty        = false;
    protected $getMediaGallery    = false;
    protected $getRelatedProducts = false;
    protected $TYPES_WITH_OPTIONS = ["configurable", "bundle"];

    public function __construct()
    {
        $store_id                    = Mage::app()->getStore()->getId();

        $this->largeImageSize        = Mage::getStoreConfig('API_section/images/large_image_size', $store_id);
        $this->smallImageSize        = Mage::getStoreConfig('API_section/images/small_image_size', $store_id);
        $this->thumbImageSize        = Mage::getStoreConfig('API_section/images/thumbnail_image_size', $store_id);
        $this->getStockQty           = Mage::getStoreConfig('API_section/product_view/get_stock_qty', $store_id);
        $this->getMediaGallery       = Mage::getStoreConfig('API_section/product_view/get_media_gallery', $store_id);
        $this->getRelatedProducts    = Mage::getStoreConfig('API_section/product_view/get_related_products', $store_id);
        $this->getOptionsInList      = Mage::getStoreConfig('API_section/product_list/get_options_in_list', $store_id);
        $this->productAttributesView = unserialize(Mage::getStoreConfig('API_section/product_view/attributes', $store_id));
        $this->productAttributesList = unserialize(Mage::getStoreConfig('API_section/product_list/attributes', $store_id));
    }

    protected function getProductAttributes($product, $config)
    {
        if(! $config) {
            return new stdClass();
        }

        $attributes = [];

        foreach ($config as $attribute) {
            if(!is_array($attribute) ||
                !array_key_exists("attribute_code", $attribute) ||
                !array_key_exists("frontend_input", $attribute)) {
                continue;
            }

            $code  = $attribute["attribute_code"];
            $input = $attribute["frontend_input"];

            switch($input) {
                case "select":
                    $attributes[$code] = $product->getAttributeText($code) ?: null;
                    break;
                case "multiselect":
                    $attributes[$code] = $product->getAttributeText($code) ?: null;
                    break;
                default:
                    $attributes[$code] = $product->getAttribute($code);
            }
        }

        return $attributes ?: new stdClass();
    }


    /**
     * Prepares a product for viewing in a category-listing, prepared for JSON-serialization.
     *
     * @return Array Associative array of product data
     */
    public function prepareListProduct($product)
    {
        $store_id    = Mage::app()->getStore()->getId();
        $small_image = (string)Mage::helper("catalog/image")
            ->init($product, 'small_image')
            ->resize($this->smallImageSize);

        $msrp = (double)$this->getMsrp($product);

        $discount_percent = $product->getShowDiscount() && $msrp
            ? floor((1 - ((double)$product->getMinimalPrice() / $msrp)) * 100)
            : null;

        return [
            "id"               => (int) $product->getEntityId(),
            "type"             => $product->getTypeId(),
            "name"             => $product->getName(),
            "sku"              => $product->getSku(),
            "urlKey"           => $product->getUrlKey(),

            "price"            => (double)$product->getMinimalPrice(),
            "specialPrice"     => $product->hasSpecialPrice() ? ((double)$product->getSpecialPrice()) : null,
            "msrp"             => $msrp,
            "discountPercent"  => $discount_percent,

            "shortDescription" => $product->getShortDescription(),
            "smallImage"       => $small_image,
            "isSalable"        => $product->getIsSalable() > 0,
            "options"          => $this->getOptionsInList ? $this->prepareOptions($product) : null,
            "attributes"       => $this->getProductAttributes($product, $this->productAttributesList)
        ];
    }

    /**
     * Converts a Product into an associative array ready for JSON-serialization.
     *
     * @return Array Associative array of product data
     */
    public function prepareProductDetail($product) {
        $store_id        = Mage::app()->getStore()->getId();
        $stockQty        = null;
        $mediaGallery    = null;
        $relatedProducts = null;
        $price            = (double)$product->getFinalPrice();

        if ($this->getStockQty) {
            $stockQty = (double)Mage::getModel('cataloginventory/stock_item')
                ->loadByProduct($product)
                ->getQty();
        }

        if (Mage::getStoreConfig('API_section/product_view/get_media_gallery', $store_id)) {
            $mediaGallery = $this->prepareMediaGallery($product);
        }

        if (Mage::getStoreConfig('API_section/product_view/get_related_products', $store_id)) {
            $relatedProducts = $this->prepareRelatedProducts($product);
        }

        $small_image     = (string)Mage::helper("catalog/image")
            ->init($product, 'small_image')
            ->resize($this->smallImageSize);
        $large_image     = (string)Mage::helper("catalog/image")
            ->init($product, 'image')
            ->resize($this->largeImageSize);
        $original_image = (string)Mage::helper("catalog/image")
            ->init($product, 'image');

        $msrp = (double)$this->getMsrp($product);

        $discount_percent = $product->getShowDiscount() && $msrp
            ? floor((1 - ((double)$price / $msrp)) * 100)
            : null;

        $inStock = $product->getIsInStock() > 0;

        // when the configurable products children is out of stock the price
        // gets calculated to "0". To avoid loading each child we simply assume
        // the product is out of stock.
        if ($product->getTypeId() === "configurable" && !$price) {
            $inStock = false;
        }

        return [
            "id"               => (int) $product->getEntityId(),
            "type"             => $product->getTypeId(),
            "name"             => $product->getName(),
            "sku"              => $product->getSku(),

            "price"            => $price,
            "specialPrice"     => $product->hasSpecialPrice() ? ((double)$product->getSpecialPrice()) : null,
            "msrp"             => $msrp,
            "discountPercent"  => $discount_percent,
            "stockQty"         => $stockQty,

            "metaDescription"  => $product->getMetaDescription(),
            "shortDescription" => $product->getShortDescription(),
            "description"      => $product->getDescription(),
            "smallImage"       => $small_image,
            "largeImage"       => $large_image,
            "originalImage"    => $original_image,
            "isSalable"        => $product->getIsSalable() > 0,
            "isInStock"        => $inStock,
            "options"          => $this->prepareOptions($product),
            "relatedProducts"  => $relatedProducts,
            "mediaGallery"     => $mediaGallery,
            "attributes"       => $this->getProductAttributes($product, $this->productAttributesView)
        ];
    }

    /**
     * Converts a Cart item into an associative array ready for JSON-serialization.
     *
     * @return Array Associative array of product cart data
     */
    public function prepareCartProduct($item)
    {
        $store_id = Mage::app()->getStore()->getId();
        $product  = $item->getProduct();
        $stockQty = null;
        $rowTotal = $item->getRowTotal()
                  + $item->getTaxAmount()
                  + $item->getHiddenTaxAmount()
                  - $item->getDiscountAmount();

        $thumbnail = (string)Mage::helper("catalog/image")
            ->init($product, 'thumbnail')
            ->resize($this->thumbImageSize);

        if ($this->getStockQty) {
            $stockQty = (double)Mage::getModel('cataloginventory/stock_item')
                ->loadByProduct($product)
                ->getQty();
        }

        $attributes = null;
        $options    = $product->getTypeInstance(true)->getOrderOptions($product);

        if( ! empty($options["info_buyRequest"]) && is_array($options["info_buyRequest"]["super_attribute"])) {
            $attributes = [];

            foreach($options["info_buyRequest"]["super_attribute"] as $k => $v) {
                $attributes["$k"] = (int)$v;
            }
        }

        return [
            "id"        => (int)$item->getId(),
            "product"   => [
                "id"           => (int)$product->getId(),
                "sku"          => $product->getSku(),
                "name"         => $product->getName(),
                "urlKey"       => $product->getUrlPath(),
                "price"        => (double)$item->getPriceInclTax(),
                "stockQty"     => $stockQty,
                "manufacturer" => $product->getAttributeText("manufacturer"),
                "thumbnail"    => $thumbnail
            ],
            "qty"        => (double)$item->getQty(),
            "rowTotal"   => (double)$rowTotal,
            "rowTax"     => (double)$item->getTaxAmount(),
            "attributes" => $attributes,
            "options"    => $this->prepareOptionsCart($product, $options)
        ];
    }

    /**
     * Prepares product options for complex products from cart data.
     */
    protected function prepareOptionsCart($product, $options) {
        if (!in_array($product->getTypeId(), $this->TYPES_WITH_OPTIONS)) {
            return null;
        }
        if(empty($options["info_buyRequest"]) || !is_array($options["info_buyRequest"]["super_attribute"])) {
            return null;
        }

        $attrs      = $options["info_buyRequest"]["super_attribute"];
        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);

        return array_map(function($option) use($attrs) {
            $value = null;

            foreach($option["values"] as $v) {
                if((int)$attrs[$option["attribute_id"]] === (int)$v["value_index"]) {
                    $value = $v;

                    break;
                }
            }

            return [
                "id"           => (int)$option["attribute_id"],
                "code"         => $option["attribute_code"],
                "title"        => $option["store_label"] ?: $option["frontend_label"] ?: $option["label"],
                "useAsDefault" => $option["use_default"] > 0,
                "position"     => (int)$option["position"],
                "value"        => $value ? [
                    "id"        => (int)$value["value_index"],
                    "label"     => $value["store_label"] ?: $value["label"] ?: $value["default_label"],
                    "isPercent" => $value["is_percent"] > 0,
                    "price"     => $value["pricing_value"]
                ] : null
            ];
        }, $rawOptions);
    }

    protected function getMsrp(Mage_Catalog_Model_Product $product) {
        if (!in_array($product->getTypeId(), $this->TYPES_WITH_OPTIONS)) {
            return $product->getMsrp();
        }

        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = array_filter($instance->getUsedProducts(null, $product), function($p) {
            return $p->isSaleable();
        });

        $msrps = [];

        foreach($rawOptions as $option) {
            foreach($children as $child) {
                foreach ($option["values"] as $value)  {
                    if ($child->getData($option["attribute_code"]) === $value["value_index"]) {
                        $msrp = $child->getMsrp();
                        if ($msrp) {
                            $msrps[] = $msrp;
                        }
                    }
                }
            }
        }

        return !empty($msrps) ? min($msrps) : null;
    }

    /**
     * @param $product
     * @return array
     */
    protected function prepareOptions(Mage_Catalog_Model_Product $product)
    {
        if (!in_array($product->getTypeId(), $this->TYPES_WITH_OPTIONS)) {
            return null;
        }

        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = array_filter($instance->getUsedProducts(null, $product), function($p) {
            return $p->isSaleable();
        });

        return array_map(function($option) use($children) {
            return [
                "id"           => (int)$option["attribute_id"],
                "code"         => $option["attribute_code"],
                "title"        => $option["store_label"] ?: $option["frontend_label"] ?: $option["label"],
                "useAsDefault" => $option["use_default"] > 0,
                "position"     => (int)$option["position"],
                "values"       => array_values(array_filter(array_map(function($value) use($children, $option) {
                    // Find the correct child product and output its data
                    foreach($children as $child) {

                        $small_image = null;
                        $large_image = null;
                        $original_image = null;

                        $small_image     = (string)Mage::helper("catalog/image")
                            ->init($child, 'small_image')
                            ->resize($this->smallImageSize);
                        $large_image     = (string)Mage::helper("catalog/image")
                            ->init($child, 'image')
                            ->resize($this->largeImageSize);
                        $original_image = (string)Mage::helper("catalog/image")
                            ->init($child, 'image');

                        if($child->getData($option["attribute_code"]) === $value["value_index"]) {
                            return [
                                "id"             => (int)$value["value_index"],
                                "smallImage"     => $small_image,
                                "largeImage"     => $large_image,
                                "originalImage"  => $original_image,
                                "label"          => $value["store_label"] ?: $value["label"] ?: $value["default_label"],
                                "isPercent"      => $value["is_percent"] > 0,
                                "price"          => (double)$child->getPrice(),
                                "msrp"           => (double)$child->getMsrp()
                            ];
                        }
                    }

                    // Empty, array_filter above will remove empty entries
                    return null;
                }, $option["values"])))
            ];
        }, $rawOptions);
    }

    protected function prepareMediaGallery($product)
    {
        $coll = $product->getMediaGalleryImages();

        if( ! $coll) {
            return [];
        }

        return array_values(array_map(function($image) use ($product) {
            return [
                "thumbnail" => (string)Mage::helper("catalog/image")
                    ->init($product, 'image', $image->getFile())
                    ->resize($this->thumbImageSize),
                "image"     => (string)Mage::helper("catalog/image")
                    ->init($product, 'image', $image->getFile())
                    ->resize($this->smallImageSize),
            ];
        }, $coll->getItems()));
    }

    protected function prepareRelatedProducts($product)
    {
        $relatedProducts           = [];
        $relatedProductsCollection = $product->getRelatedProductCollection()
            ->addAttributeToSelect('*')
            ->setPositionOrder()
            ->addStoreFilter(Mage::app()->getStore()->getStoreId());

        // TODO: Format the related products properly
        foreach ($relatedProductsCollection as $relatedProduct) {
            $relatedProduct->setThumbnail(
                (string)Mage::helper('catalog/image')
                    ->init($relatedProduct, 'small_image')
                    ->resize(285)
            );

            $relatedProductData       = $relatedProduct->getData();
            $relatedProductData['id'] = $relatedProduct->getData('entity_id');

            $relatedProducts[]        = $relatedProductData;
        }

        return $relatedProducts;
    }
}
