<?php

class Crossroads_API_Controller_Super extends Mage_Core_Controller_Front_Action
{
    /**
     * For outputting a JSON response
     *
     * @param int $status, HTTP code
     * @param mixed $data, the response data
     */
    protected function sendData($data) {
        if(!is_array($data) || !$data[0]) {
            throw new Exception("Missing return array.");
        }

        assert(gettype($data[0]) === "integer", __METHOD__.": \$status must be of type integer");

        $this->getResponse()->setHttpResponseCode($data[0]);

        if(array_key_exists(1, $data)) {
            $this->getResponse()
                ->setHeader('Content-Type', 'application/json', true)
                ->setBody(json_encode($data[1]));
        }
    }

    /**
     * For outputting a JSON error
     *
     * @param int    $status, HTTP code
     * @param string $message, message
     * @param array  $data, attached data
     */
    public static function formatError($status, $message, $data = null, $error_code = null)
    {
        assert(gettype($status) === "integer", __METHOD__.": \$status must be of type integer");

        return [$status, [
            "message"   => $message,
            "errorCode" => $error_code,
            "data"      => $data
        ]];
    }

    /**
     * Get JSON request data as an assoc array, will output null if Content-Type
     * is not application/json.
     *
     * @return array|null attached data, null if not application/json
     */
    protected function requestData() {
        $request      = $this->getRequest();
        $content_type = $request->getHeader("Content-Type");

        if(strtolower(trim($content_type)) === "application/json") {
            return json_decode($request->getRawBody(), true);
        }

        return null;
    }

    protected function handleException($e) {
        $this->getResponse()->setHttpResponseCode(500);

        Mage::logException($e);

        if(Mage::getIsDeveloperMode()) {
            $this->getResponse()
                ->setHeader('Content-Type', 'application/json', true)
                ->setBody(json_encode([
                    "message"   => $e->getMessage(),
                    "errorCode" => null,
                    "data"      => [
                        "code"  => $e->getCode(),
                        "file"  => $e->getFile(),
                        "line"  => $e->getLine(),
                        "trace" => array_map(function($l) {
                            if(array_key_exists("class", $l)) {
                                return [
                                    "call" => $l["class"].$l["type"].$l["function"],
                                    "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                    "line" => array_key_exists("line", $l) ? $l["line"] : null,
                                ];
                            }

                            return [
                                "call" => $l["function"],
                                "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                "line" => array_key_exists("line", $l) ? $l["line"] : null,
                            ];
                        }, $e->getTrace())
                    ]
                ]));
        }
    }
}
