<?php

/**
 * Controller for managing responses from Dibs. Routed through `Crossroads_API_PaymentController`.
 */
class Crossroads_API_Payment_DibsController extends Crossroads_API_Controller_Super {
    /**
     * Type for order payments.
     */
    const TYPE_PAYMENT = "TYPE_PAYMENT";

    const PAYMENT_CANCEL = "PAYMENT_CANCEL";
    const PAYMENT_ACCEPT = "PAYMENT_ACCEPT";
    const PAYMENT_SYSTEM = "PAYMENT_SYSTEM";

    /**
     * Blacklist of parameters to exclude while calculating the MAC.
     */
    protected $PARAMETER_BLACKLIST = ["MAC"];

    /**
     * Validation-specification depending on type.
     */
    protected $CALLBACK_FIELDS = [
        self::TYPE_PAYMENT => [
            "merchant" => ["type" => "string", "required" => true],
            "orderId"  => ["type" => "string", "required" => true],
            "status"   => ["type" => "string", "required" => true],
        ]
    ];

    protected function getLogFile() {
        return "dibs_callback_".date("Ymd").".log";
    }

    protected function logDibsFail($logType, $method, $message, $data = null) {
        Mage::log("$method: $message", $logType, $this->getLogFile(), true);
        if($data) {
            Mage::log("$method: ".print_r($data, true), $logType, $this->getLogFile(), true);
        }
    }

    /**
     * Retrieves orderId from query-parameters.
     *
     * @return string
     */
    protected function getOrderIncrementId() {
        $req = $this->getRequest();

        return $req->getParam("orderId") ?: $req->getParam("orderid");
    }

    /**
     * Verifies the data supplied in parameters for the specific payment type.
     */
    protected function verifyDibsData($params, $type) {
        if(!array_key_exists($type, $this->CALLBACK_FIELDS)) {
            throw new Exception("Crossroads API Dibs: Invalid callback type '$type' supplied to ".__METHOD__.".");
        }

        if(empty($params["MAC"])) {
            return false;
        }

        $macParam = $params["MAC"];

        foreach($params as $k => $v) {
            if(in_array($k, $this->PARAMETER_BLACKLIST)) {
                unset($params[$k]);
            }
        }

        $macCalc = Crossroads_API_Payment_Dibspw::encodeMac($params);

        if($macCalc !== $macParam) {
            $this->logDibsFail(LOG_ERR, __METHOD__, "Failed to match MACs", [
                "inputParams" => $params,
                "MAC"         => $macParam,
                "calculated"  => $macCalc
            ]);

            return false;
        }

        $result = Mage::helper("API/validation")->validateMap($this->CALLBACK_FIELDS[$type], $params);

        if(!$result->isValid()) {
            $this->logDibsFail(LOG_ERR, __METHOD__, "Failed to validate parameters", [
                "inputParams" => $params,
                "MAC"         => $macParam,
                "messages"    => $result->getMessages()
            ]);

            return false;
        }

        return true;
    }

    /**
     * Constructs data for insertion into dibs_pw_results.
     */
    protected function prepareDibsPwData($orderId, $params, $type) {
        return [
            'orderid'                     => $orderId,
            'status'                      => empty($params['status']) ? '' : $params['status'],
            'testmode'                    => empty($params['test']) ? 0 : $params['test'],
            'transaction'                 => empty($params['transaction']) ? '' : $params['transaction'],
            'amount'                      => empty($params['amount']) ? 0 : $params['amount'],
            'currency'                    => empty($params['currency']) ? 'SEK' : $params['currency'],
            'fee'                         => empty($params['fee']) ? 0 : $params['fee'],
            'voucheramount'               => empty($params['voucherAmount']) ? 0 : $params['voucherAmount'],
            'paytype'                     => empty($params['cardTypeName']) ? '' : $params['cardTypeName'],
            'amountoriginal'              => empty($params['amountOriginal']) ? 0 : $params['amountOriginal'],
            'ext_info'                    => serialize($params),
            'validationerrors'            => empty($params['validationErrors']) ? '' : $params['validationErrors'],
            'capturestatus'               => empty($params['capturestatus']) ? '' : $params['capturestatus'],
            'actioncode'                  => empty($params['actionCode']) ? '' : $params['actionCode'],
            'success_action'              => $type === self::PAYMENT_SYSTEM ? (strtolower($params["status"]) === "accepted") : ($type === self::PAYMENT_ACCEPT),
            'cancel_action'               => $type === self::PAYMENT_SYSTEM ? (strtolower($params["status"]) === "cancelled") : ($type === self::PAYMENT_CANCEL),
            'callback_action'             => $type === self::PAYMENT_SYSTEM,
            'success_error'               => '',
            'callback_error'              => '',
            'sysmod'                      => empty($params['s_sysmod']) ? '' : $params['s_sysmod'],
            'acquirerDeliveryAddress'     => empty($params['acquirerDeliveryAddress']) ? '' : $params['acquirerDeliveryAddress'],
            'acquirerDeliveryCountryCode' => empty($params['acquirerDeliveryCountryCode']) ? '' : $params['acquirerDeliveryCountryCode'],
            'acquirerDeliveryPostalCode'  => empty($params['acquirerDeliveryPostalCode']) ? '' : $params['acquirerDeliveryPostalCode'],
            'acquirerDeliveryPostalPlace' => empty($params['acquirerDeliveryPostalPlace']) ? '' : $params['acquirerDeliveryPostalPlace'],
            'acquirerFirstName'           => empty($params['acquirerFirstName']) ? '' : $params['acquirerFirstName'],
            'acquirerLastName'            => empty($params['acquirerLastName']) ? '' : $params['acquirerLastName']
        ];
    }

    protected function upsertDibsPwResult($data) {
        $conn    = Mage::getSingleton("core/resource")->getConnection("core_write");
        $columns = implode(", ", array_map(function($k) use($conn) {
            return $conn->quoteIdentifier($k);
        }, array_keys($data)));
        $keys    = implode(", ", array_map(function($k) {
            return ":$k";
        }, array_keys($data)));

        $query = "INSERT INTO dibs_pw_results ($columns) VALUES ($keys) ON DUPLICATE KEY UPDATE
            ext_info         = VALUES(ext_info),
            validationerrors = VALUES(validationerrors),
            capturestatus    = VALUES(capturestatus),
            actioncode       = VALUES(actioncode),
            success_action   = VALUES(success_action),
            cancel_action    = VALUES(cancel_action),
            callback_action  = VALUES(callback_action),
            success_error    = VALUES(success_error),
            callback_error   = VALUES(callback_error);";

        if(!$conn->query($query, $data)->rowCount()) {
            throw new Exception("Crossroads API Dibs: Exception while saving dibs data.\nQuery: $query\n".print_r($data, true));
        }
    }

    /**
     * @api {post} /api/payment/dibs/accept_order?orderId=:orderId Dibs order accept callback
     * @apiName paymentDibsAcceptOrder
     * @apiGroup Payment Callbacks
     * @apiDescription This endpoing receives a redirected client from dibs after payment was accepted.
     *
     * @apiParam {String} orderId Order incrementId, can also be supplied on `orderid` query-parameter.
     */
    public function accept_orderAction() {
        if($this->getRequest()->getMethod() !== "POST") {
            return $this->sendData([405]);
        }

        $storeId = Mage::app()->getStore()->getId();
        $orderId = $this->getOrderIncrementId();
        $params  = $this->getRequest()->getParams();

        if($r = $this->performOrderOperation($orderId, $params, self::PAYMENT_ACCEPT)) {
            return $this->sendData($r);
        }

        $this->getResponse()->setRedirect(Mage::getUrl("", [
            "_direct"      => Mage::getStoreConfig("API_section/payment_callbacks/payment_callback_success_redirect", $storeId),
            "_use_rewrite" => false,
            "_query"       => ["orderId" => $orderId]
        ]));
    }

    /**
     * @api {post} /api/payment/dibs/cancel_order?orderId=:orderId Dibs order cancel callback
     * @apiName paymentDibsCancelOrder
     * @apiGroup Payment Callbacks
     * @apiDescription This endpoing receives a redirected client from dibs after payment was cancelled.
     *
     * @apiParam {String} orderId Order incrementId, can also be supplied on `orderid` query-parameter.
     */
    public function cancel_orderAction() {
        if($this->getRequest()->getMethod() !== "POST") {
            return $this->sendData([405]);
        }

        $storeId = Mage::app()->getStore()->getId();
        $orderId = $this->getOrderIncrementId();
        $params  = $this->getRequest()->getParams();

        if($r = $this->performOrderOperation($orderId, $params, self::PAYMENT_CANCEL)) {
            return $this->sendData($r);
        }

        $this->getResponse()->setRedirect(Mage::getUrl("", [
            "_direct"      => Mage::getStoreConfig("API_section/payment_callbacks/payment_callback_cancel_redirect", $storeId),
            "_use_rewrite" => false,
            "_query"       => ["orderId" => $orderId]
        ]));
    }

    /**
     * @api {post} /api/payment/dibs/system_order?orderId=:orderId Dibs order system callback
     * @apiName paymentDibsSystemOrder
     * @apiGroup Payment Callbacks
     * @apiDescription This endpoint is called out of band from Dibs at times.
     *
     * @apiParam {String} orderId Order incrementId, can also be supplied on `orderid` query-parameter.
     */
    public function system_orderAction() {
        if($this->getRequest()->getMethod() !== "POST") {
            return $this->sendData([405]);
        }

        $orderId = $this->getOrderIncrementId();
        $params  = $this->getRequest()->getParams();

        if($r = $this->performOrderOperation($orderId, $params, self::PAYMENT_SYSTEM)) {
            return $this->sendData($r);
        }

        return $this->sendData(self::formatError(200, "Successfully updated order"));
    }

    protected function performOrderOperation($orderId, $params, $type) {
        if(empty($orderId)) {
            return self::formatError(400, "Missing orderId parameter", null, 6000);
        }

        if( ! $this->verifyDibsData($params, self::TYPE_PAYMENT)) {
            return self::formatError(403, "Dibs data did not verify", null, 6001);
        }

        $data = $this->prepareDibsPwData($orderId, $params, $type);

        $this->upsertDibsPwResult($data);

        $order = Mage::getModel("sales/order")->loadByIncrementId($orderId);

        if(!$order || !$order->getIncrementId() || $order->getIncrementId() != $orderId) {
            return self::formatError(404, "Order not found", null, 6002);
        }

        if($data["success_action"] && $order->getState() === Mage_Sales_Model_Order::STATE_NEW) {
            $this->convertOrder($order);

            $order->setState(Mage_Sales_Model_Order::STATE_PROCESSING, true)->save();

            if (Mage::getStoreConfig('sales_email/order/enabled') == 1) {
                $order->sendNewOrderEmail();
            }

            $order->save();
        }
        else if ($data["cancel_action"]) {
            $order->cancel();
            $order->addStatusHistoryComment("Cancelled by Dibs callback")
                ->setIsVisibleOnFront(false)
                ->setIsCustomerNotified(false);
            $order->save();
        }
    }

    protected function convertOrder($order)
    {
        $incrementId = $order->getIncrementId();
        $orderId     = $order->getId();

        // Make sure the order has Dibspw as the payment method
        $sqlQuery = "UPDATE sales_flat_order_payment SET method = 'Dibspw' WHERE parent_id = :order_id";
        Mage::getSingleton('core/resource')
            ->getConnection('core_write')
            ->query($sqlQuery, array('order_id' => $orderId));
    }
}
