<?php

class Crossroads_API_Helper_Cart
{
    public function getCouponData() {
        $quote  = Mage::getModel('checkout/cart')->getQuote();

        if( ! $quote->getCouponCode()) {
            return null;
        }

        $coupon = Mage::getModel("salesrule/coupon")->load($quote->getCouponCode(), "code");

        if(!$coupon || !$coupon->getRuleId()) {
            return null;
        }

        $rule = Mage::getModel("salesrule/rule")->load($coupon->getRuleId());

        return [
            "couponCode" => $quote->getCouponCode(),
            "ruleName"   => $rule->getName(),
        ];
    }

    public function getCart()
    {
        $quote = Mage::getModel('checkout/cart')->getQuote();

        $quote->collectTotals();

        return $this->formatQuote($quote);
    }

    public function formatQuote($quote) {
        $items    = $quote->getAllVisibleItems();
        $qty      = $quote->getQty();
        $totals   = $quote->getTotals();
        $shipping = $quote->getShippingAddress();

        foreach ($items as $item) {
            $qty += $item->getQty();
        }

        return [
            "items"      => array_map(function($p) {
                return Mage::helper("API/Product")->prepareCartProduct($p);
            }, $items),
            "summary"    => [
                "subTotal"              => array_key_exists("subtotal", $totals) ? (double) $totals["subtotal"]->getValueInclTax() : null,
                "subTotalExclTax"       => array_key_exists("subtotal", $totals) ? (double) $totals["subtotal"]->getValueExclTax() : null,
                "grandTotal"            => (double)$quote->getGrandTotal(),
                "grandTotalExclTax"     => (double)$quote->getGrandTotal() - (double)$quote->getShippingAddress()->getTaxAmount(),
                "tax"                   => array_key_exists("tax", $totals) ? (double) $totals["tax"]->getValue() : null,
                "discount"              => array_key_exists("discount", $totals) ? (double) $totals["discount"]->getValue() : null,
                // TODO: Implement, seems to require implementation of support for an invoiceFee line for totals calculation
                "invoiceFee"            => null,
                "shippingAmount"        => (double)$shipping->getShippingInclTax() ?: $shipping->getShippingAmount(),
                "shippingAmountExclTax" => (double)$shipping->getShippingAmount(),
                "quoteCurrencyCode"     => $quote->getQuoteCurrencyCode(),
                "qty"                   => $qty,
                "coupon"                => $this->getCouponData(),
                "virtual"               => $quote->isVirtual()
            ]
        ];
    }

    public function updateItem($item)
    {
        $session    = Mage::getSingleton("checkout/session");
        $request    = Mage::app()->getRequest();
        $response   = Mage::app()->getResponse();
        $cart       = Mage::getSingleton('checkout/cart');
        $attributes = null;

        if(!is_array($item)) {
            return Crossroads_API_Controller_Super::formatError(400, "item is not an associative array, got '".gettype($item)."'.", null, 2008);
        }

        // map attributes
        if (array_key_exists("attributes", $item) && $item["attributes"] !== null) {
            if(!is_array($item["attributes"])) {
                return Crossroads_API_Controller_Super::formatError(400, "item.attributes should be a map, got '".gettype($item["attributes"])."'.", null, 2003);
            }

            if( ! empty($item["attributes"])) {
                $attributes = [];

                foreach($item["attributes"] as $k => $v) {
                    $attributes[(int)$k] = (int)$v;
                }
            }
        }

        // Null-check required on $item
        if (empty($item) || empty($item['qty']) || empty($item['product'])) {
            return Crossroads_API_Controller_Super::formatError(400, "item is missing qty and/or product parameters.", null, 2004);
        }

        if( ! is_array($item["product"]) || !array_key_exists("id", $item["product"])) {
            return Crossroads_API_Controller_Super::formatError(400, "item is missing product key, or product key does not contain an object with an id property.", null, 2005);
        }

        $productId = (int)$item["product"]["id"];
        $product   = Mage::getModel('catalog/product')
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load($productId);

        if (!$product || !$product->getId()) {
            return Crossroads_API_Controller_Super::formatError(404, "Product not found.", null, 2006);
        }

        if(!empty($item["id"])) {
            $id = (int)$item["id"];

            try {
                $cart->updateItem($id, [
                    "product"         => $productId,
                    "qty"             => $item["qty"],
                    "super_attribute" => $attributes
                ]);
            }
            catch(Exception $e) {
                // Can only happen in update product
                if($e->getMessage() === Mage::helper('checkout')->__('Quote item does not exist.')) {
                    return Crossroads_API_Controller_Super::formatError(400, "Cart item with id '$id' was not found when attempting to update cart.", null, 2007);
                }

                // Determnine if we return an error to the user or rethrow
                return $this->handleProductException($e, $product);
            }

            $cart->save();
            $session->setCartWasUpdated(true);

            Mage::dispatchEvent('checkout_cart_update_item_complete', [
                'product'  => $product,
                'request'  => $request,
                'response' => $response
            ]);
        }
        else {
            try {
                $cart->addProduct($product, [
                    "product"         => $productId,
                    "qty"             => $item["qty"],
                    "super_attribute" => $attributes
                ]);
            } catch (Exception $e) {
                // Determnine if we return an error to the user or rethrow
                return $this->handleProductException($e, $product);
            }

            $cart->save();
            $session->setCartWasUpdated(true);

            Mage::dispatchEvent('checkout_cart_add_product_complete', [
                'product'  => $product,
                'request'  => $request,
                'response' => $response
            ]);
        }
    }

    /**
     * Handles exceptions caused in $cart->addProduct or $cart->updateItem gracefully, produces an error message
     * with the correct HTTP-status and an errorCode if the error is caused by user input. Will rethrow if it
     * is a system error.
     *
     * @param  Exception
     * @param  Product  The product used when encountering the exception
     * @return Array Http response, code first element and data second
     */
    protected function handleProductException($e, $product) {
        $msg = $e->getMessage();
        // Magento is not sane, it is using stringly TRANSLATED typed exceptions
        if($msg === Mage::helper('catalog')->__('Please specify the product\'s option(s).') ||
           $msg === Mage::helper('catalog')->__('Please specify the product\'s required option(s).'))  {
            return Crossroads_API_Controller_Super::formatError(400, "Please specify the product's option(s).", null, 2002);
        }

        if($msg === Mage::helper('checkout')->__('The product could not be found.')) {
            return Crossroads_API_Controller_Super::formatError(400, "Product coult not be found", null, 2009);
        }

        if($msg === Mage::helper('cataloginventory')->__('This product is currently out of stock.')) {
            return Crossroads_API_Controller_Super::formatError(400, "Product is currently out of stock", null, 2000);
        }

        if(stripos($msg, Mage::helper('sales')->__('Item qty declaration error.')) !== false ||
           stripos($msg, Mage::helper('cataloginventory')->__('Item qty declaration error.\nThe requested quantity for \"%s\" is not available.', $product->getName())) !== false ||
           stripos($msg, Mage::helper('cataloginventory')->__('The requested quantity for "%s" is not available.', $product->getName())) !== false) {
            return Crossroads_API_Controller_Super::formatError(400, "Product quantity is not available.", null, 2001);
        }

        throw $e;
    }
}