<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;
use GraphQL\Type\Definition\Type;
use GraphQL\Type\Definition\UnionType;
use MageQL\Context;
use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class UnionBuilder extends AbstractBuilder {
    /**
    * @var Array<string>
    */
    protected $types = [];

    /**
    * @var callable
    */
    protected $resolveType;

    public function __construct(array $params = []) {
        parent::__construct($params["description"]);

        $this->types = $params["types"];
        $this->resolveType = $params["resolveType"];
    }

    public function createInstance(Registry $registry, string $typeName): Type {
        return new UnionType([
            "resolveType" =>  $this->resolveType,
            "name" => $typeName,
            "description" => $this->description,
            "types" => function () use($registry) {
                $types = [];

                foreach($this->types as $name) {
                    $types[] = $registry->getType($name);
                }

                return $types;
            },
        ]);
    }
}

class Crossroads_AdventCalendar_Model_Schema extends MageQL_Core_Model_Schema_Abstract {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        /*
        case "Category":
            return $this->object("A category containing products")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");
         */
        case "AdventCalendarDoorToday":
            return $this->object("A category containing products")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");
        case "AdventCalendarDoorOpen":
            return $this->object("A category containing products")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");
        case "AdventCalendarDoorLocked":
            return $this->object("A locked category containing only the image")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");
        case "AdventCalendarDoor":
            return new UnionBuilder([
                "resolveType" =>
                /**
                 * @param Mage_Catalog_Model_Category | Varien_Object $obj
                */
                function($obj) {
                    if($obj->today) {
                        return "AdventCalendarDoorToday";
                    }

                    if($obj->open) {
                        return "AdventCalendarDoorOpen";
                    }

                    return "AdventCalendarDoorLocked";
                },
                "description" => "A calendar door. Might be locked and only contain an image.",
                "types" => ["AdventCalendarDoorOpen", "AdventCalendarDoorToday", "AdventCalendarDoorLocked"],
            ]);
        default:
            return null;
        }
    }

  public function getTypeFields(string $typeName, Registry $registry): array {
    switch($typeName) {
        case "AdventCalendarDoorToday":
        case "AdventCalendarDoorOpen":
            return [
                "category" => $this->field("Category!", "The category for this door"),
                "day" => $this->field("Int", "The day of the door"),
                "variant" => $this->field("String", "A possible style variant for the door"),
            ];
        case "AdventCalendarDoorLocked":
            return [
                "day" => $this->field("Int", "The day of the door"),
                "image" => $this->field("ImageData", "The blurred door image")
                    ->setResolver(function(Varien_Object $obj) {
                        $image = $obj->getImage();
                        if(!$image || $image === "/no_selection") {
                            return null;
                        }

                        return new MageQL_Catalog_Model_Attributes_Image_Category($obj->getCategory(), "thumbnail", $image);
                    }),
            ];
        case "Query":
            return [
                "adventCalendar" => $this->field("[AdventCalendarDoor!]!", "Array of the all the doors including locked ones")
                    ->addArgument(
                        "testdate",
                        $this->argument("String", "For simulating a test date. Example: 2020-12-18. You will also need to pass the password")
                    )
                    ->addArgument(
                        "password",
                        $this->argument("String", "The password required for using the testdate param. This must match the one setup in the config")
                    )
                    ->setResolver(
                        /**
                         * @param mixed $unusedSrc
                         */
                        function($unusedSrc, array $args, Context $ctx, ResolveInfo $info): array {
                            $helper = Mage::helper("Crossroads_AdventCalendar");
                            if (!$helper->isEnabled($ctx->getStore())) {
                                return [];
                            }
                            $password = $helper->getPassword($ctx->getStore());
                            $categories = Mage::getModel("catalog/category")->getCollection();
                            $catAttrs = Mage::getSingleton("mageql_catalog/attributes_category");
                            $rootCategoryPath = Mage::getModel("catalog/category")
                                ->load($ctx->getStore()->getRootCategoryId())
                                ->getPath();
                            /** @var Array<string, array{code:string}> force type for psalm */
                            $attributesByArea = $catAttrs->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_ANY);
                            $toSelect = $catAttrs->getUsedAttributes(
                                $attributesByArea,
                                $info->getFieldSelection(3)["category"] ?? []
                            );

                            foreach($toSelect as $col) {
                                $categories->addAttributeToSelect($col);
                                $categories->addAttributeToSelect("entity_id");
                                $categories->addAttributeToSelect("image");
                            }

                            // Filter by parent category to avoid getting the wrong data
                            $categories->addAttributeToFilter("path", [ "like" => sprintf("%s/%%", $rootCategoryPath) ]);
                            /// filtrera och sortera
                            $categories->addAttributeToFilter("is_active", "1");
                            //$categories->addAttributeToSort("cp.position", "asc");
                            $categories->addAttributeToSort("name", "asc");

                            $tzName = $ctx->getStore()->getConfig('general/locale/timezone');

                            assert((bool)$tzName);

                            $tz = new DateTimeZone($tzName);
                            $now = new DateTime("now", $tz);

                            if (array_key_exists("testdate", $args)) {
                                if (!array_key_exists("password", $args) || $args["password"] !== $password) {
                                    throw new Error("Must pass a matching password when simulating a testdate");
                                }

                                $now = new DateTime($args["testdate"], $tz);
                            }

                            $allIds = $helper->getDoorIds($ctx->getStore());
                            $openIds = $helper->getOpenIds($ctx->getStore(), $now);

                            // Filter closed doors and replace with AdventCalendarDoorLocked
                            $allDoors = $categories->getItems();
                            $mergedDoors = array_map(function(Mage_Catalog_Model_Category $item) use($openIds) {
                                if (!in_array($item->getId(), $openIds)) {
                                    // AdventCalendarDoorLocked
                                    return new Varien_Object([
                                        "id" => $item->getId(),
                                        "category" => $item,
                                        "image" => $item->getImage(),
                                        "open" => false,
                                    ]);
                                }
                                return new Varien_Object([
                                    "id" => $item->getId(),
                                    "category" => $item,
                                    "open" => true,
                                ]);
                            }, $allDoors);

                            // Trim unused categories
                            foreach ($mergedDoors as $key => $door) {
                                if (!in_array($door->getId(), $allIds)) {
                                    unset($mergedDoors[$key]);
                                }
                            }

                            // Sort doors by their JSON order
                            $sortedItems = [];

                            foreach($mergedDoors as $door) {
                                $day = $helper->getConfigDayById($ctx->getStore(), (int)$door->getId());
                                $variant = isset($day["variant"]) ? $day["variant"] : null;

                                // Add day number to door
                                $door->setDay($day["day"]);
                                $door->setToday((int)$now->format("d") === $day["day"] && $now->format("m") === "12");

                                if ($variant !== null) {
                                    $door->setVariant($variant);
                                }
                            }

                            foreach($allIds as $i => $value) {
                                foreach($mergedDoors as $door) {
                                    if (intval($door->getId()) === $value) {
                                        $sortedItems[$i] = $door;
                                        break;
                                    }
                                }
                            }


                            return $sortedItems;
                    }),
                    "adventCalendarStartDate" => $this->field("String", "For simulating a test date. Example: 2020-12-18. You will also need to pass the password")
                        ->setResolver(function($unusedSrc, array $unusedArgs, Context $ctx) {
                            $helper = Mage::helper("Crossroads_AdventCalendar");
                            if (!$helper->isEnabled($ctx->getStore())) {
                                return null;
                            }
                            return $helper->getStartDate($ctx->getStore());
                        }),
                    "adventCalendarEndDate" => $this->field("String", "For simulating a test date. Example: 2020-12-18. You will also need to pass the password")
                        ->setResolver(function($unusedSrc, array $unusedArgs, Context $ctx) {
                            $helper = Mage::helper("Crossroads_AdventCalendar");
                            if (!$helper->isEnabled($ctx->getStore())) {
                                return null;
                            }
                            return $helper->getEndDate($ctx->getStore());
                        }),
                ];
    }

    return [];
  }
}
