<?php

class Crossroads_AdventCalendar_Helper_Data extends Mage_Core_Helper_Abstract
{
    const CONFIG_DATA = "AdventCalendar/general/data";
    const CONFIG_DAYS_BACK = "AdventCalendar/settings/days_back";
    const CONFIG_END_DATE = "AdventCalendar/settings/end_date";
    const CONFIG_IS_ENABLED = "AdventCalendar/general/activated";
    const CONFIG_PASSWORD = "AdventCalendar/settings/password";
    const CONFIG_START_DATE = "AdventCalendar/settings/start_date";
    const CONFIG_TIME = "AdventCalendar/settings/time";

    public function isEnabled(Mage_Core_Model_Store $store): bool {
       return (bool)$store->getConfig(self::CONFIG_IS_ENABLED);
    }

    public function getDaysBack(Mage_Core_Model_Store $store): int {
        return max(0, (int)$store->getConfig(self::CONFIG_DAYS_BACK));
    }

    public function getStartDate(Mage_Core_Model_Store $store): string {
       return (string)$store->getConfig(self::CONFIG_START_DATE);
    }

    public function getEndDate(Mage_Core_Model_Store $store): string {
       return (string)$store->getConfig(self::CONFIG_END_DATE);
    }

    public function getSwitchTime(Mage_Core_Model_Store $store): string {
       return (string)$store->getConfig(self::CONFIG_TIME);
    }

    public function getPassword(Mage_Core_Model_Store $store): string {
       return (string)$store->getConfig(self::CONFIG_PASSWORD);
    }

    /**
     * @return array<string, array{category_id: string}>
     */
    public function getConfigData(Mage_Core_Model_Store $store): array {
        $config = (string) $store->getConfig(self::CONFIG_DATA);
        $data = json_decode($config, true);

        if ($data === null) {
            throw new Exception(sprintf("%s: Failed to read JSON: %s", __METHOD__, json_last_error_msg()));
        }

        return $data;
    }

    /**
     * @return Array<int>
     */
    public function getDoorIds(Mage_Core_Model_Store $store) {
        $doorData = $this->getConfigData($store);

        return array_map(function($door) {
            return (int)$door["category_id"];
        }, $doorData);
    }

    /**
     * @return array<string|int>
     */
    public function getConfigDayById(Mage_Core_Model_Store $store, int $id): array {
        $config = $this->getConfigData($store);

        foreach($config as $key => $door) {
            if ((int)$door["category_id"] === $id) {
                $day = (int)substr($key, -2);
                return array_merge($door, ["day" => $day]);
            }
        }

        throw new Error("Couldn't find category_id $id in json config");
    }

    /**
     * @return Array<int>
     */
    public function getOpenIds(Mage_Core_Model_Store $store, DateTime $now): array {
        $tz = $now->getTimezone();

        assert($tz !== false);

        $switchTime = $this->getSwitchTime($store) ?: "00:00:00";
        $startDateStr = ($this->getStartDate($store) ?: $now->format("Y")."-12-01")." ".$switchTime;
        $startDate = new DateTime($startDateStr, $tz);
        $endDateStr = ($this->getEndDate($store) ?: ((int)$now->format("Y") + 1)."-01-15")." ".$switchTime;
        $endDate = new DateTime($endDateStr, $tz);

        // Return no categories if before start date or past end date
        if ($now < $startDate || $now > $endDate) {
            return [-1];
        }

        $categoryIds = [];
        $doorData = $this->getConfigData($store);
        $timestampNow = strtotime($now->format("Y-m-d H:i:s"));

        $daysBackStr = $this->getDaysBack($store) ?: 30;
        $daysBack = new DateTime(date("Y-m-d H:i:s", strtotime("-{$daysBackStr} days", $timestampNow)), $tz);
        if ($daysBack < $startDate) {
            $daysBack = $startDate;
        }

        foreach ($doorData as $date => $door) {
            $doorDate = new DateTime(date("Y-m-d",strtotime($date))." ".$switchTime, $tz);

            if ($doorDate <= $now && $doorDate >= $daysBack) {
                array_push($categoryIds, (int)$door["category_id"]);
            }
        }

        return $categoryIds;
    }
}
