<?php

declare(strict_types=1);

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class AdventCalendarTest extends TestCase {
    use MatchesSnapshots;

    const PASSWORD = "foo";

    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        $this->createCategories();
        $this->setModuleConfig();
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
        MagentoManager::reset();
        MagentoManager::initAdmin();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $setupModel->deleteConfigData("AdventCalendar/general/activated", "websites");
        $setupModel->deleteConfigData("AdventCalendar/general/data",  "websites");
        $setupModel->deleteConfigData("AdventCalendar/settings/password", "websites");

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        $this->tearDown();

        throw $e;
    }

    protected function categoryFixture(string $name): Mage_Catalog_Model_Category {
        $store = Mage::app()->getStore();
        $category = MagentoManager::loadCategoryByAttribute("varchar", "name", $name);
        $rootId = Mage::app()->getStore(MagentoManager::TESTING_STORE)->getRootCategoryId();
        $root = Mage::getModel("catalog/category")->load($rootId);

        $url = "/" . str_replace(" ", "-", $name);

        $category->addData([
            "url_key" => $url,
            "name" => $name,
            "atttribute_set_id" => $category->getDefaultAttributeSetId(),
            "description" => "advent calendar category",
            "display_mode" => "PRODUCTS",
            "is_active" => 1,
            "is_anchor" => 0,
            "path" => $category->getPath() ?: $root->getPath(),
            "parent_id" => $root->getId(),
            "store_id" => $store->getId(),

        ]);

        $category->save();

        return $category;
    }

    protected function createCategories(): void {
        for ($i = 1; $i <= 24; $i++)  {
            $this->categoryFixture("category $i");
        }
    }

    protected function setModuleConfig(): void  {
        $days = [];

        for ($i = 1; $i <= 24; $i ++) {
            $category = MagentoManager::loadCategoryByAttribute("varchar", "name", "category $i");
            $prefixedI = $i < 10 ? "0$i" : $i;

            if ($i === 15) {
                $days["12$prefixedI"] = [ "category_id" => $category->getId(), "variant" => "overlay" ];
            } else {
                $days["12$prefixedI"] = [ "category_id" => $category->getId() ];
            }
        }

        $data = json_encode($days);

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $website = (int)$store->getWebsiteId();
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $setupModel->setConfigData("AdventCalendar/general/activated", 1, "websites", $website);
        $setupModel->setConfigData("AdventCalendar/general/data", $data, "websites", $website);
        $setupModel->setConfigData("AdventCalendar/settings/password", self::PASSWORD, "websites", $website);
    }

    public function test2020_12_01(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", [ "Content-Type" => "application/graphql", ], 'query {
            adventCalendar (testdate: "2020-12-1", password: "' . self::PASSWORD . '") {
                ... on AdventCalendarDoorToday {
                    variant
                    category {
                        name
                        url
                        products {
                            items {
                                name
                            }
                        }
                    }
                }
                ... on AdventCalendarDoorOpen {
                    variant
                    category {
                        name
                        url
                    }
                }
                ... on AdventCalendarDoorLocked {
                    image { src (width: 100) }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function test2020_12_24(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", [ "Content-Type" => "application/graphql", ], 'query {
            adventCalendar (testdate: "2020-12-24", password: "' . self::PASSWORD . '") {
                ... on AdventCalendarDoorToday {
                    variant
                    category {
                        name
                        url
                        products {
                            items {
                                name
                            }
                        }
                    }
                }
                ... on AdventCalendarDoorOpen {
                    variant
                    category {
                        name
                        url
                    }
                }
                ... on AdventCalendarDoorLocked {
                    image { src (width: 100) }
                    day
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function test2020_11_25(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", [ "Content-Type" => "application/graphql", ], 'query {
            adventCalendar (testdate: "2020-11-25", password: "' . self::PASSWORD . '") {
                ... on AdventCalendarDoorToday {
                    variant
                    category {
                        name
                        url
                        products {
                            items {
                                name
                            }
                        }
                    }
                }
                ... on AdventCalendarDoorOpen {
                    variant
                    category {
                        name
                        url
                    }
                }
                ... on AdventCalendarDoorLocked {
                    image { src (width: 100) }
                    day
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function test2021_01_01(): void {
        $res = MagentoManager::runRequest(new Request("POST /graphql", [ "Content-Type" => "application/graphql", ], 'query {
            adventCalendar (testdate: "2021-01-01", password: "' . self::PASSWORD . '") {
                ... on AdventCalendarDoorToday {
                    variant
                    category {
                        name
                        url
                        products {
                            items {
                                name
                            }
                        }
                    }
                }
                ... on AdventCalendarDoorOpen {
                    variant
                    category {
                        name
                        url
                    }
                }
                ... on AdventCalendarDoorLocked {
                    image { src (width: 100) }
                    day
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
