<?php
/*
 * Specification:
 * https://groot.crossroads.se/awardit/products-api/-/blob/master/swagger.yml
 *
 * This model was copied from local Magento10 module: Crossroads/Integration
 *
 * Java logfiles on awardit-staging: /opt/tomcat/logs/catalina.out
 *
 */

use GuzzleHttp\Client;
use GuzzleHttp\Exception\{
    ClientException,
    ConnectException
};

class Awardit_Upsert_Model_Api extends Mage_Core_Model_Abstract
{
    const PAGE_LIMIT_DEFAULT = 20;
    const PAGE_LIMIT_MAX = 200;

    const AUTHORIZATION_COMPANY_ID = 1;
    const AUTO_ENABLE = true; // TODO: Move to Magento config?

    const IMAGE_MIMETYPES = ['image/jpeg', 'image/jpg', 'image/gif', 'image/png'];

    protected $localStore = null;
    protected $isCreate = false;
    protected $isProductUpdated = false;
    protected $mediaGalleryUpdated = false;
    protected $httpClient = null;
    protected static $defaultVisibility = Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH;
    protected static $defaultType = Mage_Catalog_Model_Product_Type::TYPE_SIMPLE;
    protected static $backorderDefault = Mage_CatalogInventory_Model_Stock::BACKORDERS_YES_NONOTIFY;
    protected static $defaultCustomOptionType = "drop_down"; // field / area / file / drop_down / radio / checkbox / multiple / date / date_time / time
    protected static $allowedTypes = [
        "physical" => Mage_Catalog_Model_Product_Type::TYPE_SIMPLE,
        "virtual" => Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL
    ];
    protected static $allowedStatuses = [
        "enabled" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
        "disabled" => Mage_Catalog_Model_Product_Status::STATUS_DISABLED
    ];

    public function setLocalStore(Mage_Core_Model_Store $store): void
    {
        $this->localStore = $store;
    }

    public function getLocalStore()
    {
        return $this->localStore;
    }

    public function createProductAction($params)
    {
        $this->isCreate = true;
        $store = $this->getLocalStore();
        $storeId = $store->getStoreId();
        $isAwarditProduct = true; // New products is per definition an Awardit Product
        $parsedParams = $this->parseParameters($params, $store, $isAwarditProduct);

        // Check if we are supposed to update this product
        $indexSetting = Mage::getStoreConfig('integration/upsert/index_setting', $storeId) ?: 0;
        if (
                $indexSetting == Awardit_Upsert_Model_Source_Upsert_Index::UPSERT_INDEX_PARTNER_OWNED
                    &&
                $parsedParams["partnerOwned"] === Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO
        ) {
            Mage::throwException("Only partner owned products can be created!");
        } elseif ($indexSetting == Awardit_Upsert_Model_Source_Upsert_Index::UPSERT_INDEX_NONE) {
            Mage::throwException("Product creation is disabled!");
        }

        // Check sku parameter
        if (empty($params["sku"])) {
            throw new Awardit_Upsert_InputException("Missing SKU");
        }
        $sku = $params["sku"];

        // Try to get product id using new sku, to make sure it doesn't already exist
        $productId = Mage::getModel("catalog/product")->getIdBySku($sku);
        if (!empty($productId)) {
            Mage::throwException("Product already exists");
        }

        $this->log("Creating product [{$sku}]");

        // Setup new product using defaults and provided parameters
        $newProduct = Mage::getModel("catalog/product");
        $newProduct->setStoreId(0);
        $newProduct->setSku($sku);
        $newProduct->setAttributeSetId(Mage::getModel("catalog/product")->getDefaultAttributeSetId());
        $newProduct->setTypeId($parsedParams["typeId"]);
        $newProduct->setName($parsedParams["name"]);
        $newProduct->setDescription($parsedParams["longDescription"]);
        $newProduct->setShortDescription($parsedParams["shortDescription"]);
        $newProduct->setVisibility(self::$defaultVisibility); // catalog, search
        $newProduct->setAwarditIsFeatured(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $newProduct->setAwarditPartnerOwned(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $newProduct->setAwarditTerms("");
        $newProduct->setAwarditCheckTerms(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $newProduct->setAwarditAwaitingPrice(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $newProduct->setAwarditTargetId(0);
        $newProduct->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
        $newProduct->setManufacturer($parsedParams["manufacturerId"]);
        $newProduct->setPrice($parsedParams["splitPaymentPrice"]);
        $newProduct->setInvoicePrice($parsedParams["invoicePrice"]);
        $newProduct->setMsrp($parsedParams["msrp"]);
        $newProduct->setTaxClassId($parsedParams["taxClassId"]);
        $newProduct->setWeight(1);
        $newProduct->setMediaGallery ([ "images" => [], "values" => [] ]);
        $this->updateGlobalProduct($newProduct, $parsedParams, $store);
        $this->updateBySkuMapper($newProduct, Mage_Core_Model_App::ADMIN_STORE_ID);

        // If needed, save product
        if ($this->isProductUpdated) {
            $newProduct->save();
            $this->isProductUpdated = false;
        }

        // Reload new product in local store context
        $localProduct = Mage::getModel("catalog/product");
        $newProductId = $localProduct->getIdBySku($sku);
        if (empty($newProductId)) {
            Mage::throwException("Product was not saved correctly");
        }
        $localProduct->setStoreId($storeId)->load($newProductId);
        if (!$localProduct->getId()) {
            Mage::throwException("Unable to load newly created product");
        }

        $this->updateBySkuMapper($localProduct, $storeId);
        $this->updateLocalProduct($localProduct, $parsedParams);

        // Only set these attributes if new product or not in batch mode
        if ($this->isCreate || !$parsedParams["isBatch"]) {

            $product = Mage::getModel("catalog/product")->setStoreId(0)->load($newProductId);

            // Only add custom options to Awardit products
            if ($isAwarditProduct && array_key_exists("options", $parsedParams) && is_array($parsedParams["options"])) {
                if ($this->setCustomOptions($product, $parsedParams["options"])) {
                    $this->isProductUpdated = true;
                }
            }
        }

        return [
            "sku" => $sku,
            "createdAt" => date("Y-m-d\TH:i:s\Z", strtotime($localProduct->getCreatedAt())),
            "updatedAt" => date("Y-m-d\TH:i:s\Z", strtotime($localProduct->getUpdatedAt())),
        ];
    }

    public function updateProductAction($params, $sku)
    {
        $this->isCreate = false;

        // Check sku parameter
        if (empty($sku)) {
            throw new Awardit_Upsert_InputException("Missing SKU");
        }

        // Try to get product id using sku
        $productId = Mage::getModel('catalog/product')->getIdBySku($sku);
        if (empty($productId)) {
            // Mage::throwException("Unable to find product");
            return $this->createProductAction($params);
        }

        // Try to load product in default context
        $product = Mage::getModel('catalog/product')->setStoreId(0)->load($productId);
        if (!$product->getId()) {
            throw new Awardit_Upsert_InputException("Unable to load product");
        }

        // Only update physical or digital products
        if (!in_array($product->getTypeId(), [ Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL ])) {
            throw new Awardit_Upsert_InputException("Unable to update this product type ({$product->getTypeId()}) via Upsert");
        }

        // Setup and parse input parameters
        $store = $this->getLocalStore();
        $storeId = $store->getStoreId();
        $isAwarditProduct = $this->isAwarditProduct($product);
        $parsedParams = $this->parseParameters($params, $store, $isAwarditProduct);

        // Check if we are supposed to update this product
        $indexSetting = Mage::getStoreConfig('integration/upsert/index_setting', $storeId) ?: 0;
        if (
                $indexSetting == Awardit_Upsert_Model_Source_Upsert_Index::UPSERT_INDEX_PARTNER_OWNED
                    &&
                $product->getAwarditPartnerOwned() === Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO
        ) {
            throw new Awardit_Upsert_InputException("Only partner owned products can be updated!");
        } elseif ($indexSetting == Awardit_Upsert_Model_Source_Upsert_Index::UPSERT_INDEX_NONE) {
            throw new Awardit_Upsert_InputException("Product updates is disabled!");
        }
        
        // Update product in default context
        $this->updateGlobalProduct($product, $parsedParams, $store);
        $this->log("Updating product [{$product->getSku()}]");

        // If needed, save product
        if ($this->isProductUpdated) {
            $product->save();
        }

        if ($this->mediaGalleryUpdated) {
            $this->resetImageGalleryToUseDefault($productId);
            $this->mediaGalleryUpdated = false;
        }

        // Try to load product in local context
        $localProduct = Mage::getModel('catalog/product')->setStoreFilter($store)->setStoreId($storeId)->load($productId);
        if (!$localProduct->getId()) {
            throw new Awardit_Upsert_InputException("Unable to load product");
        }

        // Update product in local context
        $this->updateLocalProduct($localProduct, $parsedParams);

        // Only set these attributes if new product or not in batch mode
        if ($this->isCreate || !$parsedParams["isBatch"]) {

            // Only add custom options to Awardit products
            if ($isAwarditProduct && array_key_exists("options", $parsedParams) && is_array($parsedParams["options"])) {
                if ($this->setCustomOptions($product, $parsedParams["options"])) {
                    $this->isProductUpdated = true;
                }
            }
        }


        return [
            "createdAt" => date("Y-m-d\TH:i:s\Z", strtotime($localProduct->getCreatedAt())),
            "updatedAt" => date("Y-m-d\TH:i:s\Z", strtotime($localProduct->getUpdatedAt())),
        ];
    }

    /**
     * @param int $pId
     */
    public function resetImageGalleryToUseDefault($pId): void
    {
        if ($pId > 0) {
            $sqlQuery = "
                DELETE catalog_product_entity_varchar.*
                FROM eav_attribute
                JOIN catalog_product_entity_varchar ON catalog_product_entity_varchar.attribute_id = eav_attribute.attribute_id
                WHERE eav_attribute.attribute_code IN ('image', 'small_image', 'thumbnail') AND eav_attribute.backend_model IS NULL AND store_id > 0 AND catalog_product_entity_varchar.entity_id = :pId
            ";
            Mage::getSingleton("core/resource")->getConnection("core_write")->query($sqlQuery, [ "pId" => $pId ]);
        }
    }

    /**
     * @param int $pId
     * @param int $sId
     * @return void
     */
    public function setProductInStoreToUseNoImages($pId, $sId): void
    {
        if ($pId > 0) {
            $sqlQuery = "
                UPDATE catalog_product_entity_varchar
                JOIN eav_attribute ON eav_attribute.attribute_id = catalog_product_entity_varchar.attribute_id
                SET catalog_product_entity_varchar.`value` = 'no_selection'
                WHERE eav_attribute.attribute_code IN ('image', 'small_image', 'thumbnail') AND eav_attribute.backend_model IS NULL AND store_id = :sId AND catalog_product_entity_varchar.entity_id = :pId
            ";
            Mage::getSingleton("core/resource")->getConnection("core_write")->query($sqlQuery, [ "sId" => $sId, "pId" => $pId ]);
        }
    }

    public function listProductsAction($params, $sku = null): array
    {
        // Setup and parse input parameters
        $store = $this->getLocalStore();
        $storeId = intval($store->getId());
        $parsedParams = $this->parseGetParameters($params, $store);
        $pos = ($parsedParams["page"] - 1) * $parsedParams["limit"];

        if (empty($sku)) {
            $sqlQuery = "
                SELECT
                    aup.*,
                    ppp.price AS points,
                    ppp.min_price AS minPoints,
                    ppp.max_price AS maxPoints
                FROM awardit_upsert_product aup
                JOIN catalog_product_entity cpe ON cpe.entity_id = aup.product_id
                LEFT JOIN points_product_price ppp ON ppp.product_id = aup.product_id AND ppp.store_id = aup.store_id AND ppp.customer_group_id = 0
                WHERE aup.store_id = ? AND aup.product_data IS NOT NULL AND (aup.purchase_price > 0 OR cpe.sku LIKE 'awd_%')
                ORDER BY aup.product_id LIMIT {$pos}, {$parsedParams["limit"]}
            ";
            $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $storeId);
        } else {
            $productId = Mage::getModel('catalog/product')->getIdBySku($sku);
            if (empty($productId)) {
                throw new Awardit_Upsert_InputException("Unable to find product");
            }
            $sqlQuery = "
                SELECT
                    aup.*,
                    ppp.price AS points,
                    ppp.min_price AS minPoints,
                    ppp.max_price AS maxPoints
                FROM awardit_upsert_product aup
                JOIN catalog_product_entity cpe ON cpe.entity_id = aup.product_id
                LEFT JOIN points_product_price ppp ON ppp.product_id = aup.product_id AND ppp.store_id = aup.store_id AND ppp.customer_group_id = 0
                WHERE aup.store_id = :sId AND aup.product_id = :pId AND aup.product_data IS NOT NULL AND (aup.purchase_price > 0 OR cpe.sku LIKE 'awd_%')
            ";
            $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, [ "sId" => $storeId, "pId" => $productId ]);
        }

        $output = [];
        foreach ($data as $index => $row) {
            $output[$index] = json_decode($row["product_data"], true);

            // Replace price by purchase price from Visma
            $output[$index]["price"] = $row["purchase_price"];

            // Copy points from pointsCore
            $output[$index]["points"] = $row["points"];
            $output[$index]["minPoints"] = $row["minPoints"];
            $output[$index]["maxPoints"] = $row["maxPoints"];

            // Recalculate isInStock, we have strange problems with this value from serializer
            $output[$index]["isInStock"] = $output[$index]["stockManage"] ? $output[$index]["stockQty"] > 0 : true;

        }

        return $output;

    }

    public function parseGetParameters($params, Mage_Core_Model_Store $store): array
    {
        $parsedParams = [];

        // Check 'limit' parameter
        if (array_key_exists("limit", $params)) {
            if (empty($params["limit"])) {
                $parsedParams["limit"] = self::PAGE_LIMIT_DEFAULT;
            } elseif(!is_numeric($params["limit"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'limit'");
            } else {
                $parsedParams["limit"] = intval($params["limit"]);
            }
            if ($parsedParams["limit"] < 1 || $parsedParams["limit"] > self::PAGE_LIMIT_MAX) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'limit'");
            }
        } else {
            $parsedParams["limit"] = self::PAGE_LIMIT_DEFAULT;
        }

        // Check 'page' parameter
        if (array_key_exists("page", $params)) {
            if (empty($params["page"])) {
                $parsedParams["page"] = 1;
            } elseif(!is_numeric($params["page"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'page'");
            } else {
                $parsedParams["page"] = intval($params["page"]);
            }
            if ($parsedParams["page"] < 1 || $parsedParams["page"] > 9999) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'page'");
            }
        } else {
            $parsedParams["page"] = 1;
        }

        return $parsedParams;
    }

    public function parseParameters($params, Mage_Core_Model_Store $store, bool $isAwarditProduct): array
    {
        $parsedParams = [];

        // Check 'batch' parameter
        $parsedParams["isBatch"] = empty($params["batch"]) ? false : true;

        // Check 'type' parameter
        if (array_key_exists("type", $params)) {
            if (empty($params["type"])) {
                $parsedParams["typeId"] = self::$defaultType;
                // throw new Awardit_Upsert_InputException("Wrong value for parameter: 'type'");
            } elseif(!array_key_exists($params["type"], self::$allowedTypes)) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'type'");
            } else {
                $parsedParams["typeId"] = self::$allowedTypes[$params["type"]];
            }
        } elseif($this->isCreate) {
            throw new Awardit_Upsert_InputException("Missing parameter: 'type'");
        }

        // Check 'manufacturer' parameter
        if (array_key_exists("manufacturer", $params)) {
            if (!empty($params["manufacturer"])) {
                $manufacturerId = Mage::Helper("integration")->getManufacturerId($params["manufacturer"]);
            } else {
                $manufacturerId = Mage::Helper("integration")->getManufacturerId("");
            }
            if (empty($manufacturerId)) {
                $manufacturerId = Mage::Helper("integration")->createManufacturer($params["manufacturer"]);
                if (empty($manufacturerId)) {
                    throw new Awardit_Upsert_InputException("Unable to create manufacturer");
                }
            } else {
                $parsedParams["manufacturerId"] = $manufacturerId;
            }
        } elseif($this->isCreate) {
            throw new Awardit_Upsert_InputException("Missing parameter: 'manufacturer'");
        }

        // Check 'status' parameter
        if (array_key_exists("status", $params)) {
            if (empty($params["status"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'status'");
            } elseif (!array_key_exists($params["status"], self::$allowedStatuses)) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'status'");
            }
            $parsedParams["status"] = self::$allowedStatuses[$params["status"]];
        } else {
            throw new Awardit_Upsert_InputException("Missing parameter: 'status'");
        }

        // Check 'stock' parameter
        if (array_key_exists("stock", $params)) {
            if (is_array($params["stock"])) {
                if (!array_key_exists("qty", $params["stock"])) {
                    throw new Awardit_Upsert_InputException("Missing parameter: 'qty' in 'stock'");
                } elseif(!is_numeric($params["stock"]["qty"])) {
                    throw new Awardit_Upsert_InputException("Wrong value for parameter: 'qty' in 'stock'");
                }
                if (!array_key_exists("backorders", $params["stock"])) {
                    throw new Awardit_Upsert_InputException("Missing parameter: 'backorders' in 'stock'");
                }
                $parsedParams["stock"] = [
                    "qty" => intval($params["stock"]["qty"]),
                    "backorders" => !empty($params["stock"]["backorders"]) ? true : false
                ];
            } elseif ($params["stock"] !== null) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'stock'");
            } else {
                $parsedParams["stock"] = null; // Do not handle stock, product is always available
            }
        } elseif ($this->isCreate) {
            throw new Awardit_Upsert_InputException("Missing parameter: 'stock'");
        }

        // Check 'name' parameter
        if (array_key_exists("name", $params)) {
            if (empty($params["name"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'name'");
            } elseif (strlen($params["name"]) > 255) {
                throw new Awardit_Upsert_InputException("String too long for parameter: 'name'");
            }
            $parsedParams["name"] = $params["name"];
        } elseif($this->isCreate) {
            throw new Awardit_Upsert_InputException("Missing parameter: 'name'");
        }

        // Check 'shortDescription' parameter
        if (array_key_exists("shortDescription", $params)) {
            if (empty($params["shortDescription"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'shortDescription'");
            } elseif (strlen($params["shortDescription"]) > 65535) {
                throw new Awardit_Upsert_InputException("String too long for parameter: 'shortDescription'");
            }
            $parsedParams["shortDescription"] = $params["shortDescription"];
        } elseif($this->isCreate) {
            throw new Awardit_Upsert_InputException("Missing parameter: 'shortDescription'");
        }

        // Check 'longDescription' parameter
        if (array_key_exists("longDescription", $params)) {
            if (empty($params["longDescription"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'longDescription'");
            } elseif (strlen($params["longDescription"]) > 65535) {
                throw new Awardit_Upsert_InputException("String too long for parameter: 'longDescription'");
            }
            $parsedParams["longDescription"] = $params["longDescription"];
        } elseif($this->isCreate) {
            throw new Awardit_Upsert_InputException("Missing parameter: 'longDescription'");
        }

        // Check 'special' parameter
        if (array_key_exists("special", $params)) {
            $parsedParams["isFeatured"] = intval($params["special"]) === 1 ? Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES : Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO;
        } elseif($this->isCreate) {
            $parsedParams["isFeatured"] = Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO;
        }

        // Check 'validFrom' parameter (check for 'validfrom' as fallback)
        if (array_key_exists("validFrom", $params)) {
            if (empty($params["validFrom"])) {
                $parsedParams["validFrom"] = "";
            } elseif (preg_match('/[0-9]{4}-[0-9]{2}-[0-9]{2}/', $params["validFrom"])) {
                $parsedParams["validFrom"] = date("Y-m-d", strtotime($params["validFrom"]));
            } else {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'validFrom'");
            }
        } elseif (array_key_exists("validfrom", $params)) {
            if (empty($params["validfrom"])) {
                $parsedParams["validfrom"] = "";
            } elseif (preg_match('/[0-9]{4}-[0-9]{2}-[0-9]{2}/', $params["validfrom"])) {
                $parsedParams["validFrom"] = date("Y-m-d", strtotime($params["validfrom"]));
            } else {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'validFrom'");
            }
        }

        // Check 'validTo' parameter (check for 'validto' as fallback)
        if (array_key_exists("validTo", $params)) {
            if (empty($params["validTo"])) {
                $parsedParams["validTo"] = "";
            } elseif (preg_match('/^[0-9]{4}-[0-9]{2}-[0-9]{2}$/', $params["validTo"])) {
                $parsedParams["validTo"] = date("Y-m-d", strtotime($params["validTo"]));
            } else {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'validTo'");
            }
        } elseif (array_key_exists("validto", $params)) {
            if (empty($params["validto"])) {
                $parsedParams["validto"] = "";
            } elseif (preg_match('/^[0-9]{4}-[0-9]{2}-[0-9]{2}$/', $params["validto"])) {
                $parsedParams["validTo"] = date("Y-m-d", strtotime($params["validto"]));
            } else {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'validTo'");
            }
        }

        // Check 'invoiceName' parameter
        if (array_key_exists("invoiceName", $params)) {
            if (empty($params["invoiceName"])) {
                $parsedParams["invoiceName"] = "";
            } elseif (strlen($params["invoiceName"]) > 255) {
                throw new Awardit_Upsert_InputException("String too long for parameter: 'invoiceName'");
            } else {
                $parsedParams["invoiceName"] = $params["invoiceName"];
            }
        }

        // Check 'terms' and 'checkok' parameters
        if (array_key_exists("terms", $params)) {
            if (strlen($params["terms"]) > 32767) {
                throw new Awardit_Upsert_InputException("String too long for parameter: 'terms'");
            } elseif (empty($params["terms"])) {
                $parsedParams["terms"] = "";
                $parsedParams["checkTerms"] = Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO;
            } else {
                $parsedParams["terms"] = $params["terms"];
                $parsedParams["checkTerms"] = !empty($params["checkok"]) ? Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES : Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO;
            }
        } elseif($this->isCreate) {
            $parsedParams["terms"] = "";
            $parsedParams["checkTerms"] = Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO;
        }

        // Check 'partnerOwned' parameter
        if (array_key_exists("partnerOwned", $params)) {
            $parsedParams["partnerOwned"] = !empty($params["partnerOwned"]) ? Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES : Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO;
        } elseif($this->isCreate) {
            $parsedParams["partnerOwned"] = Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO;
        }

        // Check 'msrp' parameter
        if (array_key_exists("msrp", $params)) {
            if (!is_numeric($params["msrp"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'msrp'");
            }
            $msrp = floatval($params["msrp"]);
            if ($msrp < 0.0) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'msrp'");
            }
            $parsedParams["msrp"] = $msrp;
        } elseif($this->isCreate) {
            $parsedParams["msrp"] = 0.0;
        }

        // Check 'invoicePrice' parameter (used in order export to Visma)
        if (array_key_exists("invoicePrice", $params)) {
            if (!is_numeric($params["invoicePrice"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'invoicePrice'");
            }
            $price = floatval($params["invoicePrice"]);
            if ($price > 0) {
                $parsedParams["invoicePrice"] = $price;
            } else {
                if ($parsedParams["status"] === Mage_Catalog_Model_Product_Status::STATUS_DISABLED) {
                    // Alow zero price if product is about to be disabled
                    $parsedParams["invoicePrice"] = $price;
                } else {
                    throw new Awardit_Upsert_InputException("Wrong value for parameter: 'invoicePrice'");
                }
            }

        // Use old parameter 'price' as fallback if 'invoicePrice' is missing.
        } elseif (array_key_exists("price", $params)) {
            if (!is_numeric($params["price"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'price'");
            }
            $price = floatval($params["price"]);
            if ($price > 0) {
                $parsedParams["invoicePrice"] = $price;
            } else {
                if ($parsedParams["status"] === Mage_Catalog_Model_Product_Status::STATUS_DISABLED) {
                    // Alow zero price if product is about to be disabled
                    $parsedParams["invoicePrice"] = $price;
                } else {
                    throw new Awardit_Upsert_InputException("Wrong value for parameter: 'price' (also: should use parameter 'invoicePrice' instead!)");
                }
            }
        } else {
            throw new Awardit_Upsert_InputException("Missing parameter: 'invoicePrice'");
        }

        // Check 'splitPaymentPrice' parameter (used as 'price')
        // Use invoice price as fallback
        if (array_key_exists("splitPaymentPrice", $params)) {
            if (is_numeric($params["splitPaymentPrice"])) {
                $price = floatval($params["splitPaymentPrice"]);
                if ($price > 0) {
                    $parsedParams["splitPaymentPrice"] = $price;
                } else {
                    $parsedParams["splitPaymentPrice"] = $parsedParams["invoicePrice"];
                }
            } else {
                $parsedParams["splitPaymentPrice"] = $parsedParams["invoicePrice"];
            }
        } else {
            $parsedParams["splitPaymentPrice"] = $parsedParams["invoicePrice"];
        }

        // Check 'points' parameter
        if (array_key_exists("points", $params)) {
            if (!is_numeric($params["points"])) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'points'");
            }
            $points = intval($params["points"]);
            if ($points < 1) {
                $points = null; // Remove points from product
            }
            $parsedParams["points"] = $points;
        } elseif($this->isCreate) {
            throw new Awardit_Upsert_InputException("Missing parameter: 'points'");
        }

        // Check 'minPoints' parameter
        if (array_key_exists("minPoints", $params)) {
            if (!(is_numeric($params["minPoints"]) || $params["minPoints"] === null)) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'minPoints'");
            }
            $minPoints = intval($params["minPoints"]);
            if ($minPoints > 0) {
                $parsedParams["minPoints"] = $minPoints;
            }
        }

        // Check 'maxPoints' parameter
        if (array_key_exists("maxPoints", $params)) {
            if (!(is_numeric($params["maxPoints"]) || $params["maxPoints"] === null)) {
                throw new Awardit_Upsert_InputException("Wrong value for parameter: 'maxPoints'");
            }
            $maxPoints = intval($params["maxPoints"]);
            if ($maxPoints > 0) {
                $parsedParams["maxPoints"] = $maxPoints;
            }
        }

        // Check 'vat' parameter
        if($this->isCreate) {
            if (array_key_exists("vat", $params)) {
                if (!is_numeric($params["vat"])) {
                    throw new Awardit_Upsert_InputException("Wrong value for parameter: 'vat'");
                }
                $vat = floatval($params["vat"]);
                if ($vat < 0.0 || $vat >= 100.0) { // 100% VAT? yeah, right...
                    throw new Awardit_Upsert_InputException("Wrong value for parameter: 'vat'");
                }
                $parsedParams["vat"] = $vat;
                $parsedParams["taxClassId"] = $this->getTaxClassId($store, $vat);
            } else {
                throw new Awardit_Upsert_InputException("Missing parameter: 'vat'");
            }
        }

        // Check 'categories' parameter
        $parsedParams["categories"] = [];
        if (!empty($params["categories"])) {
            if (is_array($params["categories"])) {
                $parsedParams["categories"] = $params["categories"];
            } else {
                throw new Awardit_Upsert_InputException("Wrong type for parameter: 'categories'");
            }
        }

        // Check 'media' parameter
        $parsedParams["media"] = [];
        if (!empty($params["media"])) {
            if (!is_array($params["media"])) {
                    throw new Awardit_Upsert_InputException("Wrong type for parameter: 'media'");
            }
            foreach ($params["media"] as $mp) {
                if (!array_key_exists("url", $mp)) {
                    throw new Awardit_Upsert_InputException("Missing parameter: 'url' in 'media'");
                } elseif (empty($mp["url"])) {
                    throw new Awardit_Upsert_InputException("Wrong value for parameter: 'url' in 'media'");
                } elseif($mp["url"] === "null") {
                    // It seems that if something goes wrong with image uploading in the Java platform, image url gets the (string) value "null".
                    continue;
                }
                $parsedParams["media"][] = [
                    "url" => $mp["url"],
                    "primary" => !empty($mp["primary"]) ? true : false
                ];
            }
        }

        // Check 'options' parameter
        if ($isAwarditProduct && array_key_exists("options", $params)) {
            if ($params["options"] === null) {
                $parsedParams["options"] = null;
            } elseif (!is_array($params["options"])) {
                throw new Awardit_Upsert_InputException("Wrong type for parameter: 'options'");
            } else {
                // [ {"name":"Golfklubba","required":true,"values":[ {"label":"Driver","value":"Driver"}, {"label":"Järnklubba","value":"Järnklubba"}, {"label":"Putte","value":"Putte"} ]} ]
                $parsedParams["options"] = [];
                foreach ($params["options"] as $optionIndex => $option) {
                    if (!array_key_exists("name", $option)) {
                        throw new Awardit_Upsert_InputException("Missing parameter: 'name' in 'options'");
                    } elseif (empty($option["name"])) {
                        throw new Awardit_Upsert_InputException("Wrong value for parameter: 'name' in 'options'");
                    } elseif (strlen($option["name"]) > 50) {
                        throw new Awardit_Upsert_InputException("String too long for parameter: 'name' in 'options'");
                    }
                    if (!array_key_exists("required", $option)) {
                        throw new Awardit_Upsert_InputException("Missing parameter: 'required' in 'options'");
                    }
                    if (!array_key_exists("values", $option)) {
                        throw new Awardit_Upsert_InputException("Missing parameter: 'values' in 'options'");
                    } elseif (!is_array($option["values"])) {
                        throw new Awardit_Upsert_InputException("Wrong type for parameter: 'values' in 'options'");
                    }
                    $parsedParams["options"][$optionIndex] = [
                        "title" => $option["name"],
                        "type" => self::$defaultCustomOptionType,
                        "is_require" => !empty($option["required"]) ? 1 : 0,
                        "values" => []
                    ];
                    foreach ($option["values"] as $valueIndex => $value) {
                        if (!array_key_exists("label", $value)) {
                            throw new Awardit_Upsert_InputException("Missing parameter: 'label' in 'values' in 'options'");
                        } elseif (empty($value["label"])) {
                            throw new Awardit_Upsert_InputException("Wrong value for parameter: 'label' in 'values' in 'options'");
                        } elseif (strlen($value["label"]) > 50) {
                            throw new Awardit_Upsert_InputException("String too long for parameter: 'label' in 'values' in 'options'");
                        }
                        if (!empty($value["value"]) && strlen($value["value"]) > 50) {
                            throw new Awardit_Upsert_InputException("String too long for parameter: 'value' in 'values' in 'options'");
                        }
                        $parsedParams["options"][$optionIndex]["values"][$valueIndex] = [
                            "title" => $value["label"],
                            "price" => 0,
                            "price_type" => "fixed",
                            "sort_order" => $valueIndex,
                            "sku" => !empty($value["value"]) ? $value["value"] : ""
                        ];
                    }
                }
            }
        }

        // Check 'targetid' parameter
        if (array_key_exists("targetid", $params)) {
            $parsedParams["targetId"] = 0;
            if (!empty($params["targetid"])) {
                $targetId = intval($params["targetid"]);
                if (!empty($targetId)) {
                    $parsedParams["targetId"] = $targetId;
                }
            }
        }

        return $parsedParams;
    }

    public function updateGlobalProduct(Mage_Catalog_Model_Product &$product, $parsedParams, Mage_Core_Model_Store $store): void
    {
        $isAwarditProduct = $this->isAwarditProduct($product);

        // Update categories
        if (array_key_exists("categories", $parsedParams)) {
            $this->setCategories($product, $parsedParams["categories"], $store);
        }

        // If product does not belong to local website, add it
        $websiteId = $store->getWebsiteId();
        $productWebsiteIds = $this->isCreate ? [] : $product->getWebsiteIds();
        if (!in_array($websiteId, $productWebsiteIds)) {
            $productWebsiteIds[] = $websiteId;
            $product->setWebsiteIds($productWebsiteIds);
            $this->isProductUpdated = true;
        }

        // Update stock
        if ($isAwarditProduct && array_key_exists("stock", $parsedParams)) {
            if ($this->isCreate) {
                if ($parsedParams["stock"] === null) {
                    $product->setStockData([
                        "use_config_manage_stock" => 0,
                        "manage_stock" => 0,
                        "use_config_backorders" => 0,
                        "backorders" => Mage_CatalogInventory_Model_Stock::BACKORDERS_NO,
                        "is_in_stock" => 1,
                        "qty" => 999999
                    ]);
                } else {
                    $product->setStockData([
                        "use_config_manage_stock" => 0,
                        "manage_stock" => 1,
                        "use_config_backorders" => 0,
                        "backorders" => $parsedParams["stock"]["backorders"] ? self::$backorderDefault : Mage_CatalogInventory_Model_Stock::BACKORDERS_NO,
                        "is_in_stock" => $parsedParams["stock"]["qty"] > 0 ? 1 : 0,
                        "qty" => $parsedParams["stock"]["qty"]
                    ]);
                }
            } else {
                $stockItem = $product->getStockItem();
                if ($parsedParams["stock"] === null && $stockItem->getData("manage_stock") > 0) {
                    $stockItem->setData("use_config_manage_stock", 1);
                    $stockItem->setData("manage_stock", 0);
                    $stockItem->setData("use_config_backorders", 1);
                    $stockItem->setData("backorders", Mage_CatalogInventory_Model_Stock::BACKORDERS_NO);
                    $stockItem->setData("is_in_stock", 1);
                    $stockItem->setData("qty", 999999);
                    $this->isProductUpdated = true;
                } elseif (!empty($parsedParams["stock"])) {
                    if ($stockItem->getData('use_config_manage_stock') != 0) {
                        $stockItem->setData('use_config_manage_stock', 0);
                        $this->isProductUpdated = true;
                    }
                    if ($stockItem->getData("manage_stock") != 1) {
                        $stockItem->setData("manage_stock", 1);
                        $this->isProductUpdated = true;
                    }
                    if ($stockItem->getData("use_config_backorders") != 0) {
                        $stockItem->setData("use_config_backorders", 0);
                        $this->isProductUpdated = true;
                    }
                    $backorders = $parsedParams["stock"]["backorders"] ? self::$backorderDefault : Mage_CatalogInventory_Model_Stock::BACKORDERS_NO;
                    if ($stockItem->getData("backorders") != $backorders) {
                        $stockItem->setData("backorders", $backorders);
                        $this->isProductUpdated = true;
                    }
                    if ($stockItem->getData("qty") != $parsedParams["stock"]["qty"]) {
                        $stockItem->setData("qty", $parsedParams["stock"]["qty"]);
                        $stockItem->setData("is_in_stock", $parsedParams["stock"]["qty"] > 0 ? 1 : 0);
                        $this->isProductUpdated = true;
                    }
                }
                if ($this->isProductUpdated) {
                    $stockItem->save();
                }
            }
        }

        // Only set these attributes if new product or not in batch mode
        if ($this->isCreate || !$parsedParams["isBatch"]) {

            // // Only add images to Awardit products
            if ($isAwarditProduct && array_key_exists("media", $parsedParams)) {
                $this->setImages($product, $parsedParams["media"], $this->isCreate);
            }
        }

    }

    public function updateLocalProduct(Mage_Catalog_Model_Product $product, $params): void
    {
        if (array_key_exists("name", $params) && strcmp($product->getName(), $params["name"]) !== 0) {
            $product->setData("name", $params["name"])->getResource()->saveAttribute($product, "name");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("shortDescription", $params) && strcmp($product->getShortDescription(), $params["shortDescription"]) !== 0) {
            $product->setData("short_description", $params["shortDescription"])->getResource()->saveAttribute($product, "short_description");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("longDescription", $params) && strcmp($product->getDescription(), $params["longDescription"]) !== 0) {
            $product->setData("description", $params["longDescription"])->getResource()->saveAttribute($product, "description");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("isFeatured", $params) && intval($product->getAwarditIsFeatured()) !== $params["isFeatured"]) {
            $product->setData("awardit_is_featured", $params["isFeatured"])->getResource()->saveAttribute($product, "awardit_is_featured");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("terms", $params) && strcmp($product->getAwarditTerms(), $params["terms"]) !== 0) {
            $product->setData("awardit_terms", $params["terms"])->getResource()->saveAttribute($product, "awardit_terms");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("checkTerms", $params) && $product->getAwarditCheckTerms() != $params["checkTerms"]) {
            $product->setData("awardit_check_terms", $params["checkTerms"])->getResource()->saveAttribute($product, "awardit_check_terms");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("partnerOwned", $params) && $product->getAwarditPartnerOwned() != $params["partnerOwned"]) {
            $product->setData("awardit_partner_owned", $params["partnerOwned"])->getResource()->saveAttribute($product, "awardit_partner_owned");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("validFrom", $params) && $product->getAwarditValidFrom() !== $params["validFrom"]) {
            $product->setData("awardit_valid_from", $params["validFrom"])->getResource()->saveAttribute($product, "awardit_valid_from");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("validTo", $params) && $product->getAwarditValidTo() !== $params["validTo"]) {
            $product->setData("awardit_valid_to", $params["validTo"])->getResource()->saveAttribute($product, "awardit_valid_to");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("invoiceName", $params) && strcmp($product->getAwarditInvoiceName(), $params["invoiceName"]) !== 0) {
            $product->setData("awardit_invoice_name", $params["invoiceName"])->getResource()->saveAttribute($product, "awardit_invoice_name");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("manufacturerId", $params) && !empty($params["manufacturerId"]) && $product->getManufacturer() != $params["manufacturerId"]) {
            $product->setData("manufacturer", $params["manufacturerId"])->getResource()->saveAttribute($product, "manufacturer");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("targetId", $params) && intval($product->getAwarditTargetId()) !== $params["targetId"]) {
            $product->setData("awardit_target_id", $params["targetId"])->getResource()->saveAttribute($product, "awardit_target_id");
            $this->isProductUpdated = true;
        }

        if ($this->setPriceAttributes($product, $params)) {
            $this->isProductUpdated = true;
        }

        // Check if we got prices
        if (array_key_exists("splitPaymentPrice", $params) && array_key_exists("invoicePrice", $params)) {

            // Check if product is awaiting price
            if ($product->getAwarditAwaitingPrice() == Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES) {

                // If we got prices and were waiting for it, set flag to not be waiting any longer
                $product->setData("awardit_awaiting_price", Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO)->getResource()->saveAttribute($product, "awardit_awaiting_price");
                $this->isProductUpdated = true;
            }
        }

        if (array_key_exists("status", $params)) {
            if ($params["status"] == Mage_Catalog_Model_Product_Status::STATUS_DISABLED || !$this->isWithinDates($product->getAwarditValidFrom(), $product->getAwarditValidTo())) {
                if ($product->getStatus() != Mage_Catalog_Model_Product_Status::STATUS_DISABLED) {
                    $product->setData("status", Mage_Catalog_Model_Product_Status::STATUS_DISABLED)->getResource()->saveAttribute($product, "status");
                    $this->isProductUpdated = true;
                }
            } else {
                if ($product->getStatus() != Mage_Catalog_Model_Product_Status::STATUS_ENABLED) {
                    $product->setData("status", Mage_Catalog_Model_Product_Status::STATUS_ENABLED)->getResource()->saveAttribute($product, "status");
                    $this->isProductUpdated = true;
                }
            }
        }

        if ($this->isProductUpdated) {
            $product->save();
            $this->isProductUpdated = false;
            $this->log("Created local product [{$product->getSku()}] for {$product->getStoreId()}");
        }

    }

    /**
     * Add attributes according to System > Configuration > Awardit > Integration > Upsert > SKU match
     *
     * @param Mage_Catalog_Model_Product $product
     * @param null|string|bool|int|Mage_Core_Model_Store $store
     * @return void
     */
    public function updateBySkuMapper(Mage_Catalog_Model_Product $product, $store): void
    {
        $unserializer = Mage::helper('core/unserializeArray');
        $map = Mage::getStoreConfig('integration/upsert/sku_properties', $store);

        if (empty($map)) {
            return; // Might be null, nothing to do
        }
        $map = $unserializer->unserialize($map);
        foreach ($map as $item) {
            $check = str_replace('*', '.*', $item['sku']);
            $result = preg_match("/^({$check})$/", $product->getSku());
            if (empty($result)) {
                continue; // Not a match
            }
            $product->setData($item['attribute_id'], $item['value']);
            $this->isProductUpdated = true;

            $this->log("Set property {$item['attribute_id']}={$item['value']} on product [{$product->getSku()}]");
        }
    }


    public function setCategories(Mage_Catalog_Model_Product $product, $categories, Mage_Core_Model_Store $store): bool
    {
        // Check categories parameter
        $rootCategoryId = $this->getRootCategoryId($store->getStoreId());
        if (empty($rootCategoryId)) {
            Mage::throwException("Missing root category for store");
        }
        $sqlQuery = "
            SELECT
                ccev.`value` AS category_name,
                tree.entity_id AS category_id,
                ccev.store_id,
                ccp.`position`
            FROM catalog_category_entity_varchar ccev
            JOIN eav_attribute eava ON eava.attribute_id = ccev.attribute_id
            JOIN (
                SELECT
                    entity_id
                FROM
                    (SELECT * FROM catalog_category_entity cce ORDER BY cce.parent_id, entity_id) items_sorted,
                    (SELECT @iv := :rootCat) initialisation
                WHERE
                    find_in_set(parent_id, @iv)
                    AND length(@iv := concat(@iv, ',', entity_id))
            ) tree ON tree.entity_id = ccev.entity_id
            LEFT JOIN catalog_category_product ccp ON ccp.category_id = tree.entity_id AND ccp.product_id = :productId
            WHERE
                eava.attribute_code = 'name'
                AND (ccev.store_id = 0 OR ccev.store_id = :storeId)
        ";
        $params = [
            "rootCat" => $rootCategoryId,
            "storeId" => $store->getStoreId(),
            "productId" => $this->isCreate ? 0 : $product->getId()
        ];
        $categoryTree = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $params);

        $newCategories = [];
        $parentCategory = Mage::getModel('catalog/category')->load($rootCategoryId);
        foreach ($categories as $categoryName) {
            $foundIt = false;
            foreach ($categoryTree as $cat) {
                // Magento database is UTF8, API data is specified as UTF8, let's hope everyone follows the spec.
                if (strcasecmp($categoryName, $cat["category_name"]) === 0) {
                    $foundIt = true;
                    // Category name matches
                    if (empty($cat["position"])) {
                        // Product did not previously belong to category
                        $newCategories[] = $cat["category_id"];
                    }
                }
            }
            if (!$foundIt) {
                $category = Mage::getModel("catalog/category");
                $category->setName($categoryName);
                //$category->setUrlKey($categoryName); // If we skip this, Magento creates one automatically
                $category->setIsActive(1);
                $category->setDisplayMode(Mage_Catalog_Model_Category::DM_PRODUCT);
                $category->setIsAnchor(0);
                $category->setPath($parentCategory->getPath());
                $category->save();

                // Product is supposed to belong to new category
                $newCategories[] = $category->getId();

                // Make sure nothing survives if we need to create another category later
                unset($category);
            }
        }

        // Add product to categories
        if (!empty($newCategories)) {
            $previousCategories = $this->isCreate ? [] : $product->getCategoryIds();
            $categoryIds = array_unique(array_merge($previousCategories, $newCategories), SORT_NUMERIC);
            $product->setCategoryIds($categoryIds);
            return true;
        }

        return false;
    }

    public function setImages(Mage_Catalog_Model_Product $product, array $media, bool $require): void
    {
        $imageWasAdded = false;
        $hashes = [];
        $errors = [];
        $awarditImageHash = $product->getAwarditImageHash();

        if (!empty($awarditImageHash)) {
            $hashes = explode(",", $awarditImageHash);
        }

        // Only one image is sent, but there is potential to send more
        foreach ($media as $i => $image) {
            $realUrl = '<none>';

            try {
                // Extract filename from url
                $filename = basename($image["url"]);

                // Build full image url
                $realUrl = $this->parseImageUrl($image["url"]);

                // Only add images not previously known
                $hash = md5($realUrl);
                if (in_array($hash, $hashes)) {
                    $this->log("Skipping known image: {$realUrl} for [{$product->getSku()}]");
                    continue;
                }

                // Add new hash to list of old ones
                $hashes[] = $hash;

                // If not yet updated, this must be first time. Prepare image gallery before adding new image
                if (!$imageWasAdded) {
                    $this->resetImageGalleryToUseDefault($product->getId());
                    $this->setProductInStoreToUseNoImages(0, $product->getId());
                    $this->removeMediaGalleryImages($product);
                }

                $this->log("Loading media file: {$realUrl}");

                // Get image source from url
                $imageSource = $this->getImageSource($realUrl);

                // Save file locally
                $filePathAndName = $this->storeImageFile($filename, $imageSource);

                // Set media tags for image
                if ($image["primary"]) {
                    $tags = [
                        0 => "image",
                        1 => "small_image",
                        2 => "thumbnail"
                    ];
                } else {
                    $tags = [];
                }

                // Add image to product
                $product->addImageToMediaGallery($filePathAndName, $tags, true, false);

                // Set attributes for last added image
                $gallery = $product->getData('media_gallery');
                $lastImage = array_pop($gallery['images']);
                $lastImage['label'] = "";
                $lastImage['position'] = 0;
                $lastImage['disabled'] = 0;
                $lastImage['label_default'] = "";
                $lastImage['position_default'] = 0;
                $lastImage['disabled_default'] = 0;
                array_push($gallery['images'], $lastImage);
                $product->setData('media_gallery', $gallery);

                $this->mediaGalleryUpdated = true;
                $this->isProductUpdated = true;
                $imageWasAdded = true;

                $this->log("Stored media file: {$filePathAndName}");

            } catch (Exception $e) {
                // Generic exception
                $this->log("{$e->getMessage()} for [{$product->getSku()}]", Zend_Log::WARN);
                $errors[] = "{$e->getMessage()} for [{$product->getSku()}]";
            }
        }

        // If updated, it means we added at least one image. Save awardit_image_hash
        if ($imageWasAdded) {
            $product->setAwarditImageHash(implode(",", $hashes));
        } elseif ($require) {
            // At least one image must be set on product. If not, skip entire product.
            $this->log("No valid image provided for [{$product->getSku()}]", Zend_Log::ERR);
            throw new Awardit_Upsert_InputException("No valid image provided. " . implode("\n", $errors));
        }
    }

    public function setHttpClient(Client $client): void
    {
        $this->httpClient = $client;
    }

    public function getHttpClient(): Client
    {
        if (!$this->httpClient) {
            $this->httpClient = new Client();
        }
        return $this->httpClient;
    }

    public function removeMediaGalleryImages(Mage_Catalog_Model_Product $product): void
    {
        // stackoverflow.com/questions/5709496/magento-programmatically-remove-product-images

        $mediaGalleryData = $product->getMediaGallery();
        if (!isset($mediaGalleryData['images']) || !is_array($mediaGalleryData['images'])) {
            return;
        }

        $toDelete = [];
        foreach ($mediaGalleryData['images'] as $image) {
            $toDelete[] = $image['value_id'];
            @unlink(Mage::getBaseDir('media') . DS . 'catalog' . DS . 'product' . $image['file']);
        }
        if (!empty($toDelete)) {
            Mage::getResourceModel('catalog/product_attribute_backend_media')->deleteGallery($toDelete);
            $product->setMediaGallery([ "images" => [], "values" => [] ]);
        }

    }

    public function setCustomOptions(Mage_Catalog_Model_Product $product, $options): void
    {
        $oldOptionsHash = $product->getAwarditOptionsHash();
        $newOptionsHash = md5(json_encode($options));

        if (!empty($oldOptionsHash) && $newOptionsHash == $oldOptionsHash) {
            Mage::log("Skipping known custom options, hash is same.", Zend_Log::DEBUG, "api-debug.log");
            return;
        }

        $hasOptions = $product->getHasOptions();

        // Check if we got custom options
        if (!empty($options)) {

            // Remove old custom options, if any exists
            if ($hasOptions) {
                foreach ($product->getOptions() as $option) {
                    $option->delete();
                }
                Mage::log("Removed old custom options", Zend_Log::DEBUG, "api-debug.log");
            }

            // Add new custom options
            $product->setCanSaveCustomOptions(true);
            $product->setHasOptions(true);
            $product->setProductOptions($options);
            Mage::log("Added custom options: " . json_encode($options), Zend_Log::DEBUG, "api-debug.log");

            // Set new hash
            $product->setAwarditOptionsHash($newOptionsHash);

            // Save product
            $product->save();

        } else {

            /*
             * Skip removal of existing options if none is supplied
             * See Jira PAB-813 for more info
             *

            // Remove old custom options, if any exists
            if ($hasOptions) {
                foreach ($product->getOptions() as $option) {
                    $option->delete();
                }
                Mage::log("Removed old custom options", Zend_Log::DEBUG, "api-debug.log");
            };

            $product->setHasOptions(false);
            $product->setRequiredOptions(false);

            // Clear options hash
            $product->setAwarditOptionsHash("");

            // Save product
            $product->save();

            */
        }
    }

    public function getRootCategoryId($storeId)
    {
        $sqlQuery = "SELECT csg.root_category_id FROM core_store cs JOIN core_store_group csg ON csg.website_id = cs.website_id WHERE cs.store_id = :id";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, ["id" => $storeId]) ?: null;
    }

    public function getTaxClassId(Mage_Core_Model_Store $store, $percent)
    {
        $sqlQuery = "
            SELECT
                tc.product_tax_class_id
            FROM tax_calculation_rate tcr
            JOIN tax_calculation tc ON tc.tax_calculation_rate_id = tcr.tax_calculation_rate_id
            WHERE
                tcr.rate = :percent
                AND tcr.tax_country_id IN (
                    SELECT
                        COALESCE(
                            MAX(IF(ccd.scope = 'stores', ccd.value, NULL)),
                            MAX(IF(ccd.scope = 'websites', ccd.value, NULL)),
                            MAX(IF(ccd.scope = 'default', ccd.value, NULL))
                        ) AS country_code
                    FROM core_config_data ccd
                    WHERE ccd.path = 'general/country/default' AND (ccd.scope_id = 0 OR ccd.scope_id = :websiteId OR ccd.scope_id = :storeId)
                )
        ";
        $params = [
            "storeId" => $store->getStoreId(),
            "websiteId" => $store->getWebsiteId(),
            "percent" => floatval($percent)
        ];
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params) ?: 0;
    }

    public function setPriceAttributes(&$product, $params): bool
    {
        $wasUpdated = false;
        $storeId = $this->getLocalStore()->getId();

        if (array_key_exists("splitPaymentPrice", $params)) {
            if (floatval($product->getPrice()) != $params["splitPaymentPrice"]) {
                $product->setData("price", $params["splitPaymentPrice"])->getResource()->saveAttribute($product, "price");
                $wasUpdated = true;
            }
        }

        if (array_key_exists("invoicePrice", $params)) {

            // Check if invoice price exists as an attribute
            if ($product->getResource()->getAttribute('invoice_price')) {

                // Yes it does:
                if (floatval($product->getInvoicePrice()) != $params["invoicePrice"]) {
                    $product->setData("invoice_price", $params["invoicePrice"])->getResource()->saveAttribute($product, "invoice_price");
                    $wasUpdated = true;
                }
            } else {

                // If not, try the old way
                $sqlQuery = "
                    INSERT INTO awardit_upsert_product (`product_id`, `store_id`, `invoice_price`) VALUES (:pId, :sId, :price)
                    ON DUPLICATE KEY UPDATE `invoice_price` = VALUES(invoice_price), `updated_at` = NOW()
                ";
                $queryParams = [
                    "price" => $params["invoicePrice"],
                    "pId" => $product->getId(),
                    "sId" => $storeId
                ];
                Mage::getSingleton("core/resource")->getConnection("core_write")->query($sqlQuery, $queryParams);
                $wasUpdated = true;
            }
        }

        if (array_key_exists("msrp", $params) && floatval($product->getMsrp()) != $params["msrp"]) {
            $product->setData("msrp", $params["msrp"])->getResource()->saveAttribute($product, "msrp");
            $wasUpdated = true;
        }

        if (array_key_exists("taxClassId", $params) && floatval($product->getTaxClassId()) != $params["taxClassId"]) {
            $product->setData("tax_class_id", $params["taxClassId"])->getResource()->saveAttribute($product, "tax_class_id");
            $wasUpdated = true;
        }

        if (array_key_exists("points", $params)) {
            $this->setPointsCoreData($params, $product, $storeId);
            $wasUpdated = true;
        }

        return $wasUpdated;

    }

    public function getJsonBody($request)
    {
        $rawBody = $request->getRawBody();
        if (!empty($rawBody)) {
            $content_type = $request->getHeader("Content-Type");
            if (stripos("application/json", trim($content_type)) !== null) {
                $data = json_decode($rawBody, true);

                if ($data === null) {
                    throw new Awardit_Upsert_InputException("Unable to decode parameters");
                }

                return $data;
            }
        }

        return null;
    }

    public function isAwarditProduct(Mage_Catalog_Model_Product $product): bool
    {
        // Find out if product originates from Awardit or CLS
        return stripos($product->getSku(), "test-") === 0 || stripos($product->getSku(), "awd_") === 0;
    }

    public function isAuthorized($key)
    {
        $sqlQuery = "SELECT token FROM crossroads_awardit_token WHERE company_id = :companyId AND deleted_at IS NULL AND token = :token";
        $params = [
            "companyId" => self::AUTHORIZATION_COMPANY_ID,
            "token" => $key
        ];
        $token = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);
        return !empty($token);
    }

    public function isWithinDates($dateFrom, $dateTo, $dateNow = null)
    {
        // If no dates are given, it's considered to be within dates
        if (empty($dateFrom) && empty($dateTo)) {
            return true;
        }

        $now = new DateTime( empty($dateNow) ? date("Y-m-d") : $dateNow );
        $from = new DateTime( empty($dateFrom) ? $now->format("Y-m-d") : $dateFrom );
        $to = new DateTime( empty($dateTo) ? $now->format("Y-m-d") : $dateTo );

        // if now == from, it's within dates
        // if now == to, it's within dates
        // if now > from AND now < to, it's within dates
        return $now >= $from && $now <= $to;

    }

    public function setPointsCoreData($data, $product, $storeId, $customerGroupId = 0)
    {
        if (!array_key_exists("points", $data)) {
            // Points must be supplied, just return and do nothing
            return null;
        }

        $pointsCoreId = Mage::helper("integration/PointsCore")->getPointsCoreId($storeId);
        if (empty($pointsCoreId)) {
            Mage::throwException("This store does not have pointsCoreId");
        }

        $isPriceUpdated = false;

        $price = Mage::getModel("points_core/product_price");
        $price->loadByStoreProductTypeCustomerGroupId(
            $this->getLocalStore(),
            $product,
            $pointsCoreId,
            $customerGroupId
        );

        if ($data["points"] > 0) {
            $newPoints = [
                "price" => $data["points"],
                "min_price" => 0,
                "max_price" => null,
            ];
        } else {
            if ($price->getId() > 0) {
                $this->log("Removing points core data for [{$product->getSku()}] in store {$storeId}");
                $price->delete();
            }
            return null;
        }

        if (array_key_exists("minPoints", $data) && $data["minPoints"] > 0) {
            $newPoints["min_price"] = $data["minPoints"];
        }

        if (array_key_exists("maxPoints", $data) && $data["maxPoints"] > 0) {
            $newPoints["max_price"] = $data["maxPoints"];
        }

        if (!$price->getId()) {
            $isPriceUpdated = true;
            $price->setProductId($product->getId());
            $price->setStoreId($storeId);
            $price->setCustomerGroupId($customerGroupId);
            $price->setType($pointsCoreId);
            $this->log("Creating points core data for [{$product->getSku()}] in store {$storeId}");
        }

        foreach ($newPoints as $key => $val) {
            if ($val != $price->getData($key)) {
                $price->setData($key, $val);
                $isPriceUpdated = true;
            }
        }

        if ($isPriceUpdated) {
            $price->save();
        }

        return $price->getId();
    }

    /**
     * Write to log.
     * @param string $message Message to log.
     * @param 0|1|2|3|4|5|6|7|null $level Any Zend_Log level constant.
     */
    public function log(string $message, ?int $level = Zend_Log::DEBUG): void
    {
        Mage::log($message, $level, "upsert-api");
    }

    private function parseImageUrl(string $url): string
    {
        if (stripos($url, "http") === 0) {
            // If url starts with http, consider it a full url
            $realUrl = $url;
        } else {
            // If not, url is relative to configured url
            $storeId = $this->getLocalStore()->getId();
            $realUrl = Mage::getStoreConfig("awardit_points/server/host", $storeId) . (stripos($url, "/") === 0 ? $url : "/" . $url);
            if (stripos($realUrl, "http") === false) {
                throw new Exception("Can not resolve image url: {$url}");
            }
        }
        return str_replace(" ", "%20", $realUrl);
    }

    private function getImageSource(string $url): string
    {
        // Get image from remote (exception on connect failure, and not 2XX status)
        try {
            $response = $this->getHttpClient()->get($url);
        } catch (ConnectException $e) {
            throw new Exception("Unable to connect: {$url}");
        } catch (ClientException $e) {
            throw new Exception("Unable to read image file: {$url}");
        }

        // Verify valid response
        $code = $response->getStatusCode();
        if ($code != 200) {
            throw new Exception("Unallowed status code {$code} on {$url}");
        }
        $contentType = $response->getHeaderLine('Content-Type');
        if (!in_array($contentType, self::IMAGE_MIMETYPES)) {
            throw new Exception("Unallowed content mime type {$contentType} on {$url}");
        }
        $contentLength = $response->getHeaderLine('Content-Length');
        if ($contentLength < 1) {
            throw new Exception("Unallowed content length {$contentLength} on {$url}");
        }

        return $response->getBody()->getContents();
    }

    private function storeImageFile(string $filename, string $imageSource): string
    {
        // Build full path and filename
        $filePathAndName = Mage::getBaseDir("media") . DS . "import" . DS . $filename;

        // Write to local directory
        $written = file_put_contents($filePathAndName, $imageSource);
        if (!$written) {
            @unlink($filePathAndName);
            throw new Exception("Unable to create image file: {$filePathAndName}");
        }

        // Check image validity
        $imageInfo = @getimagesize($filePathAndName);
        if (!$imageInfo) {
            @unlink($filePathAndName);
            throw new Exception("Invalid image file: {$filePathAndName}");
        }
        if (!in_array($imageInfo['mime'], self::IMAGE_MIMETYPES)) {
            @unlink($filePathAndName);
            throw new Exception("Unallowed image mime type {$imageInfo['mime']}: {$filePathAndName}");
        }

        // Fix wrongly named files by adding correct extension
        $pathInfo = array_merge(['extension' => ''], pathinfo($filePathAndName));
        $extension = substr($imageInfo['mime'], 6);
        if ($pathInfo['extension'] != $extension) {
            $oldPathAndName = $filePathAndName;
            $filePathAndName = "{$pathInfo['dirname']}/{$pathInfo['basename']}.{$extension}";
            rename($oldPathAndName, $filePathAndName);
        }

        return $filePathAndName;
    }
}
