<?php

class Awardit_Upsert_Model_Cli_ProductIndex extends Awardit_Integration_Model_Cli {

    protected $_defaultHelper = "awardit_upsert";

    // /usr/local/bin/magento_launcher.sh magento10 upsert -m=ProductIndex -f=indexAll
    public function CLI_indexAll($param): void
    {
        $resource = Mage::getResourceSingleton("awardit_upsert/product");

        $resource->reindexAll();
    }

    // /usr/local/bin/magento_launcher.sh magento10 upsert -m=ProductIndex -f=indexSingleStore -p="storeId:<store-id>"
    public function CLI_indexSingleStore($param): void
    {
        $resource = Mage::getResourceSingleton("awardit_upsert/product");

        $wantedParams = [
            "storeId" => "int"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        if (!empty($extractedParams["storeId"])) {
            $store = Mage::getModel('core/store')->load($extractedParams["storeId"]);
            if (!empty($store) && $store->getId() == $extractedParams["storeId"]) {
                $resource->indexSingleStore($store);
            } else {
                Mage::log("Unknown store!", Zend_Log::ERR, "awardit_upsert");
                echo "Unknown store!\n";
            }
        } else {
            Mage::log("Missing parameter!", Zend_Log::ERR, "awardit_upsert");
            echo "Missing parameter!\n";
        }
    }

    // /usr/local/bin/magento_launcher.sh magento10 upsert -m=ProductIndex -f=indexSingleProduct -p="pId:<product_id>"
    public function CLI_indexSingleProduct($param): void
    {
        $resource = Mage::getResourceSingleton("awardit_upsert/product");

        $wantedParams = [
            "pId" => "int"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        if (!empty($extractedParams["pId"])) {
            $globalProduct = Mage::getModel("catalog/product")->setStoreId(0)->load($extractedParams["pId"]);
            if ($globalProduct && $globalProduct->getId() == $extractedParams["pId"]) {
                $resource->indexSingleProduct($globalProduct);
            }
        } else {
            Mage::log("Missing parameter!", Zend_Log::ERR, "awardit_upsert");
            echo "Missing parameter!\n";
        }
    }

    // /usr/local/bin/magento_launcher.sh magento10 upsert -m=ProductIndex -f=updateProductsMissingData
    public function CLI_updateProductsMissingData($param): void
    {
        $limit = $this->getLimit();
        if ($limit < 1) {
            $limit = 1;
        } elseif ($limit > 999) {
            $limit = 999;
        }
        $processed = 0;

        $resource = Mage::getResourceSingleton("awardit_upsert/product");
        $integration = Mage::getModel("integration/Cli_LocalProductImport");

        $sqlQuery1 = "
            SELECT
                e.sku,
                e.entity_id AS product_id
            FROM catalog_product_entity e
            JOIN catalog_product_website w ON w.product_id = e.entity_id
            JOIN core_store s ON s.website_id = w.website_id
            LEFT JOIN awardit_upsert_product p ON p.product_id = e.entity_id AND p.store_id = s.store_id
            WHERE
                e.type_id IN ('simple','virtual')
                AND e.sku NOT LIKE 'awd_%'
                AND p.purchase_price IS NULL
            GROUP BY e.sku
            ORDER BY e.sku
        ";
        $magentoResult = Mage::getSingleton("core/resource")->getConnection("core_write")->fetchAssoc($sqlQuery1);

        $sqlQuery2 = "
            SELECT
                Prod.ProdNo AS sku,
                MAX(Prod.Gr2) AS visma_status
            FROM Prod
            JOIN PrDcMat ON PrDcMat.ProdNo = Prod.ProdNo
            WHERE Prod.ProdNo IN ('%s') AND (PrDcMat.R1 IN (%s) OR PrDcMat.R11 = 1)
            GROUP BY Prod.ProdNo
            ORDER BY Prod.ProdNo
        ";
        $shopIds = Mage::helper("integration/LocalData")->getInstanceUniqueVismaShopIds();

        $batches = array_chunk(array_keys($magentoResult), 100);
        foreach ($batches as $batch) {

            $vismaResult = Mage::helper("integration/LocalData")->getVismaDB()->fetchAssoc(sprintf($sqlQuery2, implode("','", $batch), implode(",", $shopIds)));
            foreach (array_keys($vismaResult) as $sku) {

                if (array_key_exists($sku, $magentoResult)) {

                    $productId = intval($magentoResult[$sku]["product_id"]);
                    $product = Mage::getModel("catalog/product")->setStoreId(0)->load($productId);

                    switch (intval($vismaResult[$sku]["visma_status"])) {
                        case Awardit_Integration_Model_Cli_ProductImport::VISMA_PRODUCT_STATUS_ACTIVE:
                        case Awardit_Integration_Model_Cli_ProductImport::VISMA_PRODUCT_STATUS_TURN_OFF:
                        case Awardit_Integration_Model_Cli_ProductImport::VISMA_PRODUCT_STATUS_LAST:
                        case Awardit_Integration_Model_Cli_ProductImport::VISMA_PRODUCT_STATUS_SOLD_OUT:
                            $integration->tagProductsForImport([$sku]);
                            $integrationProduct = $integration->getProductToImport($sku);
                            if (empty($integrationProduct)) {
                                $msg = sprintf("Found no product to import with sku [%s]", $sku);
                                Mage::log($msg, Zend_Log::ERR, "awardit_upsert");
                                if ($this->getDebugMode()) {
                                    //file_put_contents("/var/www/magento10/var/log/upsert-debug.log", "{$msg}\n", FILE_APPEND);
                                    echo "{$msg}\n";
                                }
                                break;
                            }

                            $msg = sprintf("Importing and updating index for product [%s]", $sku);
                            Mage::log($msg, Zend_Log::DEBUG, "awardit_upsert");
                            if ($this->getDebugMode()) {
                                //file_put_contents("/var/www/magento10/var/log/upsert-debug.log", "{$msg}\n", FILE_APPEND);
                                echo "{$msg}\n";
                            }
                            $integration->mergeDataToProduct($integrationProduct);
                            $integration->updateIntegrationData($integrationProduct);
                            if ($integrationProduct["import_status"] == Awardit_Integration_Model_Cli_ProductImport::IMPORT_STATUS_FETCHED) {
                                $integration->importSingleProduct($integrationProduct);
                                $integration->updateIntegrationData($integrationProduct);
                            }
                            $processed++;
                            break;

                        case Awardit_Integration_Model_Cli_ProductImport::VISMA_PRODUCT_STATUS_EXPIRED:
                        case Awardit_Integration_Model_Cli_ProductImport::VISMA_PRODUCT_STATUS_ARCHIVED:
                        case Awardit_Integration_Model_Cli_ProductImport::VISMA_PRODUCT_STATUS_SEASON:
                            $msg = sprintf("Product [%s] has status %s in Visma and is therefore being disabled", $sku, $vismaResult[$sku]["visma_status"]);
                            Mage::log($msg, Zend_Log::DEBUG, "awardit_upsert");
                            if ($this->getDebugMode()) {
                                //file_put_contents("/var/www/magento10/var/log/upsert-debug.log", "{$msg}\n", FILE_APPEND);
                                echo "{$msg}\n";
                            }
                            $product->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
                            $product->setAwarditAwaitingPrice(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
                            $product->setWebsiteIds([]);
                            $product->save();
                            $resource->removeProductIndexes($productId);
                            $processed++;
                            break;

                        default:
                            // Unknown status!
                            $msg = sprintf("Unknown Visma status (%s) for product [%s]", vismaResult[$sku]["visma_status"], $sku);
                            Mage::log($msg, Zend_Log::ERR, "awardit_upsert");
                            if ($this->getDebugMode()) {
                                //file_put_contents("/var/www/magento10/var/log/upsert-debug.log", "{$msg}\n", FILE_APPEND);
                                echo "{$msg}\n";
                            }
                            break;

                    }
                    unset($magentoResult[$sku]);
                } else {
                    $msg = sprintf("How did we get a product from Visma [%s] NOT matching sku from Magento?!", $sku);
                    Mage::log($msg, Zend_Log::ERR, "awardit_upsert");
                    if ($this->getDebugMode()) {
                        //file_put_contents("/var/www/magento10/var/log/upsert-debug.log", "{$msg}\n", FILE_APPEND);
                        echo "{$msg}\n";
                    }
                }

                if ($processed >= $limit) {
                    $magentoResult = [];
                    break 2;
                }
            }
        }

        // What remains is products that no longer have any price row in Visma
        // NOT (PrDcMat.R11 = 1 OR PrDcMat.R1 IN (<any Visma shop id configured in current Magento instance>))
        if (!empty($magentoResult)) {
            foreach($magentoResult as $sku => $data) {
                $msg = sprintf("Product [%s] has no price in Visma for this Magento instance, disabling it", $sku);
                Mage::log($msg, Zend_Log::DEBUG, "awardit_upsert");
                if ($this->getDebugMode()) {
                    //file_put_contents("/var/www/magento10/var/log/upsert-debug.log", "{$msg}\n", FILE_APPEND);
                    echo "{$msg}\n";
                }
                $productId = intval($data["product_id"]);
                $product = Mage::getModel("catalog/product")->setStoreId(0)->load($productId);
                $product->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
                $product->setAwarditAwaitingPrice(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
                $product->setWebsiteIds([]);
                $product->save();
                $resource->removeProductIndexes($productId);
                $processed++;

                if ($processed >= $limit) {
                    break;
                }
            }
        }
    }

    // /usr/local/bin/magento_launcher.sh magento10 upsert -m=ProductIndex -f=updatePurchasePrice
    public function CLI_updatePurchasePrice($param): void
    {
        $limit = $this->getLimit();
        if ($limit < 1) {
            $limit = 1;
        } elseif ($limit > 9999) {
            $limit = 9999;
        }

        // Fetch lists
        $productList = $this->getProductList();
        $storeCurrencyList = $this->getStoreCurrencyList();
        $priceInserts = [];
        $priceInsertQty = 0;
        $processedProductsQty = 0;

        // Iterate over list
        foreach ($productList as $productData) {
            $productId = intval(Mage::getModel("catalog/product")->getIdBySku($productData["sku"]));
            if (empty($productId)) {
                //echo "Product [{$productData["sku"]}] does not exist in this Magento instance.\n";
                continue;
            }

            if (empty($productData["purchase_price"])) {
                echo "Product [{$productData["sku"]}] have no purchase price!\n";
                continue;
            }

            $foundStore = false;
            foreach ($storeCurrencyList as $currencyData) {
                if ($productData["purchase_currency_id"] != $currencyData["currency_id"]) {
                    continue;
                }
                $priceInserts[] = "({$productId}, {$currencyData["store_id"]}, {$productData["purchase_price"]})";
                $priceInsertQty++;
                $foundStore = true;

                if ($priceInsertQty >= 1000) {
                    $this->setPurchasePrice($priceInserts);
                    $priceInserts = [];
                    $priceInsertQty = 0;
                }
            }

            if ($foundStore) {
                $processedProductsQty++;
                echo "Set purchase price = {$productData["purchase_price"]} in currency '{$currencyData["currency_code"]}' for product #{$productId} [{$productData["sku"]}]\n";
                if ($processedProductsQty >= $limit) {
                    break;
                }
            }

        }

        if (!empty($priceInserts)) {
            $this->setPurchasePrice($priceInserts);
        }

    }

    public function getProductList()
    {
        $sqlQuery = "
            SELECT
                Prod.ProdNo as sku,
                PrDcMat.PurcPr AS purchase_price,
                PrDcMat.PurcCur AS purchase_currency_id
            FROM Prod
            JOIN PrDcMat ON PrDcMat.ProdNo = Prod.ProdNo
            WHERE
                Prod.ProdNo IN (SELECT DISTINCT(p.ProdNo) AS ProdNo FROM PrDcMat p WHERE p.R11 = 1)
                AND Prod.Gr7 = 1
                AND Prod.Gr2 != 4
                AND PrDcMat.R11 = 1
            ORDER BY Prod.ProdNo, PrDcMat.PurcCur
        ";
        return Mage::helper("integration/LocalData")->getVismaDB()->fetchAll($sqlQuery);

    }

    public function getStoreCurrencyList()
    {
        $sqlQuery = "
            SELECT
                t.store_id,
                COALESCE(MAX(t.local_currency_code), MAX(t.default_currency_code)) AS currency_code,
                0 AS currency_id
            FROM (
                SELECT
                    s.store_id,
                    IF(d.scope_id > 0, d.value, NULL) AS local_currency_code,
                    IF(d.scope_id = 0, d.value, NULL) AS default_currency_code
                FROM core_store s
                LEFT JOIN core_config_data d ON (s.website_id = d.scope_id AND d.scope = 'websites') OR d.scope_id = 0
                WHERE d.path = 'currency/options/base'
            ) t
            GROUP BY t.store_id
            ORDER BY t.store_id
        ";
        $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery);

        foreach(array_keys($data) as $storeId) {
            $data[$storeId]["currency_id"] = Mage::helper("integration/LocalData")->getVismaCurrencyId($data[$storeId]["currency_code"]);
        }

        return $data;

    }

    public function setPurchasePrice($inserts)
    {
        //echo "Inserts:\n";
        //echo implode(",\n", $inserts) . "\n";

        $sqlQuery = "INSERT INTO awardit_upsert_product (product_id, store_id, purchase_price) VALUES ";
        $sqlQuery .= implode(",", $inserts);
        $sqlQuery .= " ON DUPLICATE KEY UPDATE purchase_price = VALUES(purchase_price)";

        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
        $stmt->execute();

        return;

    }


}
