<?php

class Awardit_Upsert_Model_Resource_Product extends Mage_Core_Model_Resource_Db_Abstract
{
    protected function _construct(): void
    {
        $this->_init("awardit_upsert/product", "index_id");
    }

    public function getIdByProductIdAndStoreId($productId, $storeId): int
    {
        $sqlQuery = "SELECT `index_id` FROM `awardit_upsert_product` WHERE `product_id` = :productId AND `store_id` = :storeId";
        return $this->_getWriteAdapter()->fetchOne($sqlQuery, [ "productId" => $productId, "storeId" => $storeId ]);
    }

    public function loadByProductIdAndStoreId($productId, $storeId): self
    {
        $indexId = $this->getIdByProductIdAndStoreId($productId, $storeId);

        if (!empty($indexId)) {
            return $this->load($indexId);
        }

        return $this;
    }

    public function reindexAll(): void
    {
        foreach (Mage::app()->getWebsites() as $website) {
            foreach ($website->getGroups() as $group) {
                $stores = $group->getStores();

                foreach ($stores as $store) {
                    $this->indexSingleStore($store);
                }
            }
        }
    }

    public function indexSingleStore(Mage_Core_Model_Store $store): void
    {
        $conn = $this->_getWriteAdapter();
        $storeId = $store->getId();
        $appEmulation = Mage::getSingleton("core/app_emulation");
        $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);
        $currentProductList = [];
        $clearIndexes = [];

        $conn->query("SET SESSION wait_timeout = 28800");
        $conn->query("SET SESSION interactive_timeout = 28800");

        try {
            $sqlQuery = "
                SELECT
                    cpe.entity_id,
                    (
                        SELECT
                            COALESCE(
                                MAX(IF(catalog_product_entity_int.store_id = cs.store_id, catalog_product_entity_int.`value`, NULL)),
                                MAX(IF(catalog_product_entity_int.store_id = 0, catalog_product_entity_int.`value`, NULL))
                            )
                        FROM catalog_product_entity_int
                        JOIN eav_attribute ON eav_attribute.attribute_id = catalog_product_entity_int.attribute_id
                        WHERE catalog_product_entity_int.entity_id = cpe.entity_id AND eav_attribute.attribute_code = 'status' AND (catalog_product_entity_int.store_id = 0 OR catalog_product_entity_int.store_id = cs.store_id)
                        GROUP BY catalog_product_entity_int.entity_id
                    ) AS product_status,
                    (
                        SELECT
                            COALESCE(
                                MAX(IF(catalog_product_entity_int.store_id = cs.store_id, catalog_product_entity_int.`value`, NULL)),
                                MAX(IF(catalog_product_entity_int.store_id = 0, catalog_product_entity_int.`value`, NULL))
                            )
                        FROM catalog_product_entity_int
                        JOIN eav_attribute ON eav_attribute.attribute_id = catalog_product_entity_int.attribute_id
                        WHERE catalog_product_entity_int.entity_id = cpe.entity_id AND eav_attribute.attribute_code = 'awardit_awaiting_price' AND (catalog_product_entity_int.store_id = 0 OR catalog_product_entity_int.store_id = cs.store_id)
                        GROUP BY catalog_product_entity_int.entity_id
                    ) AS awardit_awaiting_price
                FROM catalog_product_entity cpe
                JOIN catalog_product_website cpw ON cpw.product_id = cpe.entity_id
                JOIN core_store cs ON cs.website_id = cpw.website_id
                WHERE cs.store_id = :storeId AND cpe.type_id IN ('simple','virtual')
            ";

            $currentProductList = $conn->fetchAssoc("SELECT product_id, index_id FROM awardit_upsert_product WHERE store_id = :storeId", ["storeId" => $storeId ]);
            $collection = $conn->fetchAll($sqlQuery, ["storeId" => $storeId ]);

            if (!empty($collection)) {
                Mage::log("Refreshing upsert index for store {$storeId}", Zend_Log::DEBUG, "awardit_upsert");
                foreach ($collection as $product) {
                    if ($product["product_status"] == Mage_Catalog_Model_Product_Status::STATUS_DISABLED && $product["awardit_awaiting_price"] == Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO) {
                        $clearIndexes[] = $product["entity_id"];
                    } else {
                        if ($this->updateProduct($product["entity_id"], $store)) {
                            if (array_key_exists($product["entity_id"], $currentProductList)) {
                                unset($currentProductList[$product["entity_id"]]);
                            }
                        } else {
                            $clearIndexes[] = $product["entity_id"];
                        }
                    }
                }
            }

            $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
            $initialEnvironmentInfo = null;
        }
        catch(Exception $e) {
            Mage::logException($e);
            throw $e;
        }
        finally {
            if ($initialEnvironmentInfo) {
                $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
            }

            if (!empty($clearIndexes)) {
                $this->clearProductIndexes($clearIndexes, $storeId);
            }

            if (!empty($currentProductList)) {
                $this->clearProductIndexes(array_keys($currentProductList), $storeId);
            }
        }
    }

    public function indexSingleProduct(Mage_Catalog_Model_Product $globalProduct): void
    {
        $productId = $globalProduct->getId();
        $currentWebsites = $globalProduct->getWebsiteIds();
        $appEmulation = Mage::getSingleton("core/app_emulation");
        $initialEnvironmentInfo = null;
        $haveCorrectType = in_array($globalProduct->getTypeId(), [ Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL ]);
        $clearIndexes = [];

        Mage::log("Refreshing upsert index for product [{$globalProduct->getSku()}]", Zend_Log::DEBUG, "awardit_upsert");

        try {
            foreach (Mage::app()->getWebsites() as $website) {
                $removeWebsite = !in_array($website->getId(), $currentWebsites);

                foreach ($website->getGroups() as $group) {
                    $stores = $group->getStores();

                    foreach ($stores as $store) {
                        $storeId = $store->getId();

                        if ($removeWebsite || !$haveCorrectType) {
                            $clearIndexes[] = intval($storeId);
                            continue;
                        }

                        $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);

                        if (!$this->updateProduct($productId, $store)) {
                            $clearIndexes[] = intval($storeId);
                        }

                        $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);

                        $initialEnvironmentInfo = null;
                    }
                }
            }
        }
        catch(Exception $e) {
            Mage::logException($e);
            throw $e;
        }
        finally {
            if ($initialEnvironmentInfo) {
                $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
            }

            if (!empty($clearIndexes)) {
                $this->clearStoreIndexes($productId, $clearIndexes);
            }
        }
    }

    public function updateProduct(int $productId, Mage_Core_Model_Store $store): bool
    {
        $storeId = $store->getId();
        $conn = $this->_getWriteAdapter();
        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);

        // Skip index if produckt is disabled and NOT waiting for price.
        if ($localProduct->getStatus() == Mage_Catalog_Model_Product_Status::STATUS_DISABLED &&
            $localProduct->getAwarditAwaitingPrice() == Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO) {
            return false;
        }

        if( ! in_array($localProduct->getTypeId(), [Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL])) {
            return false;
        }

        $productData = $this->serializeItem($store, $localProduct);

        $sqlQuery = "
            INSERT INTO awardit_upsert_product (`product_id`, `store_id`, `product_data`) VALUES (:productId, :storeId, :productData)
            ON DUPLICATE KEY UPDATE `product_data` = VALUES(product_data), `updated_at` = NOW()
        ";
        $jsonFlags = JSON_PRESERVE_ZERO_FRACTION // Preserve the floats
            | JSON_UNESCAPED_SLASHES // Escaped slashes is an annoyance and a non-standard part of JSON
            | JSON_UNESCAPED_UNICODE; // We send full unicode encoded anyway

        $stmt = $conn->prepare($sqlQuery);
        $stmt->bindValue("productId", $productId);
        $stmt->bindValue("storeId", $storeId);
        $stmt->bindValue("productData", json_encode($productData, $jsonFlags));
        $stmt->execute();

        return true;
    }

    public function clearProductIndexes(array $productIds, int $storeId): bool
    {
        $conn = $this->_getWriteAdapter();

        try {
            $batches = array_chunk($productIds, 100);
            foreach ($batches as $batch) {
                $batchStr = implode(",", $batch);
                $sqlQuery = "UPDATE awardit_upsert_product SET `product_data` = NULL WHERE store_id = {$storeId} AND product_id IN ({$batchStr})";
                $conn->query($sqlQuery);
            }
            return true;

        } catch (Exception $e) {
            Mage::logException($e);
            throw $e;
        }
    }

    public function removeProductIndexes(int $productId): bool
    {
        $conn = $this->_getWriteAdapter();

        try {
            $sqlQuery = "DELETE FROM awardit_upsert_product WHERE product_id = :pId";
            $conn->query($sqlQuery, ["pId" => $productId]);
            return true;
        } catch (Exception $e) {
            Mage::logException($e);
            throw $e;
        }
    }

    public function clearStoreIndexes(int $productId, array $storeIds): bool
    {
        $conn = $this->_getWriteAdapter();

        try {
            $storeIdStr = implode(",", $storeIds);
            $sqlQuery = "UPDATE awardit_upsert_product SET `product_data` = NULL WHERE product_id = {$productId} AND store_id IN ({$storeIdStr})";
            $conn->query($sqlQuery);

            return true;

        } catch (Exception $e) {
            Mage::logException($e);
            throw $e;
        }
    }

    public function reindexProductIds(array $ids): void
    {
        foreach($ids as $id) {
            $product = Mage::getModel("catalog/product");

            $product->load($id);

            $this->indexSingleProduct($product);
        }
    }

    public function getProductParentsByChild(int $childId): array
    {
        $write = $this->_getWriteAdapter();
        $select = $write->select()
            ->from(["l" => $this->getTable("catalog/product_relation")], ["parent_id"])
            ->join(
                ["e" => $this->getTable("catalog/product")],
                "l.parent_id = e.entity_id",
                ["e.type_id"]
            )
            ->where("l.child_id = ?", $childId);

        return $write->fetchPairs($select);
    }

    public function serializeItem(Mage_Core_Model_Store $store, Mage_Catalog_Model_Product $product): array
    {
        $stock = Mage::getModel("cataloginventory/stock_item")->loadByProduct($product);
        $image = $this->serializeImage($product, "image", 600);
        $taxHelper = Mage::helper("tax");

        if($store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX)) {
            $incTax = $product->getPrice();
            $exTax = $taxHelper->getPrice($product, $incTax, false, null, null, null, $store, true);
        }
        else {
            $exTax = $product->getPrice();
            $incTax = $taxHelper->getPrice($product, $exTax, true, null, null, null, $store, false);
        }

        $taxRate = $exTax > 0 ? round(max(0, $incTax / $exTax - 1) * 100, 2) : 0;

        return [
            "id" => (int) $product->getEntityId(),
            "type" => $product->getTypeId(),
            "name" => $product->getName(),
            "sku" => $product->getSku(),
            "msrp" => (float)$product->getMsrp(),
            "categoryIds" => array_map("intval", $product->getCategoryIds() ?: []),
            "attributes" => [
                "manufacturer" => $this->getAttributeValue($product, "manufacturer"),
            ],
            "shortDescription" => $product->getShortDescription(),
            "description" => $product->getDescription(),
            "smallImage" => $image,
            "largeImage" => $image,
            "originalImage" => $this->serializeImage($product, "image"),
            "isSalable" => $product->getIsSalable() > 0,
            "mediaGallery" => $this->serializeMediaGallery($product),
            "stockQty" => $stock->getManageStock() ? (double)$stock->getQty() : null,
            "stockBackorders" => (bool)$stock->getBackorders(),
            "stockManage" => (bool)$stock->getManageStock(),
            // We have no options for simple/virtual
            "options" => [],
            "taxPercent" => $taxRate,

            // Populated by upsert/api-wrapper:
            // * msrp
            // * price
            // * points
            // * isInStock
        ];
    }

    public function serializeMediaGallery(Mage_Catalog_Model_Product $product): array
    {
        return array_values(array_map(function($image) use($product) {
            $resizedImage = (string)Mage::helper("catalog/image")
                    ->init($product, "image", $image->getFile())
                    ->keepFrame(true)
                    ->resize(600);

            return [
                "thumbnail" => (string)Mage::helper("catalog/image")
                    ->init($product, "image", $image->getFile())
                    ->keepFrame(true)
                    ->resize(70),
                "original" => Mage::getModel("catalog/product_media_config")->getMediaUrl($image->getFile()),
                "large" => $resizedImage,
                "image" => $resizedImage,
            ];
        }, $product->getMediaGalleryImages()->getItems()));
    }

    public function getAttributeValue(Mage_Catalog_Model_Product $product, string $attr): ?string
    {
        $data = $product->getData($attr);
        $attrResource = $product->getResource()->getAttribute($attr);

        if( ! $attrResource) {
            throw new Exception(sprintf(
                "Failed to fetch attribute resource for '%s'",
                $attr
            ));
        }

        return $attrResource->getSource()->getOptionText($data) ?: $data;
    }

    public function serializeImage(Mage_Catalog_Model_Product $product, string $attr, ?int $size = null): ?string
    {
        $img = Mage::helper("catalog/image");

        $img->init($product, $attr);
        $img->keepFrame(true);

        if($size) {
            $img->resize($size);
        }

        return (string)$img;
    }
}
