<?php

class Awardit_Upsert_Model_Resource_Product extends Mage_Core_Model_Resource_Db_Abstract
{
    protected function _construct(): void
    {
        $this->_init("awardit_upsert/product", "index_id");
    }

    /**
     * @return int
     */
    public function getIdByProductIdAndStoreId($productId, $storeId)
    {
        $sqlQuery = "SELECT `index_id` FROM `awardit_upsert_product` WHERE `product_id` = :productId AND `store_id` = :storeId";
        return $this->_getWriteAdapter()->fetchOne($sqlQuery, [ "productId" => $productId, "storeId" => $storeId ]);
    }

    public function loadByProductIdAndStoreId($productId, $storeId): self
    {
        $indexId = $this->getIdByProductIdAndStoreId($productId, $storeId);

        if (!empty($indexId)) {
            return $this->load($indexId);
        }

        return $this;
    }

    public function reindexAll(): void
    {
        foreach (Mage::app()->getWebsites() as $website) {
            foreach ($website->getGroups() as $group) {
                $stores = $group->getStores();

                foreach ($stores as $store) {
                    $this->indexSingleStore($store);
                }
            }
        }
    }

    public function indexSingleStore(Mage_Core_Model_Store $store): void
    {
        $conn = $this->_getWriteAdapter();
        $storeId = $store->getId();
        $appEmulation = Mage::getSingleton("core/app_emulation");
        $initialEnvironmentInfo = null;
        $currentProductList = [];
        $removeIndexes = [];

        try {
            $sqlQuery = "
                SELECT
                    cpe.entity_id,
                    (
                        SELECT
                            COALESCE(
                                MAX(IF(catalog_product_entity_int.store_id = cs.store_id, catalog_product_entity_int.`value`, NULL)),
                                MAX(IF(catalog_product_entity_int.store_id = 0, catalog_product_entity_int.`value`, NULL))
                            )
                        FROM catalog_product_entity_int
                        JOIN eav_attribute ON eav_attribute.attribute_id = catalog_product_entity_int.attribute_id
                        WHERE catalog_product_entity_int.entity_id = cpe.entity_id AND eav_attribute.attribute_code = 'status' AND (catalog_product_entity_int.store_id = 0 OR catalog_product_entity_int.store_id = cs.store_id)
                        GROUP BY catalog_product_entity_int.entity_id
                    ) AS product_status,
                    (
                        SELECT
                            COALESCE(
                                MAX(IF(catalog_product_entity_int.store_id = cs.store_id, catalog_product_entity_int.`value`, NULL)),
                                MAX(IF(catalog_product_entity_int.store_id = 0, catalog_product_entity_int.`value`, NULL))
                            )
                        FROM catalog_product_entity_int
                        JOIN eav_attribute ON eav_attribute.attribute_id = catalog_product_entity_int.attribute_id
                        WHERE catalog_product_entity_int.entity_id = cpe.entity_id AND eav_attribute.attribute_code = 'awardit_awaiting_price' AND (catalog_product_entity_int.store_id = 0 OR catalog_product_entity_int.store_id = cs.store_id)
                        GROUP BY catalog_product_entity_int.entity_id
                    ) AS awardit_awaiting_price
                FROM catalog_product_entity cpe
                JOIN catalog_product_website cpw ON cpw.product_id = cpe.entity_id
                JOIN core_store cs ON cs.website_id = cpw.website_id
                WHERE cs.store_id = :storeId AND cpe.type_id IN ('simple','virtual')
            ";

            $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);

            $factory = Mage::getModel("API/factory");
            $nullSer = Mage::getModel("API/serializer_constant", [ "value" => null ]);
            $prodSer = $factory->createProductDetailSerializer($store)
                ->setCustomOptionsSerializer($nullSer)
                ->setRelatedSerializer($nullSer);

            $currentProductList = $conn->fetchAssoc("SELECT product_id, index_id FROM awardit_upsert_product WHERE store_id = :storeId", ["storeId" => $storeId ]);
            $collection = $conn->fetchAll($sqlQuery, ["storeId" => $storeId ]);

            if (!empty($collection)) {
                Mage::log("Refreshing upsert index for store {$storeId}", Zend_Log::DEBUG, "awardit_upsert");
                foreach ($collection as $product) {
                    if ($product["product_status"] == Mage_Catalog_Model_Product_Status::STATUS_DISABLED && $product["awardit_awaiting_price"] == Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO) {
                        $removeIndexes[] = $product["entity_id"];
                    } else {
                        if ($this->updateProduct($product["entity_id"], $store, $prodSer)) {
                            if (array_key_exists($product["entity_id"], $currentProductList)) {
                                unset($currentProductList[$product["entity_id"]]);
                            }
                        } else {
                            $removeIndexes[] = $product["entity_id"];
                        }
                    }
                }
            }

            $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
            $initialEnvironmentInfo = null;

        }
        catch(Exception $e) {
            Mage::logException($e);
            throw $e;
        }
        finally {
            if ($initialEnvironmentInfo) {
                $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
            }

            $qtyToRemove = count($removeIndexes) + count($currentProductList);
            if ($qtyToRemove) {
                Mage::log("Removing {$qtyToRemove} rows of stale data from upsert index for store {$storeId}", Zend_Log::DEBUG, "awardit_upsert");

                if (!empty($removeIndexes)) {
                    $batches = array_chunk($removeIndexes, 100);
                    foreach ($batches as $batch) {
                        $this->_getWriteAdapter()->delete("awardit_upsert_product", [ "store_id = ?" => $storeId, "product_id IN (?)" => $batch ]);
                    }
                }

                if (!empty($currentProductList)) {
                    $batches = array_chunk(array_keys($currentProductList), 100);
                    foreach ($batches as $batch) {
                        $this->_getWriteAdapter()->delete("awardit_upsert_product", [ "store_id = ?" => $storeId, "product_id IN (?)" => $batch ]);
                    }
                }
            }
        }
    }

    public function indexSingleProduct(Mage_Catalog_Model_Product $globalProduct): void
    {
        $productId = $globalProduct->getId();
        $currentWebsites = $globalProduct->getWebsiteIds();
        $appEmulation = Mage::getSingleton("core/app_emulation");
        $initialEnvironmentInfo = null;
        $haveCorrectType = in_array($globalProduct->getTypeId(), [ Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL ]);
        $removeIndexes = [];

        Mage::log("Refreshing upsert index for product [{$globalProduct->getSku()}]", Zend_Log::DEBUG, "awardit_upsert");

        try {
            foreach (Mage::app()->getWebsites() as $website) {
                $removeWebsite = !in_array($website->getId(), $currentWebsites);

                foreach ($website->getGroups() as $group) {
                    $stores = $group->getStores();

                    foreach ($stores as $store) {
                        $storeId = $store->getId();

                        if ($removeWebsite || !$haveCorrectType) {
                            $removeIndexes[] = intval($storeId);
                            continue;
                        }

                        $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);

                        $factory = Mage::getModel("API/factory");
                        $nullSer = Mage::getModel("API/serializer_constant", [ "value" => null ]);
                        $prodSer = $factory->createProductDetailSerializer($store)
                            ->setCustomOptionsSerializer($nullSer)
                            ->setRelatedSerializer($nullSer);

                        if (!$this->updateProduct($productId, $store, $prodSer)) {
                            $removeIndexes[] = intval($storeId);
                        }

                        $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);

                        $initialEnvironmentInfo = null;
                    }
                }
            }
        }
        catch(Exception $e) {
            Mage::logException($e);
            throw $e;
        }
        finally {
            if ($initialEnvironmentInfo) {
                $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
            }
            if (!empty($removeIndexes)) {
                $qtyToRemove = count($removeIndexes);
                Mage::log("Removing {$qtyToRemove} rows of stale data from upsert index for product [{$globalProduct->getSku()}]", Zend_Log::DEBUG, "awardit_upsert");
                $this->_getWriteAdapter()->delete("awardit_upsert_product", [ "product_id = ?" => $productId, "store_id IN (?)" => $removeIndexes ]);
            }
        }
    }

    /**
     * @param int $productId
     */
    public function updateProduct(
        $productId,
        Mage_Core_Model_Store $store,
        Crossroads_API_Model_Serializer_Abstract $prodSer
    ): bool {
        $storeId = $store->getId();
        $conn = $this->_getWriteAdapter();
        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);

        // Skip index if produckt is disabled and NOT waiting for price.
        if ($localProduct->getStatus() == Mage_Catalog_Model_Product_Status::STATUS_DISABLED && $localProduct->getAwarditAwaitingPrice() == Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO) {
            return false;
        }

        $productData = $prodSer->serializeItem($localProduct);

        if (empty($productData)) {
            return true;
        }

        if (!empty($productData["options"])) {
            $haveChildren = false;

            foreach ($productData["options"] as $option) {
                if (!empty($option["values"])) {
                    $haveChildren = true;

                    break;
                }
            }

            if (!$haveChildren) {
                $productData = [];

                return true;
            }
        }

        if (array_key_exists("relatedProducts", $productData)) {
            unset($productData["relatedProducts"]);
        }

        $sqlQuery = "
            INSERT INTO awardit_upsert_product (`product_id`, `store_id`, `product_data`) VALUES (:productId, :storeId, :productData)
            ON DUPLICATE KEY UPDATE `product_data` = VALUES(product_data), `updated_at` = NOW()
        ";
        $stmt = $conn->prepare($sqlQuery);
        $stmt->bindValue("productId", $productId);
        $stmt->bindValue("storeId", $storeId);
        $stmt->bindValue("productData", json_encode($productData));
        $stmt->execute();

        return true;
    }

    public function reindexProductIds(array $ids): void {
        foreach($ids as $id) {
            $product = Mage::getModel("catalog/product");

            $product->load($id);

            $this->indexSingleProduct($product);
        }
    }

    /**
     * @param int $childId
     */
    public function getProductParentsByChild($childId): array {
        $write = $this->_getWriteAdapter();
        $select = $write->select()
            ->from(["l" => $this->getTable("catalog/product_relation")], ["parent_id"])
            ->join(
                ["e" => $this->getTable("catalog/product")],
                "l.parent_id = e.entity_id",
                ["e.type_id"]
            )
            ->where("l.child_id = ?", $childId);

        return $write->fetchPairs($select);
    }
}
