<?php
/*
 * Specification:
 * https://groot.crossroads.se/awardit/products-api/-/blob/master/swagger.yml
 *
 * This model was copied from local Magento10 module: Crossroads/Integration
 *
 * Java logfiles on awardit-staging: /opt/tomcat/logs/catalina.out
 * 
 */

class Awardit_Upsert_Model_Api extends Mage_Core_Model_Abstract {

    const AUTHORIZATION_COMPANY_ID = 1;
    const AUTO_ENABLE = true; // TODO: Move to Magento config?
    protected $localStore = null;
    protected $isCreate = false;
    protected $isProductUpdated = false;
    protected $mediaGalleryUpdated = false;
    protected static $defaultVisibility = Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH;
    protected static $defaultType = Mage_Catalog_Model_Product_Type::TYPE_SIMPLE;
    protected static $backorderDefault = Mage_CatalogInventory_Model_Stock::BACKORDERS_YES_NONOTIFY;
    protected static $defaultCustomOptionType = "drop_down"; // field / area / file / drop_down / radio / checkbox / multiple / date / date_time / time
    protected static $allowedTypes = [
        "physical" => Mage_Catalog_Model_Product_Type::TYPE_SIMPLE,
        "virtual" => Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL
    ];
    protected static $allowedStatuses = [
        "enabled" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
        "disabled" => Mage_Catalog_Model_Product_Status::STATUS_DISABLED
    ];

    public function setLocalStore(Mage_Core_Model_Store $store)
    {
        $this->localStore = $store;
    }

    public function getLocalStore()
    {
        return $this->localStore;
    }

    public function createProductAction($params)
    {
        $this->isCreate = true;

        // Check sku parameter
        if (empty($params["sku"])) {
            Mage::throwException("Missing SKU");
        }
        $sku = $params["sku"];

        // Try to get product id using new sku, to make sure it doesn't already exist
        $productId = Mage::getModel("catalog/product")->getIdBySku($sku);
        if (!empty($productId)) {
            Mage::throwException("Product already exists");
        }

        $store = $this->getLocalStore();
        $isAwarditProduct = true; // New products is per definition an Awardit Product
        $parsedParams = $this->parseParameters($params, $store, $isAwarditProduct);
        Mage::log("Creating product [{$sku}]", LOG_DEBUG, "api-debug.log", true);

        // Setup new product using defaults and provided parameters
        $newProduct = Mage::getModel("catalog/product");
        $newProduct->setStoreId(0);
        $newProduct->setSku($sku);
        $newProduct->setAttributeSetId(Crossroads_Integration_Helper_Data::$defaultAttributeSetId);
        $newProduct->setTypeId($parsedParams["typeId"]);
        $newProduct->setName($parsedParams["name"]);
        $newProduct->setDescription($parsedParams["longDescription"]);
        $newProduct->setShortDescription($parsedParams["shortDescription"]);
        $newProduct->setVisibility(self::$defaultVisibility); // catalog, search
        $newProduct->setAwarditIsFeatured(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $newProduct->setAwarditPartnerOwned(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $newProduct->setAwarditAwaitingPrice(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $newProduct->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
        $newProduct->setManufacturer($parsedParams["manufacturerId"]);
        $newProduct->setPrice($parsedParams["splitPaymentPrice"]);
        $newProduct->setMsrp($parsedParams["msrp"]);
        $newProduct->setTaxClassId($parsedParams["taxClassId"]);
        $newProduct->setWeight(1);
        $newProduct->setMediaGallery ([ "images" => [], "values" => [] ]);
        $this->updateGlobalProduct($newProduct, $parsedParams, $store);

        // If needed, save product
        if ($this->isProductUpdated) {
            $newProduct->save();
            $this->isProductUpdated = false;
        }

        // Reload new product in local store context
        $localProduct = Mage::getModel("catalog/product");
        $newProductId = $localProduct->getIdBySku($sku);
        if (empty($newProductId)) {
            Mage::throwException("Product was not saved correctly");
        }
        $storeId = $this->getLocalStore()->getStoreId();
        $localProduct->setStoreId($storeId)->load($newProductId);
        if (!$localProduct->getId()) {
            Mage::throwException("Unable to load newly created product");
        }

        $this->updateLocalProduct($localProduct, $parsedParams);

        // Only set these attributes if new product or not in batch mode
        if ($this->isCreate || !$parsedParams["isBatch"]) {

            $product = Mage::getModel("catalog/product")->setStoreId(0)->load($newProductId);

            // Only add custom options to Awardit products
            if ($isAwarditProduct && array_key_exists("options", $parsedParams) && is_array($parsedParams["options"])) {
                if ($this->setCustomOptions($product, $parsedParams["options"])) {
                    $this->isProductUpdated = true;
                }
            }
        }

        return [
            "sku" => $sku,
            "createdAt" => date("Y-m-d\TH:i:s\Z", strtotime($localProduct->getCreatedAt())),
            "updatedAt" => date("Y-m-d\TH:i:s\Z", strtotime($localProduct->getUpdatedAt())),
        ];
    }

    public function updateProductAction($params, $sku)
    {
        $this->isCreate = false;

        // Check sku parameter
        if (empty($sku)) {
            Mage::throwException("Missing SKU");
        }

        // Try to get product id using sku
        $productId = Mage::getModel('catalog/product')->getIdBySku($sku);
        if (empty($productId)) {
            // Mage::throwException("Unable to find product");
            return $this->createProductAction($params);
        }

        // Try to load product in default context
        $product = Mage::getModel('catalog/product')->setStoreId(0)->load($productId);
        if (!$product->getId()) {
            Mage::throwException("Unable to load product");
        }

        // Setup and parse input parameters
        $store = $this->getLocalStore();
        $storeId = $store->getStoreId();
        $isAwarditProduct = $this->isAwarditProduct($product);
        $parsedParams = $this->parseParameters($params, $store, $isAwarditProduct);

        // Update product in default context
        $this->updateGlobalProduct($product, $parsedParams, $store);
        Mage::log("Updating product [{$product->getSku()}]", LOG_DEBUG, "api-debug.log", true);

        // If needed, save product
        if ($this->isProductUpdated) {
            $product->save();
        }

        if ($this->mediaGalleryUpdated) {
            $this->resetImageGalleryToUseDefault($productId);
            $this->mediaGalleryUpdated = false;
        }

        // Try to load product in local context
        $localProduct = Mage::getModel('catalog/product')->setStoreFilter($store)->setStoreId($storeId)->load($productId);
        if (!$localProduct->getId()) {
            Mage::throwException("Unable to load product");
        }

        // Update product in local context
        $this->updateLocalProduct($localProduct, $parsedParams);

        // Only set these attributes if new product or not in batch mode
        if ($this->isCreate || !$parsedParams["isBatch"]) {

            // Only add custom options to Awardit products
            if ($isAwarditProduct && array_key_exists("options", $parsedParams) && is_array($parsedParams["options"])) {
                if ($this->setCustomOptions($product, $parsedParams["options"])) {
                    $this->isProductUpdated = true;
                }
            }
        }


        return [
            "createdAt" => date("Y-m-d\TH:i:s\Z", strtotime($localProduct->getCreatedAt())),
            "updatedAt" => date("Y-m-d\TH:i:s\Z", strtotime($localProduct->getUpdatedAt())),
        ];
    }

    public function resetImageGalleryToUseDefault($pId)
    {
        if ($pId > 0) {
            $sqlQuery = "
                DELETE catalog_product_entity_varchar.*
                FROM eav_attribute
                JOIN catalog_product_entity_varchar ON catalog_product_entity_varchar.attribute_id = eav_attribute.attribute_id
                WHERE eav_attribute.attribute_code IN ('image', 'small_image', 'thumbnail') AND eav_attribute.backend_model IS NULL AND store_id > 0 AND catalog_product_entity_varchar.entity_id = :pId
            ";
            Mage::getSingleton("core/resource")->getConnection("core_write")->query($sqlQuery, [ "pId" => $pId ]);
        }
    }

    public function setProductInStoreToUseNoImages($pId, $sId)
    {
        if ($pId > 0) {
            $sqlQuery = "
                UPDATE catalog_product_entity_varchar
                JOIN eav_attribute ON eav_attribute.attribute_id = catalog_product_entity_varchar.attribute_id
                SET catalog_product_entity_varchar.`value` = 'no_selection'
                WHERE eav_attribute.attribute_code IN ('image', 'small_image', 'thumbnail') AND eav_attribute.backend_model IS NULL AND store_id = :sId AND catalog_product_entity_varchar.entity_id = :pId
            ";
            Mage::getSingleton("core/resource")->getConnection("core_write")->query($sqlQuery, [ "sId" => $sId, "pId" => $pId ]);
        }
    }

    public function removeCustomOptionsForProduct($pId)
    {
        $sqlQuery = "DELETE catalog_product_option.* FROM catalog_product_option WHERE catalog_product_option.product_id = :pId";
        Mage::getSingleton("core/resource")->getConnection("core_write")->query($sqlQuery, [ "pId" => $pId ]);
    }

    public function listProductsAction($params, $sku = null)
    {
        // Setup and parse input parameters
        $store = $this->getLocalStore();
        $storeId = intval($store->getStoreId());
        $parsedParams = $this->parseGetParameters($params, $store);
        $pos = ($parsedParams["page"] - 1) * $parsedParams["limit"];

        if (empty($sku)) {
            $sqlQuery = "
                SELECT
                    aup.*,
                    ppp.price AS points,
                    ppp.min_price AS minPoints,
                    ppp.max_price AS maxPoints
                FROM awardit_upsert_product aup
                JOIN catalog_product_entity cpe ON cpe.entity_id = aup.product_id
                LEFT JOIN points_product_price ppp ON ppp.product_id = aup.product_id AND ppp.store_id = aup.store_id AND ppp.customer_group_id = 0
                WHERE aup.store_id = ? AND aup.product_data IS NOT NULL AND (aup.purchase_price > 0 OR cpe.sku LIKE 'awd_%')
                ORDER BY aup.product_id LIMIT {$pos}, {$parsedParams["limit"]}
            ";
            $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $storeId);
        } else {
            $productId = Mage::getModel('catalog/product')->getIdBySku($sku);
            if (empty($productId)) {
                Mage::throwException("Unable to find product");
            }
            $sqlQuery = "
                SELECT
                    aup.*,
                    ppp.price AS points,
                    ppp.min_price AS minPoints,
                    ppp.max_price AS maxPoints
                FROM awardit_upsert_product aup
                JOIN catalog_product_entity cpe ON cpe.entity_id = aup.product_id
                LEFT JOIN points_product_price ppp ON ppp.product_id = aup.product_id AND ppp.store_id = aup.store_id AND ppp.customer_group_id = 0
                WHERE aup.store_id = :sId AND aup.product_id = :pId AND aup.product_data IS NOT NULL AND (aup.purchase_price > 0 OR cpe.sku LIKE 'awd_%')
            ";
            $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, [ "sId" => $storeId, "pId" => $productId ]);
        }

        $output = [];
        foreach ($data as $index => $row) {
            $output[$index] = json_decode($row["product_data"], true);

            // Replace price by purchase price from Visma
            $output[$index]["price"] = $row["purchase_price"];

            // Copy points from pointsCore
            $output[$index]["points"] = $row["points"];
            $output[$index]["minPoints"] = $row["minPoints"];
            $output[$index]["maxPoints"] = $row["maxPoints"];

            // Recalculate isInStock, we have strange problems with this value from serializer
            $output[$index]["isInStock"] = $output[$index]["stockQty"] > 0;

        }

        return $output;

    }

    public function parseGetParameters($params, Mage_Core_Model_Store $store)
    {
        $parsedParams = [];

        // Check 'limit' parameter
        if (array_key_exists("limit", $params)) {
            if (empty($params["limit"])) {
                $parsedParams["limit"] = Crossroads_API_Model_Collection_Abstract::LIMIT_DEFAULT;
            } elseif(!is_numeric($params["limit"])) {
                Mage::throwException("Wrong value for parameter: 'limit'");
            } else {
                $parsedParams["limit"] = intval($params["limit"]);
            }
            if ($parsedParams["limit"] < 1 || $parsedParams["limit"] > Crossroads_API_Model_Collection_Abstract::LIMIT_MAX) {
                Mage::throwException("Wrong value for parameter: 'limit'");
            }
        } else {
            $parsedParams["limit"] = Crossroads_API_Model_Collection_Abstract::LIMIT_DEFAULT;
        }

        // Check 'page' parameter
        if (array_key_exists("page", $params)) {
            if (empty($params["page"])) {
                $parsedParams["page"] = 1;
            } elseif(!is_numeric($params["page"])) {
                Mage::throwException("Wrong value for parameter: 'page'");
            } else {
                $parsedParams["page"] = intval($params["page"]);
            }
            if ($parsedParams["page"] < 1 || $parsedParams["page"] > 9999) {
                Mage::throwException("Wrong value for parameter: 'page'");
            }
        } else {
            $parsedParams["page"] = 1;
        }

        return $parsedParams;
    }

    public function parseParameters($params, Mage_Core_Model_Store $store, $isAwarditProduct)
    {
        $parsedParams = [];

        // Check 'batch' parameter
        $parsedParams["isBatch"] = empty($params["batch"]) ? false : true;

        // Check 'type' parameter
        if (array_key_exists("type", $params)) {
            if (empty($params["type"])) {
                $parsedParams["typeId"] = self::$defaultType;
                // Mage::throwException("Wrong value for parameter: 'type'");
            } elseif(!array_key_exists($params["type"], self::$allowedTypes)) {
                Mage::throwException("Wrong value for parameter: 'type'");
            } else {
                $parsedParams["typeId"] = self::$allowedTypes[$params["type"]];
            }
        } elseif($this->isCreate) {
            Mage::throwException("Missing parameter: 'type'");
        }

        // Check 'manufacturer' parameter
        if (array_key_exists("manufacturer", $params)) {
            if (!empty($params["manufacturer"])) {
                $manufacturerId = Mage::Helper("integration")->getManufacturerId($params["manufacturer"]);
            } else {
                $manufacturerId = Mage::Helper("integration")->getManufacturerId("");
            }
            if (empty($manufacturerId)) {
                $manufacturerId = Mage::Helper("integration")->createManufacturer($params["manufacturer"]);
                if (empty($manufacturerId)) {
                    Mage::throwException("Unable to create manufacturer");
                }
            } else {
                $parsedParams["manufacturerId"] = $manufacturerId;
            }
        } elseif($this->isCreate) {
            Mage::throwException("Missing parameter: 'manufacturer'");
        }

        // Check 'status' parameter
        if (array_key_exists("status", $params)) {
            if (empty($params["status"])) {
                Mage::throwException("Wrong value for parameter: 'status'");
            } elseif (!array_key_exists($params["status"], self::$allowedStatuses)) {
                Mage::throwException("Wrong value for parameter: 'status'");
            }
            $parsedParams["status"] = self::$allowedStatuses[$params["status"]];
        } else {
            Mage::throwException("Missing parameter: 'status'");
        }

        // Check 'stock' parameter
        if (array_key_exists("stock", $params)) {
            if (is_array($params["stock"])) {
                if (!array_key_exists("qty", $params["stock"])) {
                    Mage::throwException("Missing parameter: 'qty' in 'stock'");
                } elseif(!is_numeric($params["stock"]["qty"])) {
                    Mage::throwException("Wrong value for parameter: 'qty' in 'stock'");
                }
                if (!array_key_exists("backorders", $params["stock"])) {
                    Mage::throwException("Missing parameter: 'backorders' in 'stock'");
                }
                $parsedParams["stock"] = [
                    "qty" => intval($params["stock"]["qty"]),
                    "backorders" => !empty($params["stock"]["backorders"]) ? true : false
                ];
            } elseif ($params["stock"] !== null) {
                Mage::throwException("Wrong value for parameter: 'stock'");
            } else {
                $parsedParams["stock"] = null; // Do not handle stock, product is always available
            }
        } elseif ($this->isCreate) {
            Mage::throwException("Missing parameter: 'stock'");
        }

        // Check 'name' parameter
        if (array_key_exists("name", $params)) {
            if (empty($params["name"])) {
                Mage::throwException("Wrong value for parameter: 'name'");
            } elseif (strlen($params["name"]) > 255) {
                Mage::throwException("String too long for parameter: 'name'");
            }
            $parsedParams["name"] = $params["name"];
        } elseif($this->isCreate) {
            Mage::throwException("Missing parameter: 'name'");
        }

        // Check 'shortDescription' parameter
        if (array_key_exists("shortDescription", $params)) {
            if (empty($params["shortDescription"])) {
                Mage::throwException("Wrong value for parameter: 'shortDescription'");
            } elseif (strlen($params["shortDescription"]) > 65535) {
                Mage::throwException("String too long for parameter: 'shortDescription'");
            }
            $parsedParams["shortDescription"] = $params["shortDescription"];
        } elseif($this->isCreate) {
            Mage::throwException("Missing parameter: 'shortDescription'");
        }

        // Check 'longDescription' parameter
        if (array_key_exists("longDescription", $params)) {
            if (empty($params["longDescription"])) {
                Mage::throwException("Wrong value for parameter: 'longDescription'");
            } elseif (strlen($params["longDescription"]) > 65535) {
                Mage::throwException("String too long for parameter: 'longDescription'");
            }
            $parsedParams["longDescription"] = $params["longDescription"];
        } elseif($this->isCreate) {
            Mage::throwException("Missing parameter: 'longDescription'");
        }

        // Check 'special' parameter
        if (array_key_exists("special", $params)) {
            $parsedParams["isFeatured"] = intval($params["special"]) === 1 ? 1 : 0;
        } elseif($this->isCreate) {
            $parsedParams["isFeatured"] = 0;
        }

        // Check 'validFrom' parameter (check for 'validfrom' as fallback)
        if (array_key_exists("validFrom", $params)) {
            if (empty($params["validFrom"])) {
                $parsedParams["validFrom"] = "";
            } elseif (preg_match('/[0-9]{4}-[0-9]{2}-[0-9]{2}/', $params["validFrom"])) {
                $parsedParams["validFrom"] = date("Y-m-d", strtotime($params["validFrom"]));
            } else {
                Mage::throwException("Wrong value for parameter: 'validFrom'");
            }
        } elseif (array_key_exists("validfrom", $params)) {
            if (empty($params["validfrom"])) {
                $parsedParams["validfrom"] = "";
            } elseif (preg_match('/[0-9]{4}-[0-9]{2}-[0-9]{2}/', $params["validfrom"])) {
                $parsedParams["validFrom"] = date("Y-m-d", strtotime($params["validfrom"]));
            } else {
                Mage::throwException("Wrong value for parameter: 'validFrom'");
            }
        }

        // Check 'validTo' parameter (check for 'validto' as fallback)
        if (array_key_exists("validTo", $params)) {
            if (empty($params["validTo"])) {
                $parsedParams["validTo"] = "";
            } elseif (preg_match('/^[0-9]{4}-[0-9]{2}-[0-9]{2}$/', $params["validTo"])) {
                $parsedParams["validTo"] = date("Y-m-d", strtotime($params["validTo"]));
            } else {
                Mage::throwException("Wrong value for parameter: 'validTo'");
            }
        } elseif (array_key_exists("validto", $params)) {
            if (empty($params["validto"])) {
                $parsedParams["validto"] = "";
            } elseif (preg_match('/^[0-9]{4}-[0-9]{2}-[0-9]{2}$/', $params["validto"])) {
                $parsedParams["validTo"] = date("Y-m-d", strtotime($params["validto"]));
            } else {
                Mage::throwException("Wrong value for parameter: 'validTo'");
            }
        }

        // Check 'invoiceName' parameter
        if (array_key_exists("invoiceName", $params)) {
            if (empty($params["invoiceName"])) {
                $parsedParams["invoiceName"] = "";
            } elseif (strlen($params["invoiceName"]) > 255) {
                Mage::throwException("String too long for parameter: 'invoiceName'");
            } else {
                $parsedParams["invoiceName"] = $params["invoiceName"];
            }
        }

        // Check 'partnerOwned' parameter
        if (array_key_exists("partnerOwned", $params)) {
            $parsedParams["partnerOwned"] = $params["partnerOwned"] ? 1 : 0;
        } elseif($this->isCreate) {
            $parsedParams["partnerOwned"] = 0;
        }

        // Check 'msrp' parameter
        if (array_key_exists("msrp", $params)) {
            if (!is_numeric($params["msrp"])) {
                Mage::throwException("Wrong value for parameter: 'msrp'");
            }
            $msrp = floatval($params["msrp"]);
            if ($msrp < 0.0) {
                Mage::throwException("Wrong value for parameter: 'msrp'");
            }
            $parsedParams["msrp"] = $msrp;
        } elseif($this->isCreate) {
            $parsedParams["msrp"] = 0.0;
        }

        // Check 'invoicePrice' parameter (used in order export to Visma)
        if (array_key_exists("invoicePrice", $params)) {
            if (!is_numeric($params["invoicePrice"])) {
                Mage::throwException("Wrong value for parameter: 'invoicePrice'");
            }
            $price = floatval($params["invoicePrice"]);
            if ($price > 0) {
                $parsedParams["invoicePrice"] = $price;
            } else {
                if ($parsedParams["status"] == "disabled") {
                    // Alow zero price if product is about to be disabled
                    $parsedParams["invoicePrice"] = $price;
                } else {
                    Mage::throwException("Wrong value for parameter: 'invoicePrice'");
                }
            }

        // Use old parameter 'price' as fallback if 'invoicePrice' is missing.
        } elseif (array_key_exists("price", $params)) {
            if (!is_numeric($params["price"])) {
                Mage::throwException("Wrong value for parameter: 'price'");
            }
            $price = floatval($params["price"]);
            if ($price > 0) {
                $parsedParams["invoicePrice"] = $price;
            } else {
                if ($parsedParams["status"] == "disabled") {
                    // Alow zero price if product is about to be disabled
                    $parsedParams["invoicePrice"] = $price;
                } else {
                    Mage::throwException("Wrong value for parameter: 'price' (also: should use parameter 'invoicePrice' instead!)");
                }
            }
        } else {
            Mage::throwException("Missing parameter: 'invoicePrice'");
        }

        // Check 'splitPaymentPrice' parameter (used as 'price')
        // Use invoice price as fallback
        if (array_key_exists("splitPaymentPrice", $params)) {
            if (is_numeric($params["splitPaymentPrice"])) {
                $price = floatval($params["splitPaymentPrice"]);
                if ($price > 0) {
                    $parsedParams["splitPaymentPrice"] = $price;
                } else {
                    $parsedParams["splitPaymentPrice"] = $parsedParams["invoicePrice"];
                }
            } else {
                $parsedParams["splitPaymentPrice"] = $parsedParams["invoicePrice"];
            }
        } else {
            $parsedParams["splitPaymentPrice"] = $parsedParams["invoicePrice"];
        }

        // Check 'points' parameter
        if (array_key_exists("points", $params)) {
            if (!is_numeric($params["points"])) {
                Mage::throwException("Wrong value for parameter: 'points'");
            }
            $points = intval($params["points"]);
            if ($points < 1) {
                $points = null; // Remove points from product
            }
            $parsedParams["points"] = $points;
        } elseif($this->isCreate) {
            Mage::throwException("Missing parameter: 'points'");
        }

        // Check 'minPoints' parameter
        if (array_key_exists("minPoints", $params)) {
            if (!(is_numeric($params["minPoints"]) || $params["minPoints"] === null)) {
                Mage::throwException("Wrong value for parameter: 'minPoints'");
            }
            $minPoints = intval($params["minPoints"]);
            if ($minPoints > 0) {
                $parsedParams["minPoints"] = $minPoints;
            }
        }

        // Check 'maxPoints' parameter
        if (array_key_exists("maxPoints", $params)) {
            if (!(is_numeric($params["maxPoints"]) || $params["maxPoints"] === null)) {
                Mage::throwException("Wrong value for parameter: 'maxPoints'");
            }
            $maxPoints = intval($params["maxPoints"]);
            if ($maxPoints > 0) {
                $parsedParams["maxPoints"] = $maxPoints;
            }
        }

        // Check 'vat' parameter
        if($this->isCreate) {
            if (array_key_exists("vat", $params)) {
                if (!is_numeric($params["vat"])) {
                    Mage::throwException("Wrong value for parameter: 'vat'");
                }
                $vat = floatval($params["vat"]);
                if ($vat < 0.0 || $vat >= 100.0) { // 100% VAT? yeah, right...
                    Mage::throwException("Wrong value for parameter: 'vat'");
                }
                $parsedParams["vat"] = $vat;
                $parsedParams["taxClassId"] = $this->getTaxClassId($store, $vat);
            } else {
                Mage::throwException("Missing parameter: 'vat'");
            }
        }

        // Check 'categories' parameter
        $parsedParams["categories"] = [];
        if (!empty($params["categories"])) {
            if (is_array($params["categories"])) {
                $parsedParams["categories"] = $params["categories"];
            } else {
                Mage::throwException("Wrong type for parameter: 'categories'");
            }
        }

        // Check 'media' parameter
        $parsedParams["media"] = [];
        if (!empty($params["media"])) {
            if (!is_array($params["media"])) {
                    Mage::throwException("Wrong type for parameter: 'media'");
            }
            foreach ($params["media"] as $mp) {
                if (!array_key_exists("url", $mp)) {
                    Mage::throwException("Missing parameter: 'url' in 'media'");
                } elseif (empty($mp["url"])) {
                    Mage::throwException("Wrong value for parameter: 'url' in 'media'");
                } elseif($mp["url"] === "null") {
                    // It seems that if something goes wrong with image uploading in the Java platform, image url gets the (string) value "null".
                    continue;
                }
                $parsedParams["media"][] = [
                    "url" => $mp["url"],
                    "primary" => !empty($mp["primary"]) ? true : false
                ];
            }
        }

        // Check 'options' parameter
        if ($isAwarditProduct && array_key_exists("options", $params)) {
            if ($params["options"] === null) {
                $parsedParams["options"] = null;
            } elseif (!is_array($params["options"])) {
                Mage::throwException("Wrong type for parameter: 'options'");
            } else {
                // [ {"name":"Golfklubba","required":true,"values":[ {"label":"Driver","value":"Driver"}, {"label":"Järnklubba","value":"Järnklubba"}, {"label":"Putte","value":"Putte"} ]} ]
                $parsedParams["options"] = [];
                foreach ($params["options"] as $optionIndex => $option) {
                    if (!array_key_exists("name", $option)) {
                        Mage::throwException("Missing parameter: 'name' in 'options'");
                    } elseif (empty($option["name"])) {
                        Mage::throwException("Wrong value for parameter: 'name' in 'options'");
                    } elseif (strlen($option["name"]) > 50) {
                        Mage::throwException("String too long for parameter: 'name' in 'options'");
                    }
                    if (!array_key_exists("required", $option)) {
                        Mage::throwException("Missing parameter: 'required' in 'options'");
                    }
                    if (!array_key_exists("values", $option)) {
                        Mage::throwException("Missing parameter: 'values' in 'options'");
                    } elseif (!is_array($option["values"])) {
                        Mage::throwException("Wrong type for parameter: 'values' in 'options'");
                    }
                    $parsedParams["options"][$optionIndex] = [
                        "title" => $option["name"],
                        "type" => self::$defaultCustomOptionType,
                        "is_require" => !empty($option["required"]) ? 1 : 0,
                        "values" => []
                    ];
                    foreach ($option["values"] as $valueIndex => $value) {
                        if (!array_key_exists("label", $value)) {
                            Mage::throwException("Missing parameter: 'label' in 'values' in 'options'");
                        } elseif (empty($value["label"])) {
                            Mage::throwException("Wrong value for parameter: 'label' in 'values' in 'options'");
                        } elseif (strlen($value["label"]) > 50) {
                            Mage::throwException("String too long for parameter: 'label' in 'values' in 'options'");
                        }
                        if (!empty($value["value"]) && strlen($value["value"]) > 50) {
                            Mage::throwException("String too long for parameter: 'value' in 'values' in 'options'");
                        }
                        $parsedParams["options"][$optionIndex]["values"][$valueIndex] = [
                            "title" => $value["label"],
                            "price" => 0,
                            "price_type" => "fixed",
                            "sort_order" => $valueIndex,
                            "sku" => !empty($value["value"]) ? $value["value"] : ""
                        ];
                    }
                }
            }
        }

        return $parsedParams;
    }

    public function updateGlobalProduct(Mage_Catalog_Model_Product &$product, $parsedParams, Mage_Core_Model_Store $store)
    {
        $isAwarditProduct = $this->isAwarditProduct($product);

        // Update categories
        if (array_key_exists("categories", $parsedParams)) {
            $this->setCategories($product, $parsedParams["categories"], $store);
        }

        // If product does not belong to local website, add it
        $websiteId = $store->getWebsiteId();
        $productWebsiteIds = $this->isCreate ? [] : $product->getWebsiteIds();
        if (!in_array($websiteId, $productWebsiteIds)) {
            $productWebsiteIds[] = $websiteId;
            $product->setWebsiteIds($productWebsiteIds);
            $this->isProductUpdated = true;
        }

        // Update stock
        if ($isAwarditProduct && array_key_exists("stock", $parsedParams)) {
            if ($this->isCreate) {
                if ($parsedParams["stock"] === null) {
                    $product->setStockData([
                        "use_config_manage_stock" => 0,
                        "manage_stock" => 0,
                        "use_config_backorders" => 0,
                        "backorders" => Mage_CatalogInventory_Model_Stock::BACKORDERS_NO,
                        "is_in_stock" => 1,
                        "qty" => 999999
                    ]);
                } else {
                    $product->setStockData([
                        "use_config_manage_stock" => 0,
                        "manage_stock" => 1,
                        "use_config_backorders" => 0,
                        "backorders" => $parsedParams["stock"]["backorders"] ? self::$backorderDefault : Mage_CatalogInventory_Model_Stock::BACKORDERS_NO,
                        "is_in_stock" => $parsedParams["stock"]["qty"] > 0 ? 1 : 0,
                        "qty" => $parsedParams["stock"]["qty"]
                    ]);
                }
            } else {
                $stockItem = $product->getStockItem();
                if ($parsedParams["stock"] === null && $stockItem->getData("manage_stock") > 0) {
                    $stockItem->setData("use_config_manage_stock", 1);
                    $stockItem->setData("manage_stock", 0);
                    $stockItem->setData("use_config_backorders", 1);
                    $stockItem->setData("backorders", Mage_CatalogInventory_Model_Stock::BACKORDERS_NO);
                    $stockItem->setData("is_in_stock", 1);
                    $stockItem->setData("qty", 999999);
                    $this->isProductUpdated = true;
                } elseif (!empty($parsedParams["stock"])) {
                    if ($stockItem->getData('use_config_manage_stock') != 0) {
                        $stockItem->setData('use_config_manage_stock', 0);
                        $this->isProductUpdated = true;
                    }
                    if ($stockItem->getData("manage_stock") != 1) {
                        $stockItem->setData("manage_stock", 1);
                        $this->isProductUpdated = true;
                    }
                    if ($stockItem->getData("use_config_backorders") != 0) {
                        $stockItem->setData("use_config_backorders", 0);
                        $this->isProductUpdated = true;
                    }
                    $backorders = $parsedParams["stock"]["backorders"] ? self::$backorderDefault : Mage_CatalogInventory_Model_Stock::BACKORDERS_NO;
                    if ($stockItem->getData("backorders") != $backorders) {
                        $stockItem->setData("backorders", $backorders);
                        $this->isProductUpdated = true;
                    }
                    if ($stockItem->getData("qty") != $parsedParams["stock"]["qty"]) {
                        $stockItem->setData("qty", $parsedParams["stock"]["qty"]);
                        $stockItem->setData("is_in_stock", $parsedParams["stock"]["qty"] > 0 ? 1 : 0);
                        $this->isProductUpdated = true;
                    }
                }
                if ($this->isProductUpdated) {
                    $stockItem->save();
                }
            }
        }

        // Only set these attributes if new product or not in batch mode
        if ($this->isCreate || !$parsedParams["isBatch"]) {

            // // Only add images to Awardit products
            if ($isAwarditProduct && array_key_exists("media", $parsedParams)) {
                $this->setImages($product, $parsedParams["media"]);
            }
        }

    }

    public function updateLocalProduct(Mage_Catalog_Model_Product $product, $params)
    {
        if (array_key_exists("name", $params) && $product->getName() != $params["name"]) {
            $product->setData("name", $params["name"])->getResource()->saveAttribute($product, "name");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("shortDescription", $params) && $product->getShortDescription() != $params["shortDescription"]) {
            $product->setData("short_description", $params["shortDescription"])->getResource()->saveAttribute($product, "short_description");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("longDescription", $params) && $product->getDescription() != $params["longDescription"]) {
            $product->setData("description", $params["longDescription"])->getResource()->saveAttribute($product, "description");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("isFeatured", $params) && intval($product->getAwarditIsFeatured()) !== $params["isFeatured"]) {
            $product->setData("awardit_is_featured", $params["isFeatured"])->getResource()->saveAttribute($product, "awardit_is_featured");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("partnerOwned", $params) && intval($product->getAwarditPartnerOwned()) !== $params["partnerOwned"]) {
            $product->setData("awardit_partner_owned", $params["partnerOwned"])->getResource()->saveAttribute($product, "awardit_partner_owned");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("validFrom", $params) && $product->getAwarditValidFrom() !== $params["validFrom"]) {
            $product->setData("awardit_valid_from", $params["validFrom"])->getResource()->saveAttribute($product, "awardit_valid_from");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("validTo", $params) && $product->getAwarditValidTo() !== $params["validTo"]) {
            $product->setData("awardit_valid_to", $params["validTo"])->getResource()->saveAttribute($product, "awardit_valid_to");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("invoiceName", $params) && $product->getAwarditInvoiceName() !== $params["invoiceName"]) {
            $product->setData("awardit_invoice_name", $params["invoiceName"])->getResource()->saveAttribute($product, "awardit_invoice_name");
            $this->isProductUpdated = true;
        }

        if (array_key_exists("manufacturerId", $params) && !empty($params["manufacturerId"]) && $product->getManufacturer() != $params["manufacturerId"]) {
            $product->setData("manufacturer", $params["manufacturerId"])->getResource()->saveAttribute($product, "manufacturer");
            $this->isProductUpdated = true;
        }

        if (in_array($product->getTypeId(), [ Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL ])) {
            // Only set price on actual products
            if ($this->setPriceAttributes($product, $params)) {
                $this->isProductUpdated = true;
            }
        } else {
            // Get children and set prices on them
            $children = Mage::getModel("catalog/product_type_configurable")->getUsedProducts(null, $product);
            foreach ($children as $child) {
                $realChild = Mage::getModel("catalog/product")->setStoreId($this->getLocalStore()->getStoreId())->load($child->getId());
                if ($this->setPriceAttributes($realChild, $params)) {
                    $realChild->save();
                    $this->isProductUpdated = true;
                }
            }
        }

        // Check if we got prices
        if (array_key_exists("splitPaymentPrice", $params) && array_key_exists("invoicePrice", $params)) {

            // Check if product is awaiting price
            if ($product->getAwarditAwaitingPrice() == Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES) {

                // If we got prices and were waiting for it, set flag to not be waiting any longer
                $product->setData("awardit_awaiting_price", Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO)->getResource()->saveAttribute($product, "awardit_awaiting_price");
                $this->isProductUpdated = true;
            }
        }

        if (array_key_exists("status", $params)) {
            if ($params["status"] == Mage_Catalog_Model_Product_Status::STATUS_DISABLED || !$this->isWithinDates($product->getAwarditValidFrom(), $product->getAwarditValidTo())) {
                if ($product->getStatus() != Mage_Catalog_Model_Product_Status::STATUS_DISABLED) {
                    $product->setData("status", Mage_Catalog_Model_Product_Status::STATUS_DISABLED)->getResource()->saveAttribute($product, "status");
                    $this->isProductUpdated = true;
                }
            } else {
                if ($product->getStatus() != Mage_Catalog_Model_Product_Status::STATUS_ENABLED) {
                    $product->setData("status", Mage_Catalog_Model_Product_Status::STATUS_ENABLED)->getResource()->saveAttribute($product, "status");
                    $this->isProductUpdated = true;
                }
            }
        }

        if ($this->isProductUpdated) {
            $product->save();
            $this->isProductUpdated = false;
        }

    }

    public function setCategories(Mage_Catalog_Model_Product $product, $categories, Mage_Core_Model_Store $store)
    {
        // Check categories parameter
        $rootCategoryId = $this->getRootCategoryId($store->getStoreId());
        if (empty($rootCategoryId)) {
            Mage::throwException("Missing root category for store");
        }
        $sqlQuery = "
            SELECT
                ccev.`value` AS category_name,
                tree.entity_id AS category_id,
                ccev.store_id,
                ccp.`position`
            FROM catalog_category_entity_varchar ccev
            JOIN eav_attribute eava ON eava.attribute_id = ccev.attribute_id
            JOIN (
                SELECT
                    entity_id
                FROM
                    (SELECT * FROM catalog_category_entity cce ORDER BY cce.parent_id, entity_id) items_sorted,
                    (SELECT @iv := :rootCat) initialisation
                WHERE
                    find_in_set(parent_id, @iv)
                    AND length(@iv := concat(@iv, ',', entity_id))
            ) tree ON tree.entity_id = ccev.entity_id
            LEFT JOIN catalog_category_product ccp ON ccp.category_id = tree.entity_id AND ccp.product_id = :productId
            WHERE
                eava.attribute_code = 'name'
                AND (ccev.store_id = 0 OR ccev.store_id = :storeId)
        ";
        $params = [
            "rootCat" => $rootCategoryId,
            "storeId" => $store->getStoreId(),
            "productId" => $this->isCreate ? 0 : $product->getId()
        ];
        $categoryTree = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $params);

        $newCategories = [];
        $parentCategory = Mage::getModel('catalog/category')->load($rootCategoryId);
        foreach ($categories as $categoryName) {
            $foundIt = false;
            foreach ($categoryTree as $cat) {
                // Magento database is UTF8, API data is specified as UTF8, let's hope everyone follows the spec.
                if (strcasecmp($categoryName, $cat["category_name"]) === 0) {
                    $foundIt = true;
                    // Category name matches
                    if (empty($cat["position"])) {
                        // Product did not previously belong to category
                        $newCategories[] = $cat["category_id"];
                    }
                }
            }
            if (!$foundIt) {
                $category = Mage::getModel("catalog/category");
                $category->setName($categoryName);
                //$category->setUrlKey($categoryName); // If we skip this, Magento creates one automatically
                $category->setIsActive(1);
                $category->setDisplayMode(Mage_Catalog_Model_Category::DM_PRODUCT);
                $category->setIsAnchor(0);
                $category->setPath($parentCategory->getPath());
                $category->save();

                // Product is supposed to belong to new category
                $newCategories[] = $category->getId();

                // Make sure nothing survives if we need to create another category later
                unset($category);
            }
        }

        // Add product to categories
        if (!empty($newCategories)) {
            $previousCategories = $this->isCreate ? [] : $product->getCategoryIds();
            $categoryIds = array_unique(array_merge($previousCategories, $newCategories), SORT_NUMERIC);
            $product->setCategoryIds($categoryIds);
            return true;
        }

        return false;
    }

    public function setImages(Mage_Catalog_Model_Product $product, $media)
    {
        $imageWasAdded = false;
        $hashes = [];
        $awarditImageHash = $product->getAwarditImageHash();
        if (!empty($awarditImageHash)) {
            $hashes = explode(",", $awarditImageHash);
            if (!is_array($hashes)) {
                $hashes = [$hashes];
            }
        }

        // Only one image is sent, but there is potential to send more
        foreach ($media as $i => $image) {

            // Extract filename from url
            $filenameParts = explode("/", $image["url"]);
            $filename = $filenameParts[count($filenameParts) - 1];

            // Build full path and filename
            $filePathAndName = Mage::getBaseDir("media") . DS . "import" . DS . $filename;

            if (stripos($image["url"], "http") === 0) {
                // If url starts with http, consider it a full url
                $realUrl = $image["url"];
            } else {
                // If not, url is relative to configured url
                $storeId = $this->getLocalStore()->getId();
                $realUrl = Mage::getStoreConfig("awardit_points/server/host", $storeId) . (stripos($image["url"], "/") === 0 ? $image["url"] : "/" . $image["url"]);
            }

            $realUrl = str_replace(" ", "%20", $realUrl);

            // Only add images not previously known
            $hash = md5($realUrl);
            if (in_array($hash, $hashes)) {
                Mage::log("Skipping known image: {$realUrl}", LOG_DEBUG, "api-debug.log", true);
                continue;
            }

            // Add new hash to list of old ones
            $hashes[] = $hash;

            // If not yet updated, this must be first time. Prepare image gallery before adding new image
            if (!$imageWasAdded) {
                $this->resetImageGalleryToUseDefault($product->getId());
                $this->setProductInStoreToUseNoImages(0, $product->getId());
                $this->removeMediaGalleryImages($product);
            }

            // Read media file
            Mage::log("Loading media file: {$realUrl}", LOG_DEBUG, "api-debug.log", true);
            $rawData = @file_get_contents($realUrl);
            if ($rawData === false) {
                Mage::throwException("Unable to load media file: 'url' in 'media[{$i}]");
            }

            // Save media file
            $bytes = file_put_contents($filePathAndName, $rawData);
            if ($bytes === false) {
                Mage::throwException("Unable to create media file: '{$filename}'");
            }

            // Set media tags for image
            if ($image["primary"]) {
                $tags = [
                    0 => "image",
                    1 => "small_image",
                    2 => "thumbnail"
                ];
            } else {
                $tags = [];
            }

            // Add image to product
            $product->addImageToMediaGallery($filePathAndName, $tags, true, false);

            // Set attributes for last added image
            $gallery = $product->getData('media_gallery');
            $lastImage = array_pop($gallery['images']);
            $lastImage['label'] = "";
            $lastImage['position'] = 0;
            $lastImage['disabled'] = 0;
            $lastImage['label_default'] = "";
            $lastImage['position_default'] = 0;
            $lastImage['disabled_default'] = 0;
            array_push($gallery['images'], $lastImage);
            $product->setData('media_gallery', $gallery);

            $this->mediaGalleryUpdated = true;
            $this->isProductUpdated = true;
            $imageWasAdded = true;
            unset($rawData);

        }

        // If updated, it means we added at least one image. Save awardit_image_hash
        if ($imageWasAdded) {
            $product->setAwarditImageHash(implode(",", $hashes));
        }

    }

    public function removeMediaGalleryImages(Mage_Catalog_Model_Product $product)
    {
        // stackoverflow.com/questions/5709496/magento-programmatically-remove-product-images

        $mediaGalleryData = $product->getMediaGallery();
        if (!isset($mediaGalleryData['images']) || !is_array($mediaGalleryData['images'])) {
            return;
        }

        $toDelete = [];
        foreach ($mediaGalleryData['images'] as $image) {
            $toDelete[] = $image['value_id'];
            @unlink(Mage::getBaseDir('media') . DS . 'catalog' . DS . 'product' . $image['file']);
        }
        if (!empty($toDelete)) {
            Mage::getResourceModel('catalog/product_attribute_backend_media')->deleteGallery($toDelete);
            $product->setMediaGallery([ "images" => [], "values" => [] ]);
        }

    }

    public function setCustomOptions(Mage_Catalog_Model_Product &$product, $options)
    {
        $oldOptionsHash = $product->getAwarditOptionsHash();
        $newOptionsHash = md5(json_encode($options));

        if (!empty($oldOptionsHash) && $newOptionsHash == $oldOptionsHash) {
            Mage::log("Skipping known custom options, hash is same.", LOG_DEBUG, "api-debug.log", true);
            return;
        }

        $productId = $product->getId();
        $this->removeCustomOptionsForProduct($productId);
        Mage::log("Removed old custom options", LOG_DEBUG, "api-debug.log", true);

        $product = null;
        $product = Mage::getModel('catalog/product')->setStoreId(0)->load($productId);

        if (!empty($options)) {
            // Add new custom options
            $product->setCanSaveCustomOptions(true);
            $product->setHasOptions(true);
            $product->setProductOptions($options);
            $product->setAwarditOptionsHash($newOptionsHash);
            Mage::log("Added custom options: " . json_encode($options), LOG_DEBUG, "api-debug.log", true);

        } else {
            // Remove custom options hash
            $product->setHasOptions(false);
            $product->setRequiredOptions(false);
            $product->setAwarditOptionsHash("");
            $product->save();
        }
        $product->save();

    }

    public function getRootCategoryId($storeId)
    {
        $sqlQuery = "SELECT csg.root_category_id FROM core_store cs JOIN core_store_group csg ON csg.website_id = cs.website_id WHERE cs.store_id = :id";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, ["id" => $storeId]) ?: null;
    }

    public function getTaxClassId(Mage_Core_Model_Store $store, $percent)
    {
        $sqlQuery = "
            SELECT
                tc.product_tax_class_id
            FROM tax_calculation_rate tcr
            JOIN tax_calculation tc ON tc.tax_calculation_rate_id = tcr.tax_calculation_rate_id
            WHERE
                tcr.rate = :percent
                AND tcr.tax_country_id IN (
                    SELECT
                        COALESCE(
                            if(ccd.scope = 'stores', ccd.value, NULL),
                            if(ccd.scope = 'websites', ccd.value, NULL),
                            if(ccd.scope = 'default', ccd.value, NULL)
                        ) AS country_code
                    FROM core_config_data ccd
                    WHERE ccd.path = 'general/country/default' AND (ccd.scope_id = 0 OR ccd.scope_id = :websiteId OR ccd.scope_id = :storeId)
                    GROUP BY 1
                )
        ";
        $params = [
            "storeId" => $store->getStoreId(),
            "websiteId" => $store->getWebsiteId(),
            "percent" => floatval($percent)
        ];
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params) ?: 0;
    }

    public function setPriceAttributes(&$product, $params)
    {
        $wasUpdated = false;
        $storeId = $this->getLocalStore()->getId();

        if (array_key_exists("splitPaymentPrice", $params)) {
            if (floatval($product->getPrice()) != $params["splitPaymentPrice"]) {
                $product->setData("price", $params["splitPaymentPrice"])->getResource()->saveAttribute($product, "price");
                $wasUpdated = true;
            }
        }

        if (array_key_exists("invoicePrice", $params)) {
            $sqlQuery = "
                INSERT INTO awardit_upsert_product (`product_id`, `store_id`, `invoice_price`) VALUES (:pId, :sId, :price)
                ON DUPLICATE KEY UPDATE `invoice_price` = VALUES(invoice_price), `updated_at` = NOW()
            ";
            $queryParams = [
                "price" => $params["invoicePrice"],
                "pId" => $product->getId(),
                "sId" => $storeId
            ];
            Mage::getSingleton("core/resource")->getConnection("core_write")->query($sqlQuery, $queryParams);
            $wasUpdated = true;
        }

        if (array_key_exists("msrp", $params) && floatval($product->getMsrp()) != $params["msrp"]) {
            $product->setData("msrp", $params["msrp"])->getResource()->saveAttribute($product, "msrp");
            $wasUpdated = true;
        }

        if (array_key_exists("taxClassId", $params) && floatval($product->getTaxClassId()) != $params["taxClassId"]) {
            $product->setData("tax_class_id", $params["taxClassId"])->getResource()->saveAttribute($product, "tax_class_id");
            $wasUpdated = true;
        }

        if (array_key_exists("points", $params)) {
            $this->setPointsCoreData($params, $product, $storeId);
            $wasUpdated = true;
        }

        return $wasUpdated;

    }

    public function getJsonBody($request)
    {
        $rawBody = $request->getRawBody();
        if (!empty($rawBody)) {
            $content_type = $request->getHeader("Content-Type");
            if (stripos("application/json", trim($content_type)) !== null) {
                $data = json_decode($rawBody, true);

                if ($data === null) {
                    Mage::throwException("Unable to decode parameters");
                }

                return $data;
            }
        }

        return null;
    }

    public function isAwarditProduct(Mage_Catalog_Model_Product $product)
    {
        // Find out if product originates from Awardit or CLS
        return stripos($product->getSku(), "test-") === 0 || stripos($product->getSku(), "awd_") === 0;
    }

    public function isAuthorized($key)
    {
        $sqlQuery = "SELECT token FROM crossroads_awardit_token WHERE company_id = :companyId AND deleted_at IS NULL AND token = :token";
        $params = [
            "companyId" => self::AUTHORIZATION_COMPANY_ID,
            "token" => $key
        ];
        $token = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);
        return !empty($token);
    }

    public function isWithinDates($dateFrom, $dateTo, $dateNow = null)
    {
        // If no dates are given, it's considered to be within dates
        if (empty($dateFrom) && empty($dateTo)) {
            return true;
        }

        $now = new DateTime( empty($dateNow) ? date("Y-m-d") : $dateNow );
        $from = new DateTime( empty($dateFrom) ? $now->format("Y-m-d") : $dateFrom );
        $to = new DateTime( empty($dateTo) ? $now->format("Y-m-d") : $dateTo );

        // if now == from, it's within dates
        // if now == to, it's within dates
        // if now > from AND now < to, it's within dates
        return $now >= $from && $now <= $to;

    }

    public function setPointsCoreData($data, $product, $storeId, $customerGroupId = 0)
    {
        if (!array_key_exists("points", $data)) {
            // Points must be supplied, just return and do nothing
            return null;
        }

        $pointsCoreId = Mage::helper("integration/PointsCore")->getPointsCoreId($storeId);
        if (empty($pointsCoreId)) {
            Mage::throwException("This store does not have pointsCoreId");
        }

        $isPriceUpdated = false;

        $price = Mage::getModel("points_core/product_price");
        $price->loadByStoreProductTypeCustomerGroupId(
            $this->getLocalStore(),
            $product,
            $pointsCoreId,
            $customerGroupId
        );

        if ($data["points"] > 0) {
            $newPoints = [
                "price" => $data["points"],
                "min_price" => 0,
                "max_price" => null,
            ];
        } else {
            if ($price->getId() > 0) {
                Mage::Helper("awardit_upsert")->log("Removing points core data for [{$product->getSku()}] in store {$storeId}", LOG_DEBUG, "api-debug.log");
                $price->delete();
            }
            return null;
        }

        if (array_key_exists("minPoints", $data) && $data["minPoints"] > 0) {
            $newPoints["min_price"] = $data["minPoints"];
        }

        if (array_key_exists("maxPoints", $data) && $data["maxPoints"] > 0) {
            $newPoints["max_price"] = $data["maxPoints"];
        }

        if (!$price->getId()) {
            $isPriceUpdated = true;
            $price->setProductId($product->getId());
            $price->setStoreId($storeId);
            $price->setCustomerGroupId($customerGroupId);
            $price->setType($pointsCoreId);
            Mage::Helper("awardit_upsert")->log("Creating points core data for [{$product->getSku()}] in store {$storeId}", LOG_DEBUG, "api-debug.log");
        }

        foreach ($newPoints as $key => $val) {
            if ($val != $price->getData($key)) {
                $price->setData($key, $val);
                $isPriceUpdated = true;
            }
        }

        if ($isPriceUpdated) {
            $price->save();
        }

        return $price->getId();
    }

}
