<?php

class Awardit_Upsert_Model_Resource_Product extends Mage_Core_Model_Resource_Db_Abstract
{
    protected function _construct(): void
    {
        $this->_init("awardit_upsert/product", "index_id");
    }

    /**
     * @return int
     */
    public function getIdByProductIdAndStoreId($productId, $storeId)
    {
        $sqlQuery = "SELECT `index_id` FROM `awardit_upsert_product` WHERE `product_id` = :productId AND `store_id` = :storeId";
        return $this->_getWriteAdapter()->fetchOne($sqlQuery, [ "productId" => $productId, "storeId" => $storeId ]);
    }

    public function loadByProductIdAndStoreId($productId, $storeId): self
    {
        $indexId = $this->getIdByProductIdAndStoreId($productId, $storeId);

        if (!empty($indexId)) {
            return $this->load($indexId);
        }

        return $this;
    }

    public function reindexAll(): void {
        foreach (Mage::app()->getWebsites() as $website) {
            foreach ($website->getGroups() as $group) {
                $stores = $group->getStores();

                foreach ($stores as $store) {
                    $this->indexSingleStore($store);
                }
            }
        }
    }

    public function indexSingleStore(Mage_Core_Model_Store $store): void
    {
        $conn = $this->_getWriteAdapter();
        $storeId = $store->getId();
        $appEmulation = Mage::getSingleton("core/app_emulation");
        $initialEnvironmentInfo = null;

        try {
            $this->beginTransaction();

            $sqlQuery1 = "UPDATE awardit_upsert_product SET product_data = NULL, updated_at = NULL WHERE store_id = :storeId";
            $stmt1 = $conn->prepare($sqlQuery1);

            $sqlQuery2 = "
                INSERT INTO awardit_upsert_product (`product_id`, `store_id`, `product_data`) VALUES (:productId, :storeId, :productData)
                ON DUPLICATE KEY UPDATE `product_data` = VALUES(product_data), `updated_at` = NOW()
            ";
            $stmt2 = $conn->prepare($sqlQuery2);

            $mainQuery = "
                SELECT
                    t.*
                FROM (
                    SELECT
                        cpe.entity_id,
                        (
                            SELECT
                                COALESCE(
                                    MAX(IF(catalog_product_entity_int.store_id = cs.store_id, catalog_product_entity_int.`value`, NULL)),
                                    MAX(IF(catalog_product_entity_int.store_id = 0, catalog_product_entity_int.`value`, NULL))
                                )
                            FROM catalog_product_entity_int
                            JOIN eav_attribute ON eav_attribute.attribute_id = catalog_product_entity_int.attribute_id
                            WHERE catalog_product_entity_int.entity_id = cpe.entity_id AND eav_attribute.attribute_code = 'status' AND (catalog_product_entity_int.store_id = 0 OR catalog_product_entity_int.store_id = cs.store_id)
                            GROUP BY catalog_product_entity_int.entity_id
                        ) AS product_status,
                        (
                            SELECT
                                COALESCE(
                                    MAX(IF(catalog_product_entity_int.store_id = cs.store_id, catalog_product_entity_int.`value`, NULL)),
                                    MAX(IF(catalog_product_entity_int.store_id = 0, catalog_product_entity_int.`value`, NULL))
                                )
                            FROM catalog_product_entity_int
                            JOIN eav_attribute ON eav_attribute.attribute_id = catalog_product_entity_int.attribute_id
                            WHERE catalog_product_entity_int.entity_id = cpe.entity_id AND eav_attribute.attribute_code = 'awardit_awaiting_price' AND (catalog_product_entity_int.store_id = 0 OR catalog_product_entity_int.store_id = cs.store_id)
                            GROUP BY catalog_product_entity_int.entity_id
                        ) AS awardit_awaiting_price
                    FROM catalog_product_entity cpe
                    JOIN catalog_product_website cpw ON cpw.product_id = cpe.entity_id
                    JOIN core_store cs ON cs.website_id = cpw.website_id
                    WHERE cs.store_id = %d AND cpe.type_id IN ('simple','virtual')
                ) t
                WHERE t.product_status = %d OR (t.product_status = %d AND t.awardit_awaiting_price = %d)
            ";

            $stmt1->bindValue("storeId", $storeId);
            $stmt1->execute();

            $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);

            $factory = Mage::getModel("API/factory");
            $nullSer = Mage::getModel("API/serializer_constant", [ "value" => null ]);
            $prodSer = $factory->createProductDetailSerializer($store)
                ->setCustomOptionsSerializer($nullSer)
                ->setRelatedSerializer($nullSer);

            $sqlQuery = sprintf(
                $mainQuery,
                intval($storeId),
                intval(Mage_Catalog_Model_Product_Status::STATUS_ENABLED),
                intval(Mage_Catalog_Model_Product_Status::STATUS_DISABLED),
                intval(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES)
            );
            $collection = $conn->fetchAll($sqlQuery);

            foreach ($collection as $product) {
                $this->updateProduct($product["entity_id"], $store, $prodSer, $stmt2);
            }

            $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);

            $initialEnvironmentInfo = null;

            $this->commit();
        }
        catch(Exception $e) {
            Mage::logException($e);

            $this->rollBack();

            throw $e;
        }
        finally {
            if ($initialEnvironmentInfo) {
                $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
            }
        }
    }

    public function indexSingleProduct(Mage_Catalog_Model_Product $globalProduct): void
    {
        $conn = $this->_getWriteAdapter();
        $productId = $globalProduct->getId();
        $currentWebsites = $globalProduct->getWebsiteIds();
        $appEmulation = Mage::getSingleton("core/app_emulation");
        $initialEnvironmentInfo = null;

        try {
            $this->beginTransaction();

            $sqlQuery1 = "UPDATE awardit_upsert_product SET product_data = NULL, updated_at = NULL WHERE product_id = :productId";
            $stmt1 = $conn->prepare($sqlQuery1);

            $sqlQuery2 = "
                INSERT INTO awardit_upsert_product (`product_id`, `store_id`, `product_data`) VALUES (:productId, :storeId, :productData)
                ON DUPLICATE KEY UPDATE `product_data` = VALUES(product_data), `updated_at` = NOW()
            ";
            $stmt2 = $conn->prepare($sqlQuery2);

            $stmt1->bindValue("productId", $productId);
            $stmt1->execute();

            foreach (Mage::app()->getWebsites() as $website) {
                if (!in_array($website->getId(), $currentWebsites)) {
                    continue;
                }

                foreach ($website->getGroups() as $group) {
                    $stores = $group->getStores();

                    foreach ($stores as $store) {
                        $storeId = $store->getId();
                        $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);

                        $factory = Mage::getModel("API/factory");
                        $nullSer = Mage::getModel("API/serializer_constant", [ "value" => null ]);
                        $prodSer = $factory->createProductDetailSerializer($store)
                            ->setCustomOptionsSerializer($nullSer)
                            ->setRelatedSerializer($nullSer);

                        $this->updateProduct($productId, $store, $prodSer, $stmt2);

                        $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);

                        $initialEnvironmentInfo = null;
                    }
                }
            }

            $this->commit();
        }
        catch(Exception $e) {
            Mage::logException($e);

            $this->rollBack();

            throw $e;
        }
        finally {
            if ($initialEnvironmentInfo) {
                $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
            }
        }
    }

    /**
     * @param int $productId
     */
    public function updateProduct(
        $productId,
        Mage_Core_Model_Store $store,
        Crossroads_API_Model_Serializer_Abstract $prodSer,
        Varien_Db_Statement_Pdo_Mysql $stmt
    ): void {
        $storeId = $store->getId();
        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);

        // Skip index if produckt is disabled and NOT waiting for price.
        if ($localProduct->getStatus() == Mage_Catalog_Model_Product_Status::STATUS_DISABLED && $localProduct->getAwarditAwaitingPrice() == Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO) {
            return;
        }

        $productData = $prodSer->serializeItem($localProduct);

        if (empty($productData)) {
            return;
        }

        if (!empty($productData["options"])) {
            $haveChildren = false;

            foreach ($productData["options"] as $option) {
                if (!empty($option["values"])) {
                    $haveChildren = true;

                    break;
                }
            }

            if (!$haveChildren) {
                $productData = [];

                return;
            }
        }

        if (array_key_exists("relatedProducts", $productData)) {
            unset($productData["relatedProducts"]);
        }

        $stmt->bindValue("productId", $productId);
        $stmt->bindValue("storeId", $storeId);
        $stmt->bindValue("productData", json_encode($productData));
        $stmt->execute();
    }

    public function reindexProductIds(array $ids): void {
        foreach($ids as $id) {
            $product = Mage::getModel("catalog/product");

            $product->load($id);

            $this->indexSingleProduct($product);
        }
    }

    /**
     * @param int $childId
     */
    public function getProductParentsByChild($childId): array {
        $write = $this->_getWriteAdapter();
        $select = $write->select()
            ->from(["l" => $this->getTable("catalog/product_relation")], ["parent_id"])
            ->join(
                ["e" => $this->getTable("catalog/product")],
                "l.parent_id = e.entity_id",
                ["e.type_id"]
            )
            ->where("l.child_id = ?", $childId);

        return $write->fetchPairs($select);
    }
}
