<?php

declare(strict_types=1);

/**
 * @method int getIndexId()
 * @method void setIndexId(int $value)
 * @method int getProductId()
 * @method void setProductId(int $value)
 * @method int getStoreId()
 * @method void setStoreId(int $value)
 * @method float getInvoicePrice()
 * @method void setInvoicePrice(float $value)
 * @method float getPurchasePrice()
 * @method void setPurchasePrice(float $value)
 * @method string getProductData()
 * @method void setProductData(string $value)
 * @method string getCreatedAt()
 * @method string getUpdatedAt()
 */
class Awardit_Upsert_Model_Product extends Mage_Index_Model_Indexer_Abstract {
    /**
     * Data key for matching result to be saved in
     */
    const EVENT_MATCH_RESULT_KEY = "awardit_upsert_product_match_result";

    const KEY_REINDEX_PRODUCT_IDS = "awardit_upsert_product_reindex_parent_ids";
    const KEY_REINDEX_ALL = "awardit_upsert_product_reindex_all";

    protected $_matchedEntities = [
        Mage_Catalog_Model_Product::ENTITY => [
            Mage_Index_Model_Event::TYPE_DELETE,
            Mage_Index_Model_Event::TYPE_SAVE,
            Mage_Index_Model_Event::TYPE_MASS_ACTION,
        ],
        Mage_Core_Model_Config_Data::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE,
        ],
    ];

    const RELATED_CONFIG_SETTINGS = [
        Mage_Catalog_Helper_Data::XML_PATH_PRICE_SCOPE,
        Mage_CatalogInventory_Model_Stock_Item::XML_PATH_MANAGE_STOCK
    ];

    public function _construct(): void {
        parent::_construct();

        $this->_init("awardit_upsert/product");
    }

    public function getName() {
        return Mage::helper("awardit_upsert")->__("Upsert Index");
    }

    public function getDescription() {
        return Mage::helper("awardit_upsert")->__("Index product upsert");
    }

    public function matchEvent(Mage_Index_Model_Event $event) {
        $data = $event->getNewData();

        if(isset($data[self::EVENT_MATCH_RESULT_KEY])) {
            return $data[self::EVENT_MATCH_RESULT_KEY];
        }

        if ($event->getEntity() == Mage_Core_Model_Config_Data::ENTITY) {
            /**
             * @var ?Varien_Object
             */
            $data = $event->getDataObject();

            if($data && in_array($data->getPath(), self::RELATED_CONFIG_SETTINGS)) {
                $result = $data->isValueChanged();
            }
            else {
                $result = false;
            }
        }
        else {
            $result = parent::matchEvent($event);
        }

        $event->addNewData(self::EVENT_MATCH_RESULT_KEY, $result);

        return $result;
    }

    /**
     * @return void
     */
    protected function _registerEvent(Mage_Index_Model_Event $event) {
        $event->addNewData(self::EVENT_MATCH_RESULT_KEY, true);

        $entity = $event->getEntity();

        switch($entity) {
        case Mage_Core_Model_Config_Data::ENTITY:
            $process = $event->getProcess();

            $process->changeStatus(Mage_Index_Model_Process::STATUS_REQUIRE_REINDEX);

            break;

        case Mage_Catalog_Model_Product::ENTITY:
            switch($event->getType()) {
                case Mage_Index_Model_Event::TYPE_DELETE:
                    // Reindex parent product
                    /**
                     * @var Mage_Catalog_Model_Product $product
                     */
                    $product = $event->getDataObject();
                    /**
                     * @var Awardit_Upsert_Model_Resource_Product $resource
                     */
                    $resource = $this->_getResource();

                    $parentIds = $resource->getProductParentsByChild($product->getId());

                    if($parentIds) {
                        // Maybe just use products
                        $event->addNewData(self::KEY_REINDEX_PRODUCT_IDS, $parentIds);
                    }

                    break;

                case Mage_Index_Model_Event::TYPE_SAVE:
                    /* @var Mage_Catalog_Model_Product $product */
                    $product = $event->getDataObject();

                    $event->addNewData(self::KEY_REINDEX_PRODUCT_IDS, [$product->getId()]);

                    break;

                case Mage_Index_Model_Event::TYPE_MASS_ACTION:
                    /* @var Varien_Object $actionObject */
                    $actionObject = $event->getDataObject();

                    $event->addNewData(self::KEY_REINDEX_PRODUCT_IDS, $actionObject->getProductIds());

                    break;
            }

            break;
        }
    }

    /**
     * @return mixed
     */
    protected function _processEvent(Mage_Index_Model_Event $event) {
        $data = $event->getNewData();

        if( ! empty($data[self::KEY_REINDEX_ALL])) {
            return $this->reindexAll();
        }

        if( ! empty($data[self::KEY_REINDEX_PRODUCT_IDS])) {
            /**
             * @var Awardit_Upsert_Model_Resource_Product $resource
             */
            $resource = $this->_getResource();

            return $resource->reindexProductIds($data[self::KEY_REINDEX_PRODUCT_IDS]);
        }
    }
}
