<?php

use UnzerSDK\Unzer;
use UnzerSDK\Constants\WebhookEvents;
use UnzerSDK\Interfaces\DebugHandlerInterface;

/**
 * Helper class for Unzer.
 */
class Awardit_Unzer_Helper_Data extends Mage_Core_Helper_Abstract implements DebugHandlerInterface
{
    /** @const string */
    private const LOG_CHANNEL = 'unzer.log';

    /**
     * Get Unzer instance, production or sandbox, optionally with debug logging.
     * @return UnzerSDK\Unzer
     */
    public function unzer(): Unzer
    {
        $key = $this->getPrivateKey();
        $locale = Mage::getStoreConfig('payment/unzer_config/locale') ?: 'en-US';

        // Throws exception on failure
        $unzer = new Unzer($key, $locale);

        if (Mage::getStoreConfig('payment/unzer_config/debug')) {
            $unzer->setDebugHandler($this);
            $unzer->setDebugMode(true);
        }

        return $unzer;
    }

    /**
     * Get public API key, used by frontend component.
     * @return string|null
     */
    public function getPublicKey(): ?string
    {
        $sandbox = (bool)Mage::getStoreConfig('payment/unzer_config/sandbox');
        $key_ref = $sandbox ? 'public_key_sandbox' : 'public_key_production';
        $key = Mage::getStoreConfig("payment/unzer_config/{$key_ref}");
        return $key ?: null;
    }

    /**
     * Get private API key, used by backend component.
     * @return string|null
     */
    public function getPrivateKey(): ?string
    {
        $sandbox = (bool)Mage::getStoreConfig('payment/unzer_config/sandbox');
        $key_ref = $sandbox ? 'private_key_sandbox' : 'private_key_production';
        $key = Mage::getStoreConfig("payment/unzer_config/{$key_ref}");
        return $key ?: null;
    }

    /**
     * Check if mandatory configs are set.
     * @return bool
     */
    public function isAvailable(): bool
    {
        return $this->getPublicKey()
            && $this->getPrivateKey()
            && $this->getReturnUrl();
    }

    /**
     * Get return URL.
     * @return string
     */
    public function getReturnUrl(): string
    {
        return Mage::getStoreConfig('payment/unzer_config/return_url');
    }

    /**
     * Write to log.
     * @param string $message Message to log.
     * @param string $level Any Zend_Log level constant.
     */
    public function log(string $message, string $level = Zend_Log::DEBUG): void
    {
        Mage::log($message, $level, self::LOG_CHANNEL);
    }

    /**
     * Setting up Unzer webhooks
     */
    public function setUpWebhooks(): void
    {
        $config_key = "payment/unzer_notif/{$this->getPublicKey()}";
        $uid = Mage::getStoreConfig($config_key);
        $url = Mage::getStoreConfig('payment/unzer_config/webhook_url');

        if ($uid == $url) {
            return; // No update needed
        }
        $unzer = $this->unzer();
        if ($uid) {
            $unzer->deleteAllWebhooks(); // Delete any existing
        }
        if ($url) {
            $unzer->createWebhook($url, WebhookEvents::ALL);
        }

        $config = new Mage_Core_Model_Config();
        $config->saveConfig($config_key, $url, 'default', 0);
        $this->log($url ? "Webhooks disabled." : "Webhooks at {$url}.");
    }
}
