<?php

use GraphQL\Type\Definition\ResolveInfo;
use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use UnzerSDK\Exceptions\UnzerApiException;

/**
 * GraphQL operation class for Unzer.
 */
class Awardit_Unzer_Model_Graphql_Schema extends MageQL_Core_Model_Schema_Abstract
{
    /** @var Awardit_Unzer_Helper_Data */
    protected $helper;

    /**
     * Constructor for class.
     */
    public function __construct()
    {
        $this->helper = Mage::helper('awardit_unzer');
    }

    /**
     * Get Type builder or null.
     * @param string $type
     * @param MageQL\Registry $registry
     * @return AbstractBuilder|null
     */
    public function getTypeBuilder(string $type, Registry $registry): ?AbstractBuilder
    {
        return null;
    }

    /**
     * Get Type fields array.
     * @param string $type
     * @param MageQL\Registry $registry
     * @return array
     */
    public function getTypeFields(string $type, Registry $registry): array
    {
        switch ($type) {
            case 'Mutation':
                return [
                    'placeUnzerOrder' => $this->field('PlaceOrderResult!', 'Attempts to place an Unzer order with the current session quote.')
                        ->addArgument('typeId', $this->argument('ID!', 'Unzer transaction'))
                        ->setResolver([$this, 'resolvePlaceUnzerOrder']),
                    'verifyUnzerOrder' => $this->field('PlaceOrderResult!', 'Verifies Unzer order state.')
                        ->setResolver([$this, 'resolveVerifyUnzerOrder']),
                ];
        }
        return [];
    }

    /**
     * Process Unzer method on Quote and return result.
     * @param Mage_Core_Model_Store $store Current store
     * @param array $args Must contain 'typeId' item
     * @returnMageQL_Sales_Model_Order_AbstractResult
     */
    public function resolvePlaceUnzerOrder(Mage_Core_Model_Store $store, array $args): MageQL_Sales_Model_Order_AbstractResult
    {
        $payment_helper = Mage::helper('payment');
        $quote = Mage::getSingleton('mageql_sales/quote')->getQuote();
        $method = $payment_helper->getMethodInstance($quote->getPayment()->getMethod());

        // Check that we got a valid Unzer method
        if (empty($method) || !$method instanceof Awardit_Unzer_Model_Method_Abstract) {
             return new MageQL_Sales_Model_Order_Error(MageQL_Sales_SubmitOrderException::NO_PAYMENT_METHOD);
        }

        // Check availability
        $availability = $method->getAvailability($quote, true);
        switch ($availability) {
            case Awardit_Unzer_Constant::AVAILABILITY_OK:
                break; // All is fine
            case Awardit_Unzer_Constant::AVAILABILITY_ERR_COUNTRY:
                return new MageQL_Sales_Model_Order_Error(MageQL_Sales_SubmitOrderException::BILLING_ADDRESS_VALIDATION_FAILED);
            default:
                return new MageQL_Sales_Model_Order_Error(MageQL_Sales_SubmitOrderException::INVALID_PAYMENT_METHOD);
        }

        try {
            // Call the charge/reserve implementation
            $method->prepare($quote, $args['typeId']);
            $result =  MageQL_Sales_Model_Quote::mutatePlaceOrder();
            $this->helper->log("Placed order using {$method->getCode()}");

        } catch (UnzerApiException $e) {
            $this->helper->log("Failed {$method->getCode()} (Unzer): {$e->getMerchantMessage()}", Zend_Log::ERR);
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_SubmitOrderException::INVALID_PAYMENT_METHOD);

        } catch (Throwable $e) {
            $this->helper->log("Failed {$method->getCode()} (Generic): {$e->getMessage()}", Zend_Log::ERR);
            throw $e;
        }

        // Success or Error subclass
        return $result;
    }

    /**
     * Verify placed Unzer order. Use when customer returns from redirect.
     * @param Mage_Core_Model_Store $store Current store
     * @param array $args
     * @return MageQL_Sales_Model_Order_AbstractResult
     */
    public function resolveVerifyUnzerOrder(Mage_Core_Model_Store $store, array $args): MageQL_Sales_Model_Order_AbstractResult
    {
        $session = Mage::getSingleton('checkout/session');
        $payment_helper = Mage::helper('payment');
        $order_id = $session->getLastOrderId();

        $order = Mage::getModel('sales/order');
        $order->load($order_id);

        if (!$order->getId()) {
            $this->helper->log("Failed verify on {$order_id}, no order", Zend_Log::ERR);
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_SubmitOrderException::NO_PAYMENT_METHOD);
        }

        $method = $payment_helper->getMethodInstance($order->getPayment()->getMethod());
        if (empty($method) || !$method instanceof Awardit_Unzer_Model_Method_Abstract) {
            $this->helper->log("Failed verify on {$order_id}, no method", Zend_Log::ERR);
            return new MageQL_Sales_Model_Order_Error(MageQL_Sales_SubmitOrderException::NO_PAYMENT_METHOD);
        }

        $txn_status = $method->verify($order);
        if ($txn_status == Awardit_Unzer_Constant::RESULT_SUCCESS) {
            $this->helper->log("Verified order using {$method->getCode()}");
            return new MageQL_Sales_Model_Order_Result($order);
        }

        $this->helper->log("Failed verify on {$order_id} as {$txn_status}", Zend_Log::ERR);
        return new MageQL_Sales_Model_Order_Error($txn_status);
    }
}
