<?php

use UnzerSDK\Constants\PaymentState;
use UnzerSDK\Resources\Payment;
use UnzerSDK\Resources\TransactionTypes\{
    AbstractTransactionType,
    Authorization,
    Charge
};

/**
 * Base payment method class for Unzer.
 */
abstract class Awardit_Unzer_Model_Method_Abstract extends Mage_Payment_Model_Method_Abstract
{
    protected $_isGateway = true;
    protected $_canAuthorize = false; // Overloaded in specific methods
    protected $_canCapture = true;
    protected $_canRefund = true;
    protected $_canVoid = true;
    protected $_canUseInternal = false;
    protected $_canUseCheckout = true;
    protected $_canUseForMultishipping = false;
    protected $_canFetchTransactionInfo = true;
    protected $_canManageRecurringProfiles = false;

    /** @var Awardit_Unzer_Helper_Data */
    protected $helper;

    /**
     * Constructor for class.
     */
    public function __construct()
    {
        $this->helper = Mage::helper('awardit_unzer');
    }


    /* ---------- Configuration & checks --------------------------------------------- */

    /**
     * Get config payment action url.
     * @return string
     */
    public function getConfigPaymentAction(): string
    {
        return $this->getConfigData('payment_action')
            ?: Mage_Payment_Model_Method_Abstract::ACTION_AUTHORIZE_CAPTURE;
    }

    /**
     * Check whether payment method can be used.
     * @param Mage_Sales_Model_Quote|null $quote
     * @return bool
     */
    public function isAvailable($quote = null): bool
    {
        $appl_only = (bool)Mage::getStoreConfig('payment/unzer_config/applicable_only');
        return $this->getAvailability($quote, $appl_only) == Awardit_Unzer_Constant::AVAILABILITY_OK;
    }

    /**
     * Availability worker method.
     * @param Mage_Sales_Model_Quote|null $quote
     * @param bool $require_quote Also check Quote validity
     * @return bool
     */
    public function getAvailability(Mage_Sales_Model_Quote $quote, bool $require_quote): int
    {
        if (!$this->helper->isAvailable()) {
            return Awardit_Unzer_Constant::AVAILABILITY_ERR_CONFIG;
        }
        if (!parent::isAvailable($quote)) {
            return Awardit_Unzer_Constant::AVAILABILITY_ERR_ACTIVE;
        }
        if (!$require_quote) {
            return Awardit_Unzer_Constant::AVAILABILITY_OK;
        }
        if (empty($quote->getId()) || empty($quote->getItemsCount())) {
            return Awardit_Unzer_Constant::AVAILABILITY_ERR_EMPTY;
        }
        if (!$this->canUseForCountry($quote->getBillingAddress()->getCountry())) {
            return Awardit_Unzer_Constant::AVAILABILITY_ERR_COUNTRY;
        }
        if (!$this->canUseForCurrency($quote->getQuoteCurrencyCode())) {
            return Awardit_Unzer_Constant::AVAILABILITY_ERR_CURRENCY;
        }
        if (!$this->canUseCheckout()) {
            return Awardit_Unzer_Constant::AVAILABILITY_ERR_CHECKOUT;
        }
        $total = $quote->getBaseSubtotal() + $quote->getShippingAddress()->getBaseShippingAmount();
        if ($total < 0.0001) {
            return Awardit_Unzer_Constant::AVAILABILITY_ERR_ZERO;
        }
        return Awardit_Unzer_Constant::AVAILABILITY_OK;
    }


    /* ---------- Public payment operation methods ----------------------------------- */

    /**
     * Prepare quote for checkout (stores typeId for future use).
     * @param Mage_Sales_Model_Quote $quote Quote to checkout
     * @param string $type_id Unzer id to use on further processing
     * @return $this
     */
    public function prepare(Mage_Sales_Model_Quote $quote, string $type_id): self
    {
        // Store typeId and some other data for later use
        $payment = $quote->getPayment();
        $payment->setAdditionalInformation('typeId', $type_id);
        $payment->setAdditionalInformation('method', $this->getCode());
        $payment->save();

        $this->helper->log("Prepare {$this->getCode()}/{$type_id} for quote/{$quote->getId()}.");
        return $this;
    }

    /**
     * Authorize payment method.
     * @param Varien_Object $payment
     * @param float $amount
     * @return $this
     */
    public function authorize(Varien_Object $mag_payment, $amount): self
    {
        parent::authorize($mag_payment, $amount); // Checks availability

        $unzer = $this->helper->unzer();
        $order = $this->getOrder($mag_payment);

        $this->helper->log("Authorization {$amount} {$order->getBaseCurrencyCode()} for order/{$order->getIncrementId()}.");

        // Perform Unzer authorize request
        $unz_authorization = new Authorization(
            $amount,
            $order->getBaseCurrencyCode(),
            $this->helper->getReturnUrl()
        );
        $unz_authorization->setOrderId($order->getIncrementId());
        $unz_authorization->setPaymentReference($order->getStore()->getFrontendName());
        $unz_transaction = $unzer->performAuthorization(
            $unz_authorization,
            $mag_payment->getAdditionalInformation('typeId')
        );

        // Handle result by Unzer transaction
        $this->handlePaymentByTransaction($mag_payment, $unz_transaction);
        $this->handlePaymentByPayment($mag_payment, $unz_transaction->getPayment());
        $order->setCanSendNewEmailFlag($unz_transaction->isSuccess());

        // Authorization complete, no further action required
        if ($unz_transaction->isSuccess()) {
            $this->helper->log("Authorization complete {$unz_transaction->getPaymentId()} for order/{$order->getIncrementId()}.");
        }

        // Authorization pending, redirect to external page to complete
        if ($unz_transaction->isPending()) {
            // Need to set redirect on quote-payment, so getOrderPlaceRedirectUrl() below can get them
            $quote_payment = $order->getQuote()->getPayment();
            $quote_payment->setAdditionalInformation('redirectUrl', $unz_transaction->getRedirectUrl());
            $quote_payment->save();

            $this->helper->log("Authorization pending {$unz_transaction->getPaymentId()} for order/{$order->getIncrementId()}.");
        }

        // Authorization failure
        if ($unz_transaction->isError()) {
            $this->helper->log("Authorization failed {$unz_transaction->getPaymentId()} for order/{$order->getIncrementId()}.");
            // @todo: Throw exceptio
        }

        // Make info visible in admin
        $mag_payment->setTransactionAdditionalInfo(Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS, null);
        $mag_payment->setTransactionAdditionalInfo(
            Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS,
            array_filter(array_merge(
                $mag_payment->getAdditionalInformation(),
                $mag_payment->getTransactionAdditionalInfo()
            ))
        );
        $mag_payment->save();

        return $this;
    }

    /**
     * Capture payment, either directly or on previously authorized payment.
     * @param Varien_Object $payment
     * @param float $amount
     * @return $this
     */
    public function capture(Varien_Object $mag_payment, $amount): self
    {
        parent::capture($mag_payment, $amount); // Checks availability

        $unzer = $this->helper->unzer();
        $order = $this->getOrder($mag_payment);
        $mag_transaction = $mag_payment->getAuthorizationTransaction();

        if ($mag_transaction ) {
            // Charge existing authorization
            $payment_id = $mag_payment->getAdditionalInformation('paymentId');
            if (empty($payment_id)) {
                return $this; // Nothing to update
            }
            $this->helper->log("Capture {$amount} {$order->getBaseCurrencyCode()} on {$payment_id} for order/{$order->getIncrementId()}.");
            $unz_transaction = $unzer->performChargeOnPayment($payment_id, new Charge());

        } else {
            // Customer direct charge
            $this->helper->log("Capture {$amount} {$order->getBaseCurrencyCode()} for order/{$order->getIncrementId()}.");
            $unz_charge = new Charge(
                $amount,
                $order->getBaseCurrencyCode(),
                $this->helper->getReturnUrl()
            );
            $unz_charge->setOrderId($order->getIncrementId());
            $unz_charge->setPaymentReference($order->getStore()->getFrontendName());
            $unz_transaction = $unzer->performCharge(
                $unz_charge,
                $mag_payment->getAdditionalInformation('typeId')
            );
        }

        // Handle result by Unzer transaction
        $this->handlePaymentByTransaction($mag_payment, $unz_transaction);
        $this->handlePaymentByPayment($mag_payment, $unz_transaction->getPayment());
        $order->setCanSendNewEmailFlag($unz_transaction->isSuccess());

        // Charge complete, no further action required
        if ($unz_transaction->isSuccess()) {
            $this->helper->log("Capture complete {$unz_transaction->getPaymentId()} for order/{$order->getIncrementId()}.");
        }

        // Capture pending, redirect to external page to complete
        if ($unz_transaction->isPending()) {
            // Need to set redirect on quote-payment, so getOrderPlaceRedirectUrl() below can get them
            $quote_payment = $order->getQuote()->getPayment();
            $quote_payment->setAdditionalInformation('redirectUrl', $unz_transaction->getRedirectUrl());
            $quote_payment->save();

            $this->helper->log("Capture pending {$unz_transaction->getPaymentId()} for order/{$order->getIncrementId()}.");
        }

        // Capture failure
        if ($unz_transaction->isError()) {
            $this->helper->log("Capture failed {$unz_transaction->getPaymentId()} for order/{$order->getIncrementId()}.");
            // @todo: Throw exception
        }

        // Make info visible in admin
        $mag_payment->setTransactionAdditionalInfo(Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS, null);
        $mag_payment->setTransactionAdditionalInfo(
            Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS,
            array_filter(array_merge(
                $mag_payment->getAdditionalInformation(),
                $mag_payment->getTransactionAdditionalInfo()
            ))
        );
        $mag_payment->save();

        return $this;
    }

    /**
     * Cancel authorization.
     * @param  Mage_Sales_Model_Order_Payment|Varien_Object $mag_payment
     * @return $this
     */
    public function cancel(Varien_Object $mag_payment): self
    {
        $mag_payment->setIsTransactionDenied(true);
        $mag_payment->setIsTransactionClosed(true);

        $unzer = $this->helper->unzer();
        $order = $this->getOrder($mag_payment);
        $mag_transaction = $mag_payment->getAuthorizationTransaction();

        if ($mag_transaction ) {
            // Cancel existing authorization
            $payment_id = $mag_payment->getAdditionalInformation('paymentId');
            if (empty($payment_id)) {
                return $this; // Nothing to update
            }
            $this->helper->log("Cancel {$payment_id} for order/{$order->getIncrementId()}.");
            $unz_authorization = $unzer->fetchAuthorization($payment_id);
            $unz_cancellation = $unz_authorization->cancel();

            $this->handlePaymentByTransaction($mag_payment, $unz_cancellation);
            $this->handlePaymentByPayment($mag_payment, $unz_cancellation->getPayment());
        }

        // Make info visible in admin
        $mag_payment->setTransactionAdditionalInfo(Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS, null);
        $mag_payment->setTransactionAdditionalInfo(
            Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS,
            array_filter(array_merge(
                $mag_payment->getAdditionalInformation(),
                $mag_payment->getTransactionAdditionalInfo()
            ))
        );
        $mag_payment->save();

        return $this;
    }

    /**
     * Void, same as cancel.
     * @param  Mage_Sales_Model_Order_Payment|Varien_Object $payment
     * @return $this
     */
    public function void(Varien_Object $mag_payment)
    {
        parent::void($mag_payment); // Checks availability
        return $this->cancel($mag_payment);
    }

    /**
     * Refund specified amount for payment
     * @param Varien_Object $payment
     * @param float $amount
     * @return $this
     */
    public function refund(Varien_Object $mag_payment, $amount)
    {
        parent::refund($mag_payment, $amount); // Checks availability

        $mag_payment->setIsTransactionDenied(true);
        $mag_payment->setIsTransactionClosed(true);

        $unzer = $this->helper->unzer();
        $order = $this->getOrder($mag_payment);
        $mag_transaction = $mag_payment->lookupTransaction(
            false,
            Mage_Sales_Model_Order_Payment_Transaction::TYPE_CAPTURE
        );

        if ($mag_transaction ) {
            // Cancel existing capture
            $payment_id = $mag_payment->getAdditionalInformation('paymentId');
            if (empty($payment_id)) {
                return $this; // Nothing to update
            }
            $this->helper->log("Refund {$payment_id} for order/{$order->getIncrementId()}.");
            $unz_charge = $unzer->fetchChargeById($payment_id, $mag_transaction->getTxnId());
            $unz_cancellation = $unz_charge->cancel();

            $this->handlePaymentByTransaction($mag_payment, $unz_cancellation);
            $this->handlePaymentByPayment($mag_payment, $unz_cancellation->getPayment());
        }

        // Make info visible in admin
        $mag_payment->setTransactionAdditionalInfo(Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS, null);
        $mag_payment->setTransactionAdditionalInfo(
            Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS,
            array_filter(array_merge(
                $mag_payment->getAdditionalInformation(),
                $mag_payment->getTransactionAdditionalInfo()
            ))
        );
        $mag_payment->save();

        return $this;
    }

    /**
     * Verify payment, called when customer returns from redirect.
     * @param Mage_Sales_Model_Order $order
     * @return string
     */
    public function verify(Mage_Sales_Model_Order $order): string
    {
        $mag_payment = $order->getPayment();
        $mag_transaction = $this->getOpenTransaction($mag_payment);
        if (empty($mag_transaction)) {
            return 'noTxn';
        }
        $this->updateTransaction($mag_transaction);
        return $mag_transaction->getAdditionalInformation('txStatus');
    }

    /**
     * Fetch transaction info from Unzer and update accordingly.
     * @param Mage_Payment_Model_Info $mag_payment
     * @param string $txn_id
     * @return array
     */
    public function fetchTransactionInfo(Mage_Payment_Model_Info $mag_payment, $txn_id): array
    {
        $mag_transaction = $mag_payment->getTransaction($txn_id);
        if (!$mag_transaction) {
            return []; // Nothing to update
        }

        // Update and resolve
        $mag_transaction->setAdditionalInformation(Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS, null);
        $this->updateTransaction($mag_transaction);
        $mag_transaction->setAdditionalInformation(Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS, null);

        return array_filter(array_merge(
            $mag_payment->getAdditionalInformation(),
            $mag_transaction->getAdditionalInformation()
        ));
    }

    /**
     * Get URL for browser redirect.
     * @return string
     */
    public function getOrderPlaceRedirectUrl(): string
    {
        // @todo: Is there a better way to do this? This is a mess ...
        $payment = $this->getInfoInstance()->getQuote()->getPayment();
        return $payment->getAdditionalInformation('redirectUrl') ?: '';
    }


    /* ---------- Private helper methods --------------------------------------------- */

    /**
     * Handle Unzer transaction and set to Magento payment/transaction.
     * Will prepare creation of new Magento transactions. To update existing, use below method.
     * @param Mage_Payment_Model_Info $mag_payment Magento payment
     * @param UnzerSDK\Resources\TransactionTypes\AbstractTransactionType $unz_transaction Unzer transaction
     */
    private function handlePaymentByTransaction(
        Mage_Payment_Model_Info $mag_payment,
        AbstractTransactionType $unz_transaction
    ): void
    {
        $mag_payment->setTransactionId($unz_transaction->getId());
        $mag_payment->setIsTransactionDenied($unz_transaction->isError());
        $mag_payment->setIsTransactionClosed($unz_transaction->isError());
        $mag_payment->setIsTransactionPending($unz_transaction->isPending());
        $mag_payment->setIsTransactionApproved($unz_transaction->isSuccess());
        $mag_payment->setTransactionAdditionalInfo('txStatus', $this->getTransactionStatus($unz_transaction));
        $mag_payment->setTransactionAdditionalInfo('txId', $unz_transaction->getId());
        $mag_payment->setAdditionalInformation('uniqueId', $unz_transaction->getUniqueId());
        $mag_payment->setAdditionalInformation('shortId', $unz_transaction->getShortId());
        $mag_payment->setAdditionalInformation('paymentId', $unz_transaction->getPaymentId());
        $mag_payment->setAdditionalInformation('redirectUrl', $unz_transaction->getRedirectUrl());
        $mag_payment->setAdditionalInformation('returnUrl', $this->helper->getReturnUrl());
    }

    /**
     * Handle Unzer payment and set to Magento payment.
     * @param Mage_Payment_Model_Info $mag_payment Magento payment
     * @param UnzerSDK\Resources\Payment $unz_payment Unzer payment
     */
    private function handlePaymentByPayment(
        Mage_Payment_Model_Info $mag_payment,
        Payment $unz_payment
    ): void
    {
        $mag_payment->setAdditionalInformation('paymentId', $unz_payment->getId());
        $mag_payment->setAdditionalInformation('paymentState', $unz_payment->getStateName());
        $mag_payment->setAdditionalInformation('traceId', $unz_payment->getTraceId());
    }

    /**
     * Handle Unzer transaction and set to Magento transaction.
     * @param Mage_Sales_Model_Order_Payment_Transaction $mag_transaction Magento transaction
     * @param UnzerSDK\Resources\TransactionTypes\AbstractTransactionType $unz_transaction Unzer transaction
     */
    private function handleTransactionByTransaction(
        Mage_Sales_Model_Order_Payment_Transaction $mag_transaction,
        AbstractTransactionType $unz_transaction
    ): void
    {
        $mag_transaction->setIsClosed($unz_transaction->isError());
        $mag_transaction->setAdditionalInformation('txStatus', $this->getTransactionStatus($unz_transaction));
    }

    /**
     * Resolve Unzer transaction status.
     * @param UnzerSDK\Resources\TransactionTypes\AbstractTransactionType $unz_transaction Unzer transaction
     * @return string
     */
    private function getTransactionStatus(AbstractTransactionType $unz_transaction): string
    {
        if ($unz_transaction->isError()) {
            return Awardit_Unzer_Constant::RESULT_ERROR;
        }
        if ($unz_transaction->isSuccess()) {
            return Awardit_Unzer_Constant::RESULT_SUCCESS;
        }
        if ($unz_transaction->isPending()) {
            return Awardit_Unzer_Constant::RESULT_PENDING;
        }
        return Awardit_Unzer_Constant::RESULT_ERROR;
    }

    /**
     * Locate currently open Magento transaction.
     * @param Mage_Payment_Model_Info $mag_payment Magento payment
     * @return Mage_Sales_Model_Order_Payment_Transaction|null
     */
    private function getOpenTransaction(Mage_Payment_Model_Info $mag_payment): ?Mage_Sales_Model_Order_Payment_Transaction
    {
        $collection = Mage::getModel('sales/order_payment_transaction')->getCollection()
            ->setOrderFilter($this->getOrder($mag_payment))
            ->addPaymentIdFilter($mag_payment->getId())
            ->setOrder('created_at', Varien_Data_Collection::SORT_ORDER_DESC)
            ->setOrder('transaction_id', Varien_Data_Collection::SORT_ORDER_DESC);
        foreach ($collection as $mag_transaction) {
            if (!$mag_transaction->getIsClosed()) {
                return $mag_transaction;
            }
        }
        return null;
    }

    /**
     * Sometimes Order is not pre-handled on Payment. This method ensures it is.
     * @param Mage_Payment_Model_Info $mag_payment Magento payment
     * @return Mage_Sales_Model_Order
     */
    private function getOrder(Mage_Sales_Model_Order_Payment $mag_payment): Mage_Sales_Model_Order
    {
        if (empty($mag_payment->getOrder())) {
            $order = Mage::getModel('sales/order');
            $order->load($mag_payment->getParentId());
            $mag_payment->setOrder($order);
        }
        return $mag_payment->getOrder();
    }

    /**
     * Updates payment and transaction from Unzer.
     * @param Mage_Sales_Model_Order_Payment_Transaction $mag_transaction Magento transaction
     */
    private function updateTransaction(Mage_Sales_Model_Order_Payment_Transaction $mag_transaction): void
    {
        $mag_payment = $mag_transaction->getOrderPaymentObject();
        if ($mag_transaction->getIsClosed()) {
            return; // Nothing to update
        }

        $payment_id = $mag_payment->getAdditionalInformation('paymentId');
        if (empty($payment_id)) {
            return; // Nothing to update
        }

        $this->helper->log("Update transaction {$mag_transaction->getTxnId()} for {$payment_id}.");

        // Get and update by Unzer payment
        $unzer = $this->helper->unzer();
        $unz_payment = $unzer->fetchPayment($payment_id);

        // Load correct Unzer transaction
        switch ($mag_transaction->getTxnType()) {
            case Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH:
                $unz_transaction = $unz_payment->getAuthorization();
                break;
            case Mage_Sales_Model_Order_Payment_Transaction::TYPE_CAPTURE:
                $unz_transaction = $unz_payment->getCharge($mag_transaction->getTxnId());
                break;
            default:
                return; // Nothing to update
        }

        $this->handlePaymentByPayment($mag_payment, $unz_payment);
        $this->handleTransactionByTransaction($mag_transaction, $unz_transaction);
        $order = $this->getOrder($mag_payment);

        // Handle transaction dependent state changes on success

        if ($unz_transaction->isSuccess()) {
            if ($unz_transaction instanceof Authorization || $unz_transaction instanceof Charge) {
                // If Authorization transaction, resolve payment review
                $order->setCanSendNewEmailFlag(true);
                $mag_payment->setNotificationResult(true);
                $mag_payment->registerPaymentReviewAction(Mage_Sales_Model_Order_Payment::REVIEW_ACTION_ACCEPT, false);
                $order->save();
                $this->helper->log("Resolved payment review for {$order->getIncrementId()} as {$order->getStatus()}");
            }
        }

        // Handle payment dependent state changes on success

        if ($unz_payment->isCanceled()) {
            // @todo: Cancel by backend
            $this->helper->log("Cancelling  payment/{$mag_payment->getId()} for order/{$order->getIncrementId()}.");
        } elseif ($unz_payment->isCompleted()) {
            // @todo: Complete by backend
            $this->helper->log("Completed payment/{$mag_payment->getId()} for order/{$order->getIncrementId()}.");
        }

        // Make info visible in admin
        $mag_transaction->setAdditionalInformation(Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS, null);
        $mag_transaction->setAdditionalInformation(
            Mage_Sales_Model_Order_Payment_Transaction::RAW_DETAILS,
            array_filter(array_merge(
                $mag_payment->getAdditionalInformation(),
                $mag_transaction->getAdditionalInformation()
            ))
        );

        $mag_transaction->save();
        $mag_payment->save();
    }
}
