<?php

declare(strict_types=1);

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use GraphQL\Utils\SchemaPrinter;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\Driver;
use Spatie\Snapshots\MatchesSnapshots;

class GraphqlTest extends TestCase implements Driver
{
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::init('default');
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function serialize($data): string {
        return $data;
    }

    public function extension(): string {
        return "graphql";
    }

    public function match($expected, $actual): void {
        Assert::assertEquals($expected, $this->serialize($actual));
    }

    // Test TradedoublerInfo schema
    public function testInfoSchema(): void {
        $response = $this->runQuery(
            '{ __type(name: "TradedoublerInfo") { name fields { name type { name fields { name type { name } } } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    // Test TradedoublerEvents schema
    public function testEventsSchema(): void {
        $response = $this->runQuery(
            '{ __type(name: "TradedoublerEvents") { name fields { name type { name fields { name type { name } } } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

     // Test TradedoublerEvent schema
    public function testEventSchema(): void {
        $response = $this->runQuery(
            '{ __type(name: "TradedoublerEvent") { name fields { name type { name fields { name type { name } } } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    // Not enabled, entire blocks should be null
    public function testDisabled(): void {
        $this->writeSettings([
            'general/enabled' => '0',
            'general/organization_id' => '12345',
        ]);

        $response = $this->runQuery(
            'query { info { tradedoubler { organizationId programId } } '
            . 'quote { tradedoubler { sale { event exVat incVat } lead { event exVat incVat } } } }'
        );
        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    // Required organization missing, entire blocks should be null
    public function testMissingOrganization(): void {
        $this->writeSettings([
            'general/enabled' => '1',
            'general/organization_id' => '',
        ]);

        $response = $this->runQuery(
            'query { info { tradedoubler { organizationId programId } } '
            . 'quote { tradedoubler { sale { event exVat incVat } lead { event exVat incVat } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    // All required set, other left empty, emtpy event lists
    public function testMinimalSettingsWithoutQuote(): void {
        $this->writeSettings([
            'general/enabled' => '1',
            'general/organization_id' => '12345',
        ]);

        $response = $this->runQuery(
            'query { info { tradedoubler { organizationId programId } } '
            . 'quote { tradedoubler { sale { event exVat incVat } lead { event exVat incVat } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    // All settings set
    public function testAllSettingsWithQuote(): void {
        $this->writeSettings([
            'general/enabled' => '1',
            'general/organization_id' => '12345',
            'general/program_id' => '3456',
            'tracking/events_sale' => [
                ["event_id" => "S11", "name" => "Sale1"],
                ["event_id" => "S22", "name" => "Sale2"],
            ],
            'tracking/events_lead' => [
                ["event_id" => "L33", "name" => "Lead3"],
                ["event_id" => "L44", "name" => "Lead4"],
            ],
        ]);
        $store = Mage::app()->getStore();
        $product_1 = $this->getProduct($store, "L33", "S11");
        $product_2 = $this->getProduct($store, "L44", "S11");
        $quote = $this->getQuote($product_1, $product_2);

        $response = $this->runQuery(
            'query { info { tradedoubler { organizationId programId } } '
            . 'quote { tradedoubler { sale { event exVat incVat } lead { event exVat incVat } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    private function runQuery(string $query) {
        return MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], $query), 'default');
    }

    private function writeSettings(array $settings): void {
        $settings = array_merge([
            'general/enabled' => '0',
            'general/organization_id' => '',
            'general/program_id' => '',
            'tracking/events_sale' => [],
            'tracking/events_lead' => [],
        ], $settings);
        $config = new Mage_Core_Model_Config();
        foreach ($settings as $key => $value) {
            if (is_array($value)) {
                $value = serialize($value);
            }
            $config->saveConfig("awardit_tradedoubler/tradedoubler_{$key}", $value, "default", 0);
        }
    }

    private function getProduct(
        Mage_Core_Model_Store $store,
        string $lead,
        string $sale
    ): Mage_Catalog_Model_Product {
        $template = Mage::getModel("catalog/product");
        $template->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $product = Mage::getModel('catalog/product')->setData($template->getData())
            ->setSku("product-{$lead}-{$sale}-" . rand(1, 9999))
            ->setId(null)
            ->setStoreId($store->getId())
            ->setTradedoublerEventLead($lead)
            ->setTradedoublerEventSale($sale)
            ->save();
        return $product;
    }

    private function getQuote(Mage_Catalog_Model_Product ...$products): Mage_Sales_Model_Quote {
        $quote = Mage::getSingleton("checkout/session")->getQuote();
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "DE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "DE",
        ]));
        $quote->setQuoteCurrencyCode('EUR');
        $quote->setCustomerEmail('test@awardit.com');
        $options = new Varien_Object(["qty" => 1]);
        foreach ($products as $product) {
            $quote->addProduct($product, $options);
        }
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();
        return $quote;
    }
}
