<?php

declare(strict_types=1);

use MageQL\Registry;
use MageQL\Schema\AbstractSchema;
use MageQL\Type\AbstractBuilder;

class Awardit_Tradedoubler_Model_Schema extends AbstractSchema {

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "TradedoublerInfo":
            return $this->object("Tradedoubler settings and info");
        case "TradedoublerEvents":
            return $this->object("Tradedoubler events for current quote");
        case "TradedoublerEvent":
            return $this->object("Tradedoubler tracking event");
        case "TradedoublerDeprecatedEvents":
            return $this->object("Deprecated");
        }
        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "StoreInfo":
            return [
                "tradedoubler" => $this->field(
                    "TradedoublerInfo",
                    "Tradedoubler settings and info"
                )
                    ->setResolver(function(Mage_Core_Model_Store $store): ?array {
                        $helper = Mage::helper("awardit_tradedoubler");
                        if (!$helper->isEnabled($store)) {
                            return null;
                        }
                        $settings = $helper->getGeneralSettings($store);
                        return [
                            "organizationId" => $settings["organization_id"],
                            "programId" => $settings["program_id"],
                            "events" => null,
                        ];
                    }),
            ];
        case "Quote":
            return [
                "tradedoubler" => $this->field(
                    "TradedoublerEvents",
                    "Tradedoubler events"
                )
                    ->setResolver(function($quote, $args, $ctx, $info) {
                        $helper = Mage::helper("awardit_tradedoubler");
                        if (!$helper->isEnabled($quote->getStore())) {
                            return null;
                        }
                        $items = MageQL_Sales_Model_Quote::resolveItems($quote, $args, $ctx, $info);
                        $sales = $leads = [];
                        foreach ($items as $item) {
                            $product = $item->getProduct();
                            $product->load($product->getId);
                            if ($sale = $product->getTradedoublerEventSale()) {
                                $sales = $this->mapEvent($sales, $sale, $item, $args, $ctx);
                            }
                            if ($lead = $product->getTradedoublerEventLead()) {
                                $leads = $this->mapEvent($leads, $lead, $item, $args, $ctx);
                            }
                        }
                       return [
                            'sale' => array_values($sales),
                            'lead' => array_values($leads),
                        ];
                    }),
            ];
        case "TradedoublerInfo":
            return [
                "organizationId" => $this->field("String!", "Organization ID"),
                "programId" => $this->field("String", "Program ID"),
                "events" => $this->field("TradedoublerDeprecatedEvents", "Deprecated, always null"),
            ];
        case "TradedoublerEvents":
            return [
                "sale" => $this->field("[TradedoublerEvent!]!", "Sale events"),
                "lead" => $this->field("[TradedoublerEvent!]!", "Lead events"),
            ];
        case "TradedoublerDeprecatedEvents":
            return [
                "sale" => $this->field("String", "Deprecated"),
                "lead" => $this->field("String", "Deprecated"),
            ];
        case "TradedoublerEvent":
            return [
                "event" => $this->field("String!", "Event"),
                "exVat" => $this->field("Float!", "Subtotal excluding VAT"),
                "incVat" => $this->field("Float!", "Subtotal including VAT"),
            ];
        }

        return [];
    }

    private function mapEvent(array $list, string $event, $item, $args, $ctx): array {
        if (!array_key_exists($event, $list)) {
            $list[$event] = [
                "event" => $event,
                "exVat" => 0,
                "incVat" => 0,
            ];
        }
        $list[$event]["exVat"] += MageQL_Sales_Model_Item::resolveRowTotalExVat(
            $item,
            $args,
            $ctx
        );
        $list[$event]["incVat"] += MageQL_Sales_Model_Item::resolveRowTotalIncVat(
            $item,
            $args,
            $ctx
        );
        return $list;
    }
}
