<?php

declare(strict_types=1);

use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use GraphQL\Utils\SchemaPrinter;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\Driver;
use Spatie\Snapshots\MatchesSnapshots;

class GraphqlTest extends TestCase implements Driver
{    
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::init('default');
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function serialize($data): string {
        return $data;
    }

    public function extension(): string {
        return "graphql";
    }

    public function match($expected, $actual): void {
        Assert::assertEquals($expected, $this->serialize($actual));
    }

    // Test TradedoublerInfo schema
    public function testSchema(): void {
        $response = $this->runQuery(
            '{ __type(name: "TradedoublerInfo") { name fields { name type { name fields { name type { name } } } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }
    
    // Not enabled, entire block should be null
    public function testDisabled(): void {
        $this->writeSettings([
            'general/enabled' => '0',
            'general/organization_id' => '12345',
        ]);
        
        $response = $this->runQuery(
            'query { info { tradedoubler { organizationId programId events { sale lead } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    // Required organization missing, entire block should be null
    public function testMissingOrganization(): void {
        $this->writeSettings([
            'general/enabled' => '1',
            'general/organization_id' => '',
        ]);
        
        $response = $this->runQuery(
            'query { info { tradedoubler { organizationId programId events { sale lead } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    // All required set, all other left empty
    public function testMinimalSettings(): void {
        $this->writeSettings([
            'general/enabled' => '1',
            'general/organization_id' => '12345',
        ]);
        
        $response = $this->runQuery(
            'query { info { tradedoubler { organizationId programId events { sale lead } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    // All settings set
    public function testAllSettings(): void {
        $this->writeSettings([
            'general/enabled' => '1',
            'general/organization_id' => '12345',
            'general/program_id' => '3456',
            'tracking/event_sale' => '7890',
            'tracking/event_lead' => '8901',
        ]);
        
        $response = $this->runQuery(
            'query { info { tradedoubler { organizationId programId events { sale lead } } } }'
        );

        $this->assertEquals(200, $response->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $response->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($response->getBody());
    }

    private function runQuery(string $query) {
        return MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], $query), 'default');
    }
    
    private function writeSettings(array $settings): void {
        $settings = array_merge([
            'general/enabled' => '0',
            'general/organization_id' => '',
            'general/program_id' => '',
            'tracking/event_sale' => '',
            'tracking/event_lead' => '',
        ], $settings);
        
        $config = new Mage_Core_Model_Config();
        foreach ($settings as $key => $value) {
            $config->saveConfig("awardit_tradedoubler/tradedoubler_{$key}", $value, "default", 0);
        }
    }
}

