<?php

declare(strict_types=1);

class Awardit_Tradedoubler_Model_Api_Adapter
{
    private const CACHE_LIFETIME = 604800; // 1 week

    /**
     * @var Awardit_Tradedoubler_Model_Api_Client
     */
    private $client;
    /**
     * @var Awardit_Tradedoubler_Helper_Data
     */
    private $helper;

    /**
     * Setup worker
     */
    public function __construct()
    {
        $this->client = Mage::getModel('awardit_tradedoubler/api_client');
        $this->helper = Mage::helper('awardit_tradedoubler');
        $flat = Mage::helper('catalog/product_flat');
        $flat->disableFlatCollection(); // This doesn't work properly when using flat tables
    }

    /**
     * Main worker for updating product feeds
     */
    public function updateCatalog(): void
    {
        $this->helper->log('Updating TradeDoubler catalog.');

        $cache = Mage::app()->getCacheInstance();

        foreach ($this->getStores() as $store) {
            if (!$this->helper->isCatalogEnabled($store)) {
                continue; // No fun to be had
            }
            $general = $this->helper->getGeneralSettings($store);
            if (empty($general["api_token"])) {
                return;
            }

            $available_feeds = [];
            foreach ($this->client->getFeeds($general["api_token"]) as $feed) {
                $this->helper->log("Running store {$store->getId()} for feed {$feed->feedId}");
                if ($feed->deleted) {
                    $this->clearDeleted($store, $feed);
                    continue; // Ignore deleted feeds
                }
                if ($cache->test("tradedoubler_feed_{$feed->feedId}")) {
                    // continue; // Skip feeds in cache @todo: Fix this so it actually works
                }

                $catalog = $this->getLocalCatalog($store, $feed->feedId);
                $result = $this->client->replaceProducts(
                    $general["api_token"],
                    $feed->feedId,
                    $catalog
                );
                $cache->save(json_encode($catalog), "tradedoubler_feed_{$feed->feedId}", [], self::CACHE_LIFETIME);
                $count = count($catalog['products']);
                $status = $result ? 'success' : 'failure';
                $this->helper->log("Enabled {$count} items for feed {$feed->feedId}: {$status}");
            }
        }
    }

    /**
     * Get relevant products in catalog and format according to Tradedoubler
     * @param Mage_Core_Model_Store $store Store to update feed for
     * @param int $feed_id Feed identifier
     * @return array<array-key, array<array-key, mixed>> Catalog in Tradedoubler format
     */
    private function getLocalCatalog(Mage_Core_Model_Store $store, int $feed_id): array
    {
        $items = [];

        $collection = Mage::getResourceModel('catalog/product_collection');
        $collection->setStore($store);
        $collection->setVisibility(Mage::getSingleton('catalog/product_visibility')->getVisibleInCatalogIds());
        $collection->addAttributeToFilter('tradedoubler_feed', ['eq' => $feed_id]);

        foreach ($collection as $product) {
            $product->setStoreId($store->getId())->load($product->getId());

            $images = $product->getMediaGalleryImages();
            if (count($images) < 1 || empty($product->getSku())) {
                continue; // Required in export, ignore if missing
            }
            $image = $images->getFirstItem();
            $categories = [];
            foreach ($this->helper->parseCategories($product) as $category) {
                $categories[]["id"] = $category;
            }

            $item = [
                "categories" => $categories,
                "name" => $product->getName(),
                "description" => $product->getDescription(),
                "price" => round($product->getPrice(), 2),
                "sourceProductId" => $product->getSku(),
                "productUrl" => $product->getUrlInStore(),
                "productImage" => [
                    "url" => $image->getUrl(),
                ]
            ];
            $items[] = $item;
        }
        return ["products" => $items];
    }

    /**
     * If feed is deleted, remove corresponding feed mapping on all products
     * @param Mage_Core_Model_Store $store Store to update feed for
     * @param int $feed_id Feed identifier
     */
    private function clearDeleted(Mage_Core_Model_Store $store, int $feed_id): void
    {
        $collection = Mage::getResourceModel('catalog/product_collection');
        $collection->setStore($store);
        $collection->addAttributeToFilter('tradedoubler_feed', ['eq' => $feed_id]);
        foreach ($collection as $product) {
            $product->setTradedoublerFeed(0);
            $product->save();
        }
    }

    /**
     * Get all stores
     * @return array Stores
     */
    private function getStores(): array
    {
        return Mage::app()->getStores();
    }
}
