<?php

class Awardit_Tradedoubler_Helper_Data extends Mage_Core_Helper_Abstract {

    /** @const string */
    private const LOG_CHANNEL = 'tradedoubler.log';

    /**
     * If Tradedoubler is enabled and has mandatory settings
     * @param Mage_Core_Model_Store|int|null $store
     * @return bool If enabled
     */
    public function isEnabled($store = null): bool {
        /**
         * @var array{enabled: bool, organization_id: string, program_id: string} $settings
         */
        $settings = $this->getGeneralSettings($store);
        return $settings["enabled"] && !empty($settings["organization_id"]);
    }

    /**
     * If Tradedoubler catalog manager is enabled
     * @param Mage_Core_Model_Store|int|null $store
     * @return bool If enabled
     */
    public function isCatalogEnabled($store = null): bool
    {
        $general = $this->getGeneralSettings($store);
        $catalog = $this->getCatalogSettings($store);
        return $general["api_token"] && $catalog["enabled"] && $this->isEnabled($store);
    }

    /**
     * Get general settings
     * @param Mage_Core_Model_Store|int|null $store
     * @return array{enabled: bool, organization_id: string, program_id: string|null, api_token: string|null}
     */
    public function getGeneralSettings($store = null): array {
        return [
            "enabled" => (bool)$this->getSetting('general/enabled', $store),
            "organization_id" => (string)$this->getSetting('general/organization_id', $store),
            "program_id" => (string)$this->getSetting('general/program_id', $store) ?: null,
            "api_token" => (string)$this->getSetting('general/api_token', $store) ?: null,
        ];
    }

    /**
     * Get tracking events
     * @param Mage_Core_Model_Store|int|null $store
     * @return array{lead: array<array-key, mixed>, sale: array<array-key, mixed>}
     */
    public function getTrackingEvents($store = null): array {
        $unserializer = Mage::helper("core/unserializeArray");
        return [
            "lead" => $unserializer->unserialize($this->getSetting('tracking/events_lead', $store)),
            "sale" => $unserializer->unserialize($this->getSetting('tracking/events_sale', $store)),
        ];
    }

    /**
     * Get catalog settings
     * @param Mage_Core_Model_Store|int|null $store
     * @return array{enabled: bool}
     */
    public function getCatalogSettings($store = null): array {
        return [
            "enabled" => (bool)$this->getSetting('catalog/enabled', $store),
        ];
    }

    /**
     * Ensure unique, non-empty categories as integer values
     * @param Mage_Catalog_Model_Product|string|null $source
     * @return array<int>
     */
    public function parseCategories($source): array {
        if ($source instanceof Mage_Catalog_Model_Product) {
            $source = $source->getTradedoublerCategories();
        }
        $source = strval($source);
        return array_values(array_map('intval', array_unique(array_filter(explode(",", $source)))));
    }

    /**
     * Get a setting
     * @param $path Setting key
     * @param Mage_Core_Model_Store|int|null $store
     * @return mixed
     */
    private function getSetting(string $path, $store = null) {
        return Mage::getStoreConfig("awardit_tradedoubler/tradedoubler_{$path}", $store);
    }

    /**
     * Write to log.
     * @param string $message Message to log.
     * @param int $level Any Zend_Log level constant.
     */
    public function log(string $message, int $level = Zend_Log::DEBUG): void
    {
        Mage::log($message, $level, self::LOG_CHANNEL);
    }
}
