<?php

declare(strict_types=1);

use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Request;

class Awardit_Tradedoubler_Model_Api_Client {

    private const BASE_URI = "https://api.tradedoubler.com/1.0/";

    /**
     * @var Client
     */
    private $client;

    /**
     * Setup client
     */
    public function __construct()
    {
        $verify = empty(Mage::getStoreConfig('awardit_tradedoubler/tradedoubler_general/api_no_verify'));
        $this->client = new Client([
            'base_uri' => self::BASE_URI,
            'verify' => $verify,
        ]);
    }

    /**
     * Update catalog (createm update, delete products)
     * @param string $token Account setting
     * @param int $fid Feed id to update
     * @param array<array-key, array<array-key, mixed>> $catalog Catalog data, as expected by Tradedoubler
     * @return bool If operation performed without errors
     */
    public function replaceProducts(string $token, int $fid, array $catalog): bool {
        $result = $this->client->post(
            "products;fid={$fid};mode=replace?token={$token}",
            [
                "headers" => [
                    "Content-Type" => "application/json; charset=utf-8"
                ],
                "body" => $this->encode($catalog),
            ]
        );
        $decoded = $this->decode($result->getBody()->__toString());
        return empty($decoded->errors);
    }

    /**
     * Get feeds
     * @param string $token Account setting
     * @return array List of feeds
     */
    public function getFeeds(string $token): array {
        $result = $this->client->get("productFeeds?token={$token}");
        $decoded = $this->decode($result->getBody()->__toString());
        return $decoded->feeds;
    }

    /**
     * Get categories
     * @param string $token Account setting
     * @param string $language Language for labels, English by default
     * @return array Tree structure of categories
     */
    public function getCategories(string $token, string $language = "en"): array {
        $result = $this->client->get("productCategories;language={$language}?token={$token}");
        $decoded = $this->decode($result->getBody()->__toString());
        return $decoded->categoryTrees;
    }

    /**
     * Encode to JSON
     * @param mixed $data Data to be encoded
     * @return string Encoded data
     */
    private function encode($data): string {
        return json_encode($data, JSON_THROW_ON_ERROR, 512);
    }

    /**
     * Decode JSON
     * @param string $json Data to be decoded
     * @return mixed Decoded data
     */
    private function decode(string $json) {
        return json_decode($json, null, 512, JSON_THROW_ON_ERROR);
    }
}
