<?php

use JoliCode\Slack\{
    Client,
    ClientFactory
};

/**
 * Wrapper class for Slack integration.
 */
class Awardit_Slack_Helper_Data extends Mage_Core_Helper_Abstract
{
    /** @var JoliCode\Slack\Client|null */
    private $client = null;

    /**
     * Get all channels this Slack App has access to.
     * @return array<string,string> Id as key, name as value.
     */
    public function getAvailableChannels(): array
    {
        return $this->handle(function() {
            $client = $this->getClient();
            $available = [];
            foreach ($client->iterateConversationsList([
                'exclude_archived' => true,
                'types' => 'public_channel',
            ]) as $channel) {
                if ($channel->getIsMember()) {
                    $available[$channel->getId()] = $channel->getName();
                }
            }
            asort($available, SORT_NATURAL);
            return $available;
        }, []);
    }

    /**
     * Post message in channel.
     * @param string $channel Id or name for channel.
     * @param string $text Message to post.
     * @param array $blocks Optional list of layout blocks.
     * @return bool If method was able to write.
     */
    public function postInChannel(string $channel, string $text, array $blocks = []): bool
    {
        return $this->handle(function() use ($channel, $text, $blocks): bool {
            $client = $this->getClient();
            $data = [
                'channel' => $channel,
                'text' => $text,
            ];
            if (!empty($blocks)) {
                $data['blocks'] = json_encode($blocks);
            }
            $result = $client->chatPostMessage($data);
            return $result ? $result->getOk() : false;
        }, false);
    }

    /**
     * Exception catcher and logger.
     * @param callable $callback Code to run within handler.
     * @param mixed $on_error What to return or error.
     * @return mixed Callback result or error result.
     */
    private function handle(callable $callback, $on_error = null)
    {
        try {
            return $callback();
        } catch (Throwable $e) {
            Mage::logException($e);
            return $on_error;
        }
    }

    /**
     * Get Slack client (may cause exception).
     * @return JoliCode\Slack\Client Slack client.
     */
    private function getClient(): Client
    {
        if (is_null($this->client)) {
            $token = Mage::getStoreConfig('awardit_slack/general/token');
            $this->client = ClientFactory::create($token);
        }
        return $this->client;
    }
}
