<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event;

use JsonSerializable;
use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\EventInterface;

/**
 * @psalm-type TaxClass "default"
 * @psalm-type ProductType "physical"|"virtual"|"dropship"
 * @psalm-type Price array{price:string, currency:string}
 * @psalm-type Payload array{
 *   name?:?string,
 *   type:ProductType,
 *   weight?:?float,
 *   archived:bool,
 *   sellable:bool,
 *   visible:bool,
 *   taxClass:TaxClass,
 *   allowShortPick:bool,
 *   manufacturer?:?string,
 *   mpn?:?string,
 *   brand?:?string,
 *   supplier?:?string,
 *   supplierSku?:?string,
 *   msrp:list<Price>,
 *   purchasePrice:?Price,
 *   categories:list<string>
 * }
 */
class ErpProduct implements AutoRevisionedEventInterface, EventInterface, JsonSerializable
{
    use AutoRevisionedEventTrait;

    public static function getMessageType(): string
    {
        return "AwarditErpProductV1";
    }

    public static function getMessageTopic(): string
    {
        return "awardit-erp-product";
    }

    public static function fromReceivedMessage(ReceivedMessageInterface $message): self
    {
        /** @var Payload */
        $payload = $message->getEventPayload();

        $event = new self(
            sku: $message->getEventEntityId(),
            name: $payload["name"] ?? null,
            type: $payload["type"],
            weight: $payload["weight"] ?? null,
            archived: $payload["archived"],
            sellable: $payload["sellable"],
            visible: $payload["visible"],
            taxClass: $payload["taxClass"],
            allowShortPick: $payload["allowShortPick"],
            manufacturer: $payload["manufacturer"] ?? null,
            mpn: $payload["mpn"] ?? null,
            brand: $payload["brand"] ?? null,
            supplier: $payload["supplier"] ?? null,
            supplierSku: $payload["supplierSku"] ?? null,
            msrp: $payload["msrp"],
            purchasePrice: $payload["purchasePrice"] ?? null,
            categories: $payload["categories"] ?? [],
        );

        $event->setRevision($message->getEventRevision());

        return $event;
    }

    public function __construct(
        /**
         * Product Stock Keeping Unit.
         */
        readonly public string $sku,
        /**
         * Product name.
         */
        readonly public ?string $name,
        /**
         * @var ProductType
         */
        readonly public string $type,
        /**
         * Product weight in kg, for physical products.
         */
        readonly public ?float $weight,
        /**
         * True if the product is archived.
         */
        readonly public bool $archived,
        /**
         * If we can sell the product.
         */
        readonly public bool $sellable,
        /**
         * If the product is visible to the customer.
         */
        readonly public bool $visible,
        /**
         * If we allow sales despite not having enouch in stock.
         */
        readonly public bool $allowShortPick,
        /**
         * Manufacturer name.
         */
        readonly public ?string $manufacturer,
        /**
         * Manufacturer Part Number.
         */
        readonly public ?string $mpn,
        /**
         * Brand name.
         */
        readonly public ?string $brand,
        /**
         * Supplier name.
         */
        readonly public ?string $supplier,
        /**
         * Supplier stock keeping unit id.
         */
        readonly public ?string $supplierSku,
        /**
         * Tax classification of the product.
         *
         * @var TaxClass
         */
        readonly public string $taxClass,
        /**
         * Manufacturer suggested retail prices.
         *
         * The price-key is a string-formatted decimal.
         *
         * @var list<Price>
         */
        readonly public array $msrp,
        /**
         * The purchase price.
         *
         * The price-key is a string-formatted decimal.
         *
         * @var ?Price
         */
        readonly public ?array $purchasePrice,
        /**
         * List of categories the product belongs to, paths with `/` as level separator.
         *
         * Example:
         *
         * ```
         * [
         *    "Electronics/Lighting/Kits",
         *    "Security/Miscellaneous",
         *    "Campaign products",
         * ]
         * ```
         *
         * @var list<string>
         */
        readonly public array $categories = [],
    ) {
    }

    public function getEntityId(): string
    {
        return $this->sku;
    }

    public function getIsDeleted(): bool
    {
        return $this->archived;
    }

    /**
     * @return Payload
     */
    public function formatMessagePayload(): array
    {
        return [
            "name" => $this->name,
            "type" => $this->type,
            "weight" => $this->weight,
            "archived" => $this->archived,
            "sellable" => $this->sellable,
            "visible" => $this->visible,
            "allowShortPick" => $this->allowShortPick,
            "taxClass" => $this->taxClass,
            "manufacturer" => $this->manufacturer,
            "mpn" => $this->mpn,
            "brand" => $this->brand,
            "supplier" => $this->supplier,
            "supplierSku" => $this->supplierSku,
            "msrp" => $this->msrp,
            "purchasePrice" => $this->purchasePrice,
            "categories" => $this->categories,
        ];
    }

    public function jsonSerialize(): array
    {
        return get_object_vars($this);
    }
}
