<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event;

use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\EventInterface;

/**
 * @psalm-type Price array{code:string, currency:string, value:string, includesTax:bool}
 * @psalm-type Payload array{
 *   priceList: string,
 *   prices: list<Price>,
 * }
 */
readonly class ProductPricingPrice implements EventInterface
{
    public static function getMessageType(): string
    {
        return "AwarditPricingProductPriceV1";
    }

    public static function getMessageTopic(): string
    {
        return "awardit-pricing-product-price";
    }

    public static function fromReceivedMessage(ReceivedMessageInterface $message): self
    {
        /** @var Payload */
        $payload = $message->getEventPayload();

        return new self(
            sku: $message->getEventEntityId(),
            priceList: $payload["priceList"],
            revision: $message->getEventRevision(),
            archived: $message->getEventIsDeleted(),
            prices: $payload["prices"],
        );
    }

    public function __construct(
        /**
         * Product Stock Keeping Unit.
         */
        public string $sku,
        /**
         * Price list code for the price list which this product-price is part
         * of, human readable.
         */
        public string $priceList,
        /**
         * Product price revision for the particular product in the price list.
         */
        public int $revision,
        /**
         * If this price-list entry is archived.
         */
        public bool $archived,
        /**
         * List of prices for this particular product and tenant.
         *
         * * code: string, human-readable code identifying the type of price
         * * currency: string, ISO 3-letter currency-code, or a string
         *             identifying a point currency
         * * value: string, string-formatted float for bignum, `^\d+\.\d+`
         * * includesTax: boolean
         *
         * @var list<Price>
         */
        public array $prices,
    ) {
        assert($revision > 0);
    }

    public function getEntityId(): string
    {
        return $this->sku;
    }

    public function getIsDeleted(): bool
    {
        return $this->archived;
    }

    public function getRevision(): int
    {
        return $this->revision;
    }

    /**
     * @return Payload
     */
    public function formatMessagePayload(): array
    {
        return [
            "priceList" => $this->priceList,
            "prices" => $this->prices,
        ];
    }
}
