<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Aws;

use Awardit\Aws\Json;
use Awardit\SimpleEvent\EventHandlerInterface;
use Awardit\SimpleEvent\EventRegistry;
use Awardit\SimpleEvent\Exception;
use Awardit\SimpleEvent\Metadata;
use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\V1\ReceivedMessage;
use Aws\Sqs\SqsClient;
use Psr\Log\LoggerInterface;
use Throwable;

// TODO: Support message delivery no matter if raw message delivery is enabled
// or not (specify a flag, or auto-detect?)
/**
 * @psalm-type AttributeValue array{
 *   BinaryListValues: string[],
 *   BinaryValue: string,
 *   DataType: string,
 *   StringListValues: string[],
 *   StringValue: string
 * }
 * @psalm-type SqsMessage array{
 *   Attributes: Array<string, string>,
 *   Body: string,
 *   MD5OfBody: string,
 *   MD5OfMessageAttributes: string,
 *   MessageAttributes: Array<string, AttributeValue>,
 *   MessageId: string,
 *   ReceiptHandle: string
 * }
 */
class SqsEventListener
{
    private const array SYSTEM_ATTRIBUTE_NAMES = [
        "ApproximateReceiveCount",
        "MessageDeduplicationId",
        "SequenceNumber"
    ];

    /**
     * @var list<SqsMessage>
     */
    private array $pendingMessages = [];

    /**
     * @api
     */
    public function __construct(
        private LoggerInterface $log,
        private SqsClient $client,
        /**
         * The AWS URL of the SQS queue to consume.
         */
        private string $queueUrl,
        private EventRegistry $registry,
        /**
         * Maximum number of messages to fetch simultaneouslly, 1 to 10.
         */
        private int $maxNumberOfMessages = 1,
        /**
         * Wait time in seconds in case the queue does not contain any messages.
         */
        private int $waitTimeSeconds = 30
    ) {
        assert($this->waitTimeSeconds >= 0);
        assert($this->maxNumberOfMessages >= 1);
        assert($this->maxNumberOfMessages <= 10);
    }

    /**
     * Starts a loop processing events indefinitely.
     *
     * NOTE: Make sure waitTimeSeconds is greater than zero, otherwise the SQS
     * Event API will be spammed by a lot of requests.
     *
     * @api
     */
    public function run(EventHandlerInterface $handler): void
    {
        while (true) {
            $this->receiveEvents($handler);
        }
    }

    /**
     * Attempts to fetch messages from the queue, and then runs the event
     * handler for each obtained event.
     *
     * @api
     * @return bool Returns true if at least one event was processed
     */
    public function receiveEvents(EventHandlerInterface $handler): bool
    {
        // Make sure we only fetch events if we do not already have a few
        if (empty($this->pendingMessages)) {
            $this->fillPending();
        }

        $handledEvents = false;

        while (count($this->pendingMessages) > 0) {
            $message = array_shift($this->pendingMessages);
            ["decoded" => $decoded, "meta" => $meta] = $this->decodeMessage($message);
            $event = $this->registry->createEvent($decoded);

            $this->log->debug("Constructed event from received message", [
                "method" => __METHOD__,
                "event" => $event,
                "event.class" => $event::class,
            ]);

            if ($handler->handle($event, $meta)) {
                $this->deleteMessage($message);
            }

            $handledEvents = true;
        }

        return $handledEvents;
    }

    /**
     * Returns true if there are events which have been fetched but not yet
     * been processed.
     *
     * @api
     */
    public function hasPendingEvents(): bool
    {
        return count($this->pendingMessages) > 0;
    }

    /**
     * Fills the pending events buffer with messages.
     */
    private function fillPending(): void
    {
        $args = [
            "MessageAttributeNames" => ["All"],
            "MessageSystemAttributeNames" => self::SYSTEM_ATTRIBUTE_NAMES,
            "QueueUrl" => $this->queueUrl,
            "MaxNumberOfMessages" => $this->maxNumberOfMessages,
            "waitTimeSeconds" => $this->waitTimeSeconds
        ];

        $this->log->debug("Attempting to receive messages from {QueueUrl}", [
            "method" => __METHOD__,
            ...$args,
        ]);

        $result = $this->client->receiveMessage($args);
        /** @var SqsMessage[] */
        $messages = $result->get("Messages") ?? [];

        $this->log->debug("Received {count} messages from {QueueUrl}", [
            "count" => count($messages),
            "QueueUrl" => $this->queueUrl,
        ]);

        foreach ($messages as $message) {
            $this->pendingMessages[] = $message;
        }
    }

    /**
     * @param SqsMessage $message
     * @return array{decoded:ReceivedMessageInterface, meta:Metadata}
     */
    private function decodeMessage(array $message): array
    {
        $body = Json::decode($message["Body"]);
        $version = (int)($body["version"] ?? 1);

        switch ($version) {
            case 1:
                return [
                    "decoded" => new ReceivedMessage($body),
                    "meta" => new Metadata(
                        correlationId: (string)($body["meta"]["correlationId"] ?? "<UNKNOWN>"),
                        sender: (string)($body["meta"]["sender"] ?? "<UNKNOWN>"),
                    ),
                ];
            default:
                throw new Exception(sprintf("Unsupported message version %d", $version));
        }
    }

    /**
     * Deletes the supplied message from the SQS queue, indicating that it has
     * been successfully handled.
     *
     * @param SqsMessage $message
     */
    private function deleteMessage(array $message): void
    {
        $this->log->info("Deleting message", [
            "method" => __METHOD__,
            "message" => [
                "ReceiptHandle" => $message["ReceiptHandle"],
            ],
        ]);

        $this->client->deleteMessage([
            "QueueUrl" => $this->queueUrl,
            "ReceiptHandle" => $message["ReceiptHandle"],
        ]);
    }
}
