<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event;

use Awardit\SimpleEvent\Exception;
use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\EventInterface;

/**
 * Abstract representation of a service-internal event which is not supposed to
 * be sent publicly on a topic.
 *
 * * getMessageTopic: Will throw to prevent the event from being emitted on a
 *                    Simple Notification Service topic
 * * getMessageType:  Automatically gives the event a type-name based on
 *                    namespace and class
 * * getRevision:     Automatically uses the current time in case a revision is
 *                    not explicitly provided
 * * getIsDeleted:    Returns false by default
 */
abstract class AbstractInternalEvent implements EventInterface
{
    final public static function getMessageTopic(): string
    {
        throw new Exception(sprintf(
            "%s is an internal event and should never be emitted on a topic",
            get_called_class()
        ));
    }

    public static function getMessageType(): string
    {
        // Some name which will not conflict with any of our normal event types
        return "php:" . get_called_class();
    }

    private int $revision;

    public function __construct(int $revision = 0)
    {
        $this->revision = $revision <= 0 ? time() : $revision;
    }

    public function getRevision(): int
    {
        return $this->revision;
    }

    public function getIsDeleted(): bool
    {
        // By default we do not consider internal events as being able to be deleted
        return false;
    }
}
