<?php

namespace Awardit\SimpleEvent\Aws\Localstack;

use Awardit\Aws\Exception;
use Awardit\Aws\Json;
use Awardit\SimpleEvent\{
    EventHandlerInterface,
    ReceivedMessageInterface,
    EventRegistry,
    Metadata,
};
use Awardit\SimpleEvent\Event\ErpProduct;
use Awardit\SimpleEvent\V1\ReceivedMessage;
use Aws\Exception\AwsException;
use Aws\Sqs\SqsClient;
use Psr\Log\LoggerInterface;
use Throwable;

/**
 * Handler and listener for SQS in localstack
 * @api
 * @psalm-type SqsMessage array{MessageId:string, ReceiptHandle:string, Body:string}
 */
class LocalSqsConsumer
{
    private LoggerInterface $logger;
    private EventHandlerInterface $eventHandler;
    private SqsClient $sqsClient;
    private string $queueUrl;
    private EventRegistry $registry;

    public function __construct(
        LoggerInterface $logger,
        EventHandlerInterface $eventHandler,
        SqsClient $sqsClient,
        string $queueUrl,
    ) {
        $this->logger = $logger;
        $this->eventHandler = $eventHandler;
        $this->sqsClient = $sqsClient;
        $this->queueUrl = $queueUrl;

        $this->registry = new EventRegistry([ErpProduct::class]);
    }

    public function listen(): void
    {
        swoole_timer_tick(3000, function () {
            $this->poll();
        });

        swoole_event_wait();
    }

    private function poll(): void
    {
        $this->logger->debug('Polling for messages on ' . $this->queueUrl);
        try {
            $result = $this->sqsClient->receiveMessage([
                'QueueUrl' => $this->queueUrl,
                // 'WaitTimeSeconds' => 30,
                'MaxNumberOfMessages' => 10,
            ]);

            /** @var array<SqsMessage>|null */
            $messages = $result->get('Messages');

            if (is_array($messages) && count($messages) > 0) {
                $this->logger->debug('Got ' . count($messages) . ' message(s)');
                $this->handle($messages);
            }
        } catch (AwsException $e) {
            $this->logger->critical("Could not get message(s): " . $e->getMessage());
        }
    }

    /**
     * @param array<SqsMessage> $messages
     */
    private function handle(array $messages): void
    {
        foreach ($messages as $message) {
            try {
                ["decoded" => $decoded, "meta" => $meta] = $this->decodeMessage($message);

                $event = $this->registry->createEvent($decoded);

                $this->logger->debug("Constructed event from received message", [
                    "method" => __METHOD__,
                    "event" => $event,
                    "event.class" => $event::class,
                ]);

                if (!$this->eventHandler->handle($event, $meta)) {
                    $this->logger->info("Message " . $message['MessageId'] . " failed based on return value");
                } else {
                    $this->deleteMessage($message['ReceiptHandle']);
                }
            } catch (Throwable $t) {
                $this->logger->error("Failed to process record {messageId}: {message}", [
                    "message" => $t->getMessage(),
                    "method" => __METHOD__,
                    "messageId" => $message["MessageId"],
                    "record" => $message,
                    "exception" => $t,
                ]);
            }
        }
    }

    /**
     * @param string $receiptHandle
     */
    private function deleteMessage($receiptHandle): void
    {
        try {
            $this->sqsClient->deleteMessage([
                'QueueUrl' => $this->queueUrl,
                'ReceiptHandle' => $receiptHandle,
            ]);
        } catch (AwsException $e) {
            $this->logger->error("Could not delete message with id " . $receiptHandle . ": " . $e->getMessage());
        }
    }

    /**
     * @param SqsMessage $record
     * @return array{decoded:ReceivedMessageInterface, meta:Metadata}
     */
    private function decodeMessage(array $record): array
    {
        $body = Json::decode($record["Body"]);
        $version = (int)($body["version"] ?? 1);

        switch ($version) {
            case 1:
                return [
                    "decoded" => new ReceivedMessage($body),
                    "meta" => new Metadata(
                        correlationId: (string)($body["meta"]["correlationId"] ?? "<UNKNOWN>"),
                        sender: (string)($body["meta"]["sender"] ?? "<UNKNOWN>"),
                    ),
                ];
            default:
                throw new Exception(sprintf("Unsupported message version %d", $version));
        }
    }
}
