# Awardit SimpleEvent

SimpleEvent is a library which makes it simple to create, emit, and listen to
events.

## Installing

```bash
composer require awardit/simple-event
```

## Examples

Emitting an event:

```php
$emitter->emit(new SomeEvent(
    id: $myId,
    data: ....
));
```

Listening for an event:

```php
// Poll the listener with a callback which is executed for the event received
// from a queue
$listener->receive(function(EventInterface $event): bool {
    // Do something with $event

    if ($event instanceof MyEvent) {
        // if we want to perform different logic based on event
    }

    // Finally delete the message from the queue since we are done with it
    return true;
});
```

Diagram showing how different components of SimpleEvent interacts:

```mermaid
graph TD
    subgraph SNS[Simple Notification Service]
        TopicSomeData[SomeData Topic]
    end

    subgraph SQS[Simple Queue Service]
        QueueMyService[MyService Queue]
    end

    subgraph ServiceOther[Other Service]
        EmitterSnsOther[SNS Event Emitter]
    end

    subgraph ServiceMy[My Service]
        EmitterSqsMy[SQS Event Emitter]
        ListenerSqsMy[SQS Event Listener]
    end

    SubscriptionSomeData{{Subscription}}

    TopicSomeData --> SubscriptionSomeData
    SubscriptionSomeData --> QueueMyService
    EmitterSnsOther --> TopicSomeData
    EmitterSqsMy --> QueueMyService
    QueueMyService --> ListenerSqsMy
```

## Local testing

* [Docker](https://www.docker.com/)
* [AWS CLI](https://docs.aws.amazon.com/cli/latest/userguide/getting-started-install.html) Version 2

Start Localstack using docker compose via a script which also configures our
local profile and test resources:

```bash
./scripts/start-localstack.sh
```

### Manual setup

To manually do the above:

```bash
docker compose up -d localstack
```

Verify the service(s) are running by using `docker ps` or `http://localhost:4566`,
we should get a 200 OK response.

Now we can start to set up our simulated cloud environment, starting with a
credentials profile:

```bash
aws configure set aws_access_key_id "dummy" --profile localstack
aws configure set aws_secret_access_key "dummy" --profile localstack
aws configure set region "eu-north-1" --profile localstack
aws configure set endpoint_url http://localhost.localstack.cloud:4566 --profile localstack
aws configure set output "json" --profile localstack
aws configure set cli_pager "cat" --profile localstack

# We can also alias the aws CLI to automatically add the profile
alias awslocal='aws --profile localstack'
```

Configuring our Simple Notification Service topic and Simple Queue Service:

```bash
awslocal sns create-topic --name simple-event-test
awslocal sqs create-queue --queue-name simple-event-test
awslocal sns subscribe --topic-arn "arn:aws:sns:eu-north-1:000000000000:simple-event-test" --protocol sqs --attributes RawMessageDelivery=true --notification-endpoint "arn:aws:sqs:eu-north-1:000000000000:simple-event-test"
```

To manually receive events from the queue we just set up we can use the following command:

```bash
awslocal sqs receive-message --queue-url http://localhost.localstack.cloud:4566/000000000000/simple-event-test
```

Then to run the example:

```bash
docker compose run example-basic
```

### Troubleshooting

#### Localstack 500 error: `ValueError: not enough values to unpack (expected 6, got 5)`

Verify that the ARN is correctly constructed in all parameters, eg. the following code
reads the ARN of SNS:

```python
_, _, service, region, account, resource = arn.split(":", 5)
```

In general it should start with `arn:aws:`, then followed by service, region, and account.

### How to obtain the ARN of a queue

```bash
awslocal sqs get-queue-attributes --attribute-names All --queue-url http://localstack:4566/000000000000/simple-event-test
```

### How to empty an SQS queue

```bash
awslocal sqs purge-queue --queue-url http://localhost.localstack.cloud:4566/000000000000/simple-event-test
```

### Missing attributes when receiving messages from SNS subscription

Most likely `RawMessageDelivery` is not set to true.

```
awslocal sqs receive-message --attribute-names All --message-attribute-names All --queue-url http://localhost.localstack.cloud:4566/000000000000/simple-event-test
awslocal sqs purge-queue --queue-url http://localhost.localstack.cloud:4566/000000000000/simple-event-test
```

### Missing environment variables

```bash
export AWS_PROFILE=localstack
export AWS_ACCESS_KEY_ID=$(aws --profile localstack configure get aws_access_key_id)
export AWS_SECRET_ACCESS_KEY=$(aws --profile localstack configure get aws_secret_access_key)
export AWS_DEFAULT_REGION=$(aws --profile localstack configure get region)
export AWS_ENDPOINT_URL=$(aws --profile localstack configure get endpoint_url)
```

### TODO

To run the tests:

```bash
docker compose run test
```
