<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent;

use ArgumentCountError;
use Awardit\SimpleEvent\EventInterface;
use Awardit\SimpleEvent\Event\{
    AutoRevisionedEventInterface,
    ErpProduct,
};
use Awardit\SimpleEvent\V1\ReceivedMessage;
use JsonSerializable;
use PHPUnit\Framework\TestCase;
use TypeError;

class ErpProductTest extends TestCase
{
    public function testEventBasics(): void
    {
        $event = new ErpProduct(
            'ABC-123', // sku
            'physical', // type
            false, // archived
            true, // sellable
            true, // visible
            false, // allowShortPick
            'default', // taxClass
            'Test Product', // name (default: null)
            12.34, // weight (default: null)
            'Test Manufacturer', // manufacturer (deprecated default: null)
            'DEF-456', // mpn (deprecated default: null)
            'Test Brand', // brand (default: null)
            'Test Supplier', // supplier (default: null)
            'GHI-789', // supplierSku (default: null)
            [
                ['currency' => 'SEK', 'price' => '123.45'],
                ['currency' => 'EUR', 'price' => '234.56'],
            ], // msrp (default: [])
            ['currency' => 'SEK', 'price' => '123.45'], // purchasePrice (default: null)
            [
                'Electronics/Lighting/Kits',
                'Security/Miscellaneous',
                'Campaign products',
            ], // categories (default: [])
            [
                ['currency' => 'SEK', 'price' => '123.45'],
                ['currency' => 'EUR', 'price' => '234.56'],
            ], // purchasePrices (default: [])
        );
        $this->assertInstanceof(AutoRevisionedEventInterface::class, $event);
        $this->assertInstanceof(EventInterface::class, $event);
        $this->assertInstanceof(JsonSerializable::class, $event);
        $this->assertEquals('AwarditErpProductV1', $event->getMessageType());
        $this->assertEquals('awardit-erp-product', $event->getMessageTopic());
        $this->assertEquals('ABC-123', $event->getEntityId());
        $this->assertFalse($event->getIsDeleted());
        $this->assertEquals([
            'sku' => 'ABC-123',
            'name' => 'Test Product',
            'type' => 'physical',
            'weight' => 12.34,
            'archived' => false,
            'sellable' => true,
            'visible' => true,
            'allowShortPick' => false,
            'taxClass' => 'default',
            'manufacturer' => 'Test Manufacturer',
            'mpn' => 'DEF-456',
            'brand' => 'Test Brand',
            'supplier' => 'Test Supplier',
            'supplierSku' => 'GHI-789',
            'msrp' => [
                ['currency' => 'SEK', 'price' => '123.45'],
                ['currency' => 'EUR', 'price' => '234.56'],
            ],
            'purchasePrice' => ['currency' => 'SEK', 'price' => '123.45'],
            'categories' => [
                'Electronics/Lighting/Kits',
                'Security/Miscellaneous',
                'Campaign products',
            ],
            'purchasePrices' => [
                ['currency' => 'SEK', 'price' => '123.45'],
                ['currency' => 'EUR', 'price' => '234.56'],
            ],
        ], $event->formatMessagePayload());
    }

    public function testEventDefaults(): void
    {
        $event = new ErpProduct(
            'ABC-123', // sku
            'physical', // type
            false, // archived
            true, // sellable
            true, // visible
            false, // allowShortPick
            'default', // taxClass
        );
        $this->assertEquals([
            'sku' => 'ABC-123',
            'name' => null,
            'type' => 'physical',
            'weight' => null,
            'archived' => false,
            'sellable' => true,
            'visible' => true,
            'allowShortPick' => false,
            'taxClass' => 'default',
            'manufacturer' => null,
            'mpn' => null,
            'brand' => null,
            'supplier' => null,
            'supplierSku' => null,
            'msrp' => [],
            'purchasePrice' => null,
            'categories' => [],
            'purchasePrices' => [],
        ], $event->formatMessagePayload());
    }

    public function testPurchasePricePropagation(): void
    {
        $event = new ErpProduct(
            sku: 'ABC-123', // sku
            type: 'physical', // type
            archived: false, // archived
            sellable: true, // sellable
            visible: true, // visible
            allowShortPick: false, // allowShortPick
            taxClass: 'default', // taxClass
            purchasePrice: ['currency' => 'SEK', 'price' => '123.45'],
        );
        $this->assertEquals([
            'sku' => 'ABC-123',
            'name' => null,
            'type' => 'physical',
            'weight' => null,
            'archived' => false,
            'sellable' => true,
            'visible' => true,
            'allowShortPick' => false,
            'taxClass' => 'default',
            'manufacturer' => null,
            'mpn' => null,
            'brand' => null,
            'supplier' => null,
            'supplierSku' => null,
            'msrp' => [],
            'purchasePrice' => ['currency' => 'SEK', 'price' => '123.45'],
            'categories' => [],
            'purchasePrices' => [
                ['currency' => 'SEK', 'price' => '123.45'],
            ],
        ], $event->formatMessagePayload());
    }

    public function testFromMessage(): void
    {
        $message = new ReceivedMessage([
            'version' => 1,
            'meta' => [
                'revision' => 2,
            ],
            'payload' => [
               'sku' => 'ABC-123',
               'type' => 'physical',
               'archived' => false,
               'sellable' => true,
               'visible' => true,
               'allowShortPick' => false,
               'taxClass' => 'default',
            ],
        ]);

        $event = ErpProduct::fromReceivedMessage($message);
        $this->assertEquals([
            'sku' => 'ABC-123',
            'name' => null,
            'type' => 'physical',
            'weight' => null,
            'archived' => false,
            'sellable' => true,
            'visible' => true,
            'allowShortPick' => false,
            'taxClass' => 'default',
            'manufacturer' => null,
            'mpn' => null,
            'brand' => null,
            'supplier' => null,
            'supplierSku' => null,
            'msrp' => [],
            'purchasePrice' => null,
            'categories' => [],
            'purchasePrices' => [],
        ], $event->formatMessagePayload());
    }

    public function testEmptyCreate(): void
    {
        $this->expectException(ArgumentCountError::class);
        /** @psalm-suppress TooFewArguments */
        new ErpProduct();
    }

    public function testInvalidCreate(): void
    {
        $this->expectException(TypeError::class);
        /** @psalm-suppress InvalidArgument */
        new ErpProduct(1, 2, 3, 4, 5, 6, 7);
    }
}
