<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent;

/**
 * An event representation, find concrete implementations in the
 * \Awardit\SimpleEvent\Event namespace.
 *
 * Preferably the methods of the implementation should be used
 * by producers and consumers of the events.
 */
interface EventInterface
{
    /**
     * The entity id of the data contained in the event.
     *
     * @api
     */
    public function getEntityId(): string;
    /**
     * The revision of the entity contained in the event data.
     *
     * @api
     */
    public function getRevision(): int;
    /**
     * Returns true if the content of the message represents a deleted item.
     *
     * @api
     */
    public function getIsDeleted(): bool;

    /**
     * This static method validates an incoming message and construct an
     * instance of the event.
     *
     * @internal Convers received messages into concrete implementations
     * @psalm-internal Awardit\SimpleEvent
     */
    public static function fromReceivedMessage(ReceivedMessageInterface $message): EventInterface;
    /**
     * The on-wire event data type associated with this class.
     *
     * Note that event type != event topic, a topic can contain multiple event
     * types with the same data in different configurations. This allows us to
     * do graceful updates across services.
     *
     * @internal Identifies the type of event on the wire in EventEmitter and EventHandler
     * @psalm-internal Awardit\SimpleEvent
     */
    public static function getMessageType(): string;
    /**
     * The event topic associated with this class.
     *
     * @internal
     * @api
     * @psalm-internal Awardit\SimpleEvent
     */
    public static function getMessageTopic(): string;

    /**
     * Formats the event into plain event data for on-wire representation.
     *
     * NOTE: The EventEmitter will use getEntityId, getRevision, getIsDeleted
     *       and provide that in metadata.
     *
     * @internal Used by awardit\simple-event when constructing the on-wire data.
     * @psalm-internal Awardit\SimpleEvent
     */
    public function formatMessagePayload(): array;
}
