<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event\ProductPricingPrice;

/**
 * A specific price of a given type for a product.
 *
 * @psalm-type Payload array{
 *   type:string,
 *   customerGroup?:?string,
 *   currency:string,
 *   price:string,
 *   lowestPrice30days:string
 * }
 * @readonly
 */
class Price
{
    /**
     * Human-readable price type identifier.
     *
     * Examples:
     *   * "client"
     *   * "endCustomer"
     *   * "extraFreightCost"
     */
    public string $type;
    /**
     * Currency in 3-letter ISO, or name of point price.
     *
     * Examples:
     *   * "SEK"
     *   * "EUR"
     *   * "awardit_points"
     */
    public string $currency;
    /**
     * Product price for the type, currency, and customer group, as a
     * float-formatted string, `^\d+(\.\d+)?$`.
     *
     * Examples:
     *   * "0.2"
     *   * "199.0"
     *   * "534.49"
     */
    public string $price;
    /**
     * Lowest product price for the type, currency, and customer group,
     * during last 30 days as a float-formatted string, `^\d+(\.\d+)?$`.
     */
    public string $lowestPrice30days;
    /**
     * Human-readable customer group identifier, null if default price.
     *
     * Examples:
     *   * "diamond"
     */
    public ?string $customerGroup = null;

    /**
     * @param Payload $data
     */
    public function __construct(
        array $data
    ) {
        $this->type = $data["type"];
        $this->customerGroup = $data["customerGroup"] ?? null;
        $this->currency = $data["currency"];
        $this->price = $data["price"];
        $this->lowestPrice30days = $data["lowestPrice30days"];
    }

    /**
     * @return Payload
     */
    public function formatMessagePayload(): array
    {
        return [
            "type" => $this->type,
            "customerGroup" => $this->customerGroup,
            "currency" => $this->currency,
            "price" => $this->price,
            "lowestPrice30days" => $this->lowestPrice30days,
        ];
    }
}
