<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event;

use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\EventInterface;
use Awardit\SimpleEvent\Event\ProductPricingPrice\Price;
use Awardit\SimpleEvent\Event\ProductPricingPrice\Campaign;

/**
 * Event describing a price update for a product in a given store.
 *
 * @psalm-import-type Payload from Price as PricePayload
 * @psalm-import-type Payload from Campaign as CampaignPayload
 * @psalm-type Payload array{
 *   priceList: string,
 *   sku: string,
 *   archived: bool,
 *   prices: list<PricePayload>,
 *   campaign: ?CampaignPayload
 * }
 * @psalm-type Data array{
 *   sku: string,
 *   priceList: string,
 *   revision: int,
 *   archived: bool,
 *   prices: list<Price>,
 *   campaign?: ?Campaign
 * }
 * @readonly
 */
class ProductPricingPrice implements EventInterface
{
    public static function getMessageType(): string
    {
        return "AwarditPricingProductPriceV1";
    }

    public static function getMessageTopic(): string
    {
        return "awardit-pricing-product-price";
    }

    public static function fromReceivedMessage(ReceivedMessageInterface $message): self
    {
        /** @var Payload */
        $payload = $message->getEventPayload();

        return new self([
            "sku" => $payload["sku"],
            "priceList" => $payload["priceList"],
            "revision" => $message->getEventRevision(),
            "archived" => $payload["archived"],
            "prices" => array_map(
                /**
                 * @param PricePayload $data
                 */
                fn (array $data): Price => new Price($data),
                $payload["prices"]
            ),
            "campaign" => empty($payload["campaign"]) ? null : new Campaign($payload["campaign"]),
        ]);
    }

    /**
     * Separator for the entity id, may not be present in SKU or priceList.
     */
    public const ENTITY_ID_SEPARATOR = ":";

    /**
     * Product Stock Keeping Unit.
     *
     * May not contain the id-separator ":".
     */
    public string $sku;
    /**
     * Human-readable price list code for the price list which this
     * product-price is part of.
     *
     * May not contain the id-separator ":".
     *
     * Examples:
     *   * "sas_sek"
     *   * "sgdsextra_nok"
     */
    public string $priceList;
    /**
     * Product price revision for the particular product in the price list.
     */
    public int $revision;
    /**
     * If this price-list entry is archived.
     */
    public bool $archived;
    /**
     * List of prices for this particular product and price list.
     *
     * @var list<Price>
     */
    public array $prices;
    /**
     * Currently active campaign, if any.
     *
     * An active campaign indicates that we might be required to show
     * lowestPrice30days where it differs from the current price.
     */
    public ?Campaign $campaign = null;

    /**
     * @param Data $data
     */
    public function __construct(
        array $data
    ) {
        assert($data["revision"] > 0);
        assert(strlen($data["sku"]) > 0);
        assert(strlen($data["priceList"]) > 0);
        assert(strpos($data["sku"], self::ENTITY_ID_SEPARATOR) === false);
        assert(strpos($data["priceList"], self::ENTITY_ID_SEPARATOR) === false);

        $this->sku = $data["sku"];
        $this->priceList = $data["priceList"];
        $this->revision = $data["revision"];
        $this->archived = $data["archived"];
        $this->prices = $data["prices"];
        $this->campaign = $data["campaign"] ?? null;
    }

    public function getEntityId(): string
    {
        return $this->priceList . self::ENTITY_ID_SEPARATOR . $this->sku;
    }

    public function getIsDeleted(): bool
    {
        return $this->archived;
    }

    public function getRevision(): int
    {
        return $this->revision;
    }

    /**
     * @return Payload
     */
    public function formatMessagePayload(): array
    {
        return [
            "priceList" => $this->priceList,
            "sku" => $this->sku,
            "archived" => $this->archived,
            "prices" => array_map(fn (Price $x): array => $x->formatMessagePayload(), $this->prices),
            "campaign" => $this->campaign ? $this->campaign->formatMessagePayload() : null,
        ];
    }
}
