#!/usr/bin/env php
<?php

/**
 * Example which publishes an event on AWS Simple Notification Service with
 * automatic revision-assignment, and then consumes the same event from a
 * Simple Queue Service.
 *
 * Parameters in environment variables:
 *  * TOPIC_ARN_AWARDIT_TEST_EVENT: The ARN for the topic to publish to
 *  * REDIS_HOST: The host for Redis used to store event snapshots/revisions
 *  * SQS_QUEUE_URL: The queue url to read from
 */

declare(strict_types=1);

use Awardit\Aws\Logger;
use Awardit\Aws\Util;
use Awardit\Aws\Env;
use Awardit\SimpleEvent\Aws\SnsEventEmitter;
use Awardit\SimpleEvent\Aws\SqsEventListener;
use Awardit\SimpleEvent\Aws\TopicsFromEnv;
use Awardit\SimpleEvent\EventEmitter\AutoRevisionedEmitter;
use Awardit\SimpleEvent\EventHandlerInterface;
use Awardit\SimpleEvent\EventInterface;
use Awardit\SimpleEvent\EventRegistry;
use Awardit\SimpleEvent\Event\TestEvent;
use Awardit\SimpleEvent\Metadata;
use Aws\Sdk;

require_once dirname(__DIR__) . "/vendor/autoload.php";

// Register error logger
Logger::registerErrorHandler();
Env::loadDotenv(dirname(__DIR__));

$sqsQueue = Env::mustGetValue("SQS_QUEUE_URL");
$redisHost = Env::mustGetValue("REDIS_HOST");
$log = Logger::getLogger("app");
$sdk = new Sdk(Util::getAwsSdkCommonConfig());
$redis = new Redis([
    "host" => $redisHost,
]);
$emitterInner = new SnsEventEmitter(
    log: $log,
    client: $sdk->createSns(),
    topics: new TopicsFromEnv(),
    sender: "example-simple-event-auto-revision",
    correlationId: bin2hex(random_bytes(32)),
);
$listener = new SqsEventListener(
    log: $log,
    client:$sdk->createSqs(),
    queueUrl: $sqsQueue,
    registry: EventRegistry::withDefaults(),
);
$emitter = new AutoRevisionedEmitter($log, $emitterInner, $redis);

$emitter->emit(new TestEvent([
    "id" => "a-test-event-id",
    "data" => [
        "some data" => "this is test message",
    ],
]));

$listener->receiveEvents(new class implements EventHandlerInterface {
    public function handle(EventInterface $event, Metadata $meta): bool
    {
        var_dump($event);

        // True indicates that we are done processing the message and want to delete it
        return true;
    }
});
