<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\EventEmitter;

use Awardit\Aws\Json;
use Awardit\SimpleEvent\EventEmitterInterface;
use Awardit\SimpleEvent\Event\AutoRevisionedEventInterface;
use Awardit\SimpleEvent\Exception;
use Psr\Log\LoggerInterface;
use Redis;
use Throwable;

// TODO: S3 variant? PostgreSQL variant?
/**
 * Utility for assigning revisions to event-data where revisions need to be
 * retrofitted without modifying the datastructure.
 *
 * It automatically assigns revisions to events based on their contents
 * compared to the previously emimtted event for the given entity.
 *
 * NOTE: If the system has a way to keep track of revisions internally then
 *       this utility should NOT be used.
 */
class AutoRevisionedEmitter
{
    /**
     * Compare and swap based on revision and event data, only updating if the
     * revision and old data matches.
     *
     * Arguments:
     *  * Key containing revision
     *  * Key containing event data
     *  * Key for storing the datetime
     *  * Current revision
     *  * Old event data
     *  * New event data
     *  * Current unix datetime
     *
     * Usage:
     *
     * ```
     * $redis->eval(self::LUA_CAS_REVISION_VALUE, [
     *     $kRev, $kEvent, $kTime,
     *     $rev, $oldEvent, $event, $time
     * ], 3);
     * ```
     *
     * @var string
     */
    private const LUA_CAS_REVISION_VALUE = <<<'LUA'
local currRevision = redis.call("get", KEYS[1]);
local currEvent = redis.call("get", KEYS[2]);

if (currRevision == false or tonumber(currRevision) == tonumber(ARGV[1])) and
   (currEvent == false or currEvent == ARGV[2]) then
    redis.call("set", KEYS[2], ARGV[3]);
    redis.call("set", KEYS[3], ARGV[4]);

    return "ok"
else
    return currRevision
end
LUA;

    /**
     * @api
     */
    public function __construct(
        private LoggerInterface $log,
        private EventEmitterInterface $emitter,
        // TODO: Use an adapter interface
        private Redis $redis,
        private string $prefix = "sear:"
    ) {
    }

    /**
     * Attempts to emit an event with an automatically assigned revision, if
     * the data does not differ compared to previous data emitted for the
     * entity then no event is emitted.
     *
     * @param array{forceResendDebug?:bool} $options
     * @api
     */
    public function emit(AutoRevisionedEventInterface $event, array $options = []): void
    {
        $forceResendDebug = $options["forceResendDebug"] ?? false;
        /** @var string */
        $type = call_user_func([get_class($event), "getMessageType"]);
        $id = $event->getEntityId();
        // We skip the revision here when comparing
        // TODO: Make sure to grab the relevant fields (and skip the fields
        // which depend on the sender/time of event creation)
        $meta = [
            "id" => $id,
            "deleted" => $event->getIsDeleted(),
        ];
        $payload = $event->formatMessagePayload();
        $eventData = [
            "meta" => $meta,
            "payload" => $payload,
        ];

        // Wrap the key in brackets to instruct how it is hashed so all our
        // keys end up in the same hash slot
        $primaryKey = "{" . $this->prefix . $type . ":" . $id . "}";
        $revisionKey = $primaryKey . ":revision";
        $eventKey = $primaryKey . ":event";
        $timeKey = $primaryKey . ":time";

        // First just check if we need to send the event in the first place
        /** @var false|string */
        $prevEvent = $this->redis->get($eventKey);

        if ($prevEvent !== false) {
            $oldEventData = null;

            try {
                $oldEventData = Json::decode($prevEvent);
            } catch (Throwable $t) {
                $this->log->error(
                    "Failed to decode JSON for stored data for {event.type}({event.id}): {exception}",
                    [
                        "method" => __METHOD__,
                        "event.type" => $type,
                        "event.id" => $id,
                        "exception" => $t,
                    ],
                );

                // Just send the event anyway, since we will overwrite this key
            }

            // Deep comparison of data, fails if JSON parsing failed
            if ($oldEventData == $eventData) {
                if ($forceResendDebug) {
                    /** @var false|string */
                    $revision = $this->redis->get($revisionKey);

                    assert($revision !== false);

                    $event->setRevision((int)$revision);

                    $this->log->notice(
                        // phpcs:ignore Generic.Files.LineLength.TooLong
                        "Event {event.type}({event.id}) re-emitted due to forceResendDebug with revision {event.revision}",
                        [
                            "method" => __METHOD__,
                            "event.type" => $type,
                            "event.id" => $id,
                            "event.revision" => $revision,
                        ]
                    );

                    $this->emitter->emit($event);
                } else {
                    $this->log->notice(
                        "Event {event.type}({event.id}) skipped emit due to no differences with previous data",
                        [
                        "method" => __METHOD__,
                        "event.type" => $type,
                        "event.id" => $id,
                        ]
                    );
                }

                return;
            }
        }

        $nextEvent = Json::encode($eventData);
        $time = time();

        // Increment revision atomically so we only ever send the revision once
        /** @var false|int */
        $revision = $this->redis->incr($revisionKey);

        if ($revision === false) {
            throw new Exception(sprintf(
                "%s: Failed to increment revision in redis: %s",
                __METHOD__,
                $this->redis->getLastError() ?? "<UNKNOWN>"
            ));
        }

        $this->log->notice("Event {event.type}({event.id}) got auto-assigned revision {event.revision}", [
            "method" => __METHOD__,
            "event.type" => $type,
            "event.id" => $id,
            "event.revision" => $revision,
        ]);

        $event->setRevision($revision);

        $this->emitter->emit($event);

        /** @var false|string */
        $res = $this->redis->eval(self::LUA_CAS_REVISION_VALUE, [
            $revisionKey,
            $eventKey,
            $timeKey,
            $revision,
            $prevEvent,
            $nextEvent,
            $time,
        ], 3);

        if ($res === false) {
            throw new Exception(sprintf(
                "%s: Failed to save event data in redis: %s",
                __METHOD__,
                $this->redis->getLastError() ?? "<UNKNOWN>"
            ));
        }

        if ($res !== "ok") {
            // If we fail here it means we might have sent an extra event with
            // the same data, or maybe with differing data. The main cause is
            // multiple senders reading the same data instead of using some
            // kind of lock or sharding to avoid sending data related to the
            // same entities.
            $this->log->error(
                // phpcs:ignore Generic.Files.LineLength.TooLong
                "Race condition detected while emitting event {event.type}({event.id}) revision {event.revision}, conflicting revision {conflictingRevision} saved during emit",
                [
                    "method" => __METHOD__,
                    "event.type" => $type,
                    "event.id" => $id,
                    "event.revision" => $revision,
                    "conflictingRevision" => (int)$res,
                ]
            );
        }
    }
}
