<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event;

use JsonSerializable;
use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\EventInterface;

/**
 * Event describing basic ERP information for a product.
 *
 * @psalm-type TaxClass "default"
 * @psalm-type ProductType "physical"|"virtual"|"dropship"
 * @psalm-type Price array{price:string, currency:string}
 * @psalm-type Payload array{
 *   sku?: string,
 *   name?: ?string,
 *   type: ProductType,
 *   weight: ?float,
 *   archived: bool,
 *   sellable: bool,
 *   visible: bool,
 *   taxClass: TaxClass,
 *   allowShortPick: bool,
 *   manufacturer?: ?string,
 *   mpn?: ?string,
 *   brand?: ?string,
 *   supplier?: ?string,
 *   supplierSku?: ?string,
 *   msrp: list<Price>,
 *   purchasePrice?: ?Price,
 *   categories: list<string>,
 *   purchasePrices?: ?list<Price>,
 * }
 * @psalm-type ConstructorProps array{
 *   sku: string,
 *   name?: ?string,
 *   type: ProductType,
 *   weight?: ?float,
 *   archived: bool,
 *   sellable: bool,
 *   visible: bool,
 *   taxClass: TaxClass,
 *   allowShortPick: bool,
 *   manufacturer?: ?string,
 *   mpn?: ?string,
 *   brand?: ?string,
 *   supplier?: ?string,
 *   supplierSku?: ?string,
 *   msrp?: list<Price>,
 *   purchasePrice?: ?Price,
 *   categories?: list<string>,
 *   purchasePrices?: ?list<Price>,
 * }
 */
class ErpProduct implements AutoRevisionedEventInterface, EventInterface, JsonSerializable
{
    use AutoRevisionedEventTrait;

    public static function getMessageType(): string
    {
        return "AwarditErpProductV1";
    }

    public static function getMessageTopic(): string
    {
        return "awardit-erp-product";
    }

    public static function fromReceivedMessage(ReceivedMessageInterface $message): self
    {
        /** @var Payload */
        $payload = $message->getEventPayload();

        $event = new self([
            "sku" => $payload["sku"] ?? $message->getEventEntityId(),
            "name" => $payload["name"] ?? null,
            "type" => $payload["type"],
            "weight" => $payload["weight"] ?? null,
            "archived" => $payload["archived"],
            "sellable" => $payload["sellable"],
            "visible" => $payload["visible"],
            "taxClass" => $payload["taxClass"],
            "allowShortPick" => $payload["allowShortPick"],
            "manufacturer" => $payload["manufacturer"] ?? null,
            "mpn" => $payload["mpn"] ?? null,
            "brand" => $payload["brand"] ?? null,
            "supplier" => $payload["supplier"] ?? null,
            "supplierSku" => $payload["supplierSku"] ?? null,
            "msrp" => $payload["msrp"] ?? [],
            "purchasePrice" => $payload["purchasePrice"] ?? null,
            "categories" => $payload["categories"] ?? [],
            "purchasePrices" => $payload["purchasePrices"] ?? array_filter([$payload["purchasePrice"] ?? null]),
        ]);

        $event->setRevision($message->getEventRevision());

        return $event;
    }

    /**
     * Product Stock Keeping Unit.
     *
     * @readonly
     */
    public string $sku;
    /**
     * Product name.
     *
     * @readonly
     */
    public ?string $name;
    /**
     * @var ProductType
     *
     * @readonly
     */
    public string $type;
    /**
     * Product weight in kg, for physical products.
     *
     * @readonly
     */
    public ?float $weight;
    /**
     * True if the product is archived.
     *
     * @readonly
     */
    public bool $archived;
    /**
     * If we can sell the product.
     *
     * @readonly
     */
    public bool $sellable;
    /**
     * If the product is visible to the customer.
     *
     * @readonly
     */
    public bool $visible;
    /**
     * If we allow sales despite not having enouch in stock.
     *
     * @readonly
     */
    public bool $allowShortPick;
    /**
     * Manufacturer name.
     *
     * @readonly
     */
    public ?string $manufacturer;
    /**
     * Manufacturer Part Number.
     *
     * @readonly
     */
    public ?string $mpn;
    /**
     * Brand name.
     *
     * @readonly
     */
    public ?string $brand;
    /**
     * Supplier name.
     *
     * @readonly
     */
    public ?string $supplier;
    /**
     * Supplier stock keeping unit id.
     *
     * @readonly
     */
    public ?string $supplierSku;
    /**
     * Tax classification of the product.
     *
     * @readonly
     * @var TaxClass
     */
    public string $taxClass;
    /**
     * Manufacturer suggested retail prices.
     *
     * The price-key is a string-formatted decimal.
     *
     * @readonly
     * @var list<Price>
     */
    public array $msrp;
    /**
     * The purchase price.
     *
     * The price-key is a string-formatted decimal.
     *
     * @readonly
     * @var ?Price
     */
    public ?array $purchasePrice;

    /**
     * List of categories the product belongs to, paths with `/` as level separator.
     *
     * Example:
     *
     * ```
     * [
     *    "Electronics/Lighting/Kits",
     *    "Security/Miscellaneous",
     *    "Campaign products",
     * ]
     * ```
     *
     * @readonly
     * @var list<string>
     */
    public array $categories;

    /**
     * Purchase price by currency.
     *
     * The price-key is a string-formatted decimal.
     *
     * @readonly
     * @var list<Price>
     */
    public array $purchasePrices;

    /**
     * @param ConstructorProps $payload
     */
    public function __construct(array $payload)
    {
        assert(strlen($payload["sku"]) > 0);

        $this->sku = $payload["sku"];
        $this->name = $payload["name"] ?? null;
        $this->type = $payload["type"];
        $this->weight = $payload["weight"] ?? null;
        $this->archived = $payload["archived"];
        $this->sellable = $payload["sellable"];
        $this->visible = $payload["visible"];
        $this->allowShortPick = $payload["allowShortPick"];
        $this->taxClass = $payload["taxClass"];
        $this->manufacturer = $payload["manufacturer"] ?? null;
        $this->mpn = $payload["mpn"] ?? null;
        $this->brand = $payload["brand"] ?? null;
        $this->supplier = $payload["supplier"] ?? null;
        $this->supplierSku = $payload["supplierSku"] ?? null;
        $this->msrp = $payload["msrp"] ?? [];
        $this->purchasePrice = $payload["purchasePrice"] ?? null;
        $this->categories = $payload["categories"] ?? [];
        $this->purchasePrices = $payload["purchasePrices"] ?? array_filter([$this->purchasePrice]);
    }

    public function getEntityId(): string
    {
        return $this->sku;
    }

    public function getIsDeleted(): bool
    {
        return $this->archived;
    }

    /**
     * @return Payload
     */
    public function formatMessagePayload(): array
    {
        return [
            "sku" => $this->sku,
            "name" => $this->name,
            "type" => $this->type,
            "weight" => $this->weight,
            "archived" => $this->archived,
            "sellable" => $this->sellable,
            "visible" => $this->visible,
            "allowShortPick" => $this->allowShortPick,
            "taxClass" => $this->taxClass,
            "manufacturer" => $this->manufacturer,
            "mpn" => $this->mpn,
            "brand" => $this->brand,
            "supplier" => $this->supplier,
            "supplierSku" => $this->supplierSku,
            "msrp" => $this->msrp,
            "purchasePrice" => $this->purchasePrice,
            "categories" => $this->categories,
            "purchasePrices" => $this->purchasePrices,
        ];
    }

    public function jsonSerialize(): array
    {
        return get_object_vars($this);
    }
}
