#!/usr/bin/env php
<?php

// phpcs:disable PSR1.Files.SideEffects.FoundWithSymbols
// phpcs:disable PSR1.Classes.ClassDeclaration.MissingNamespace

/**
 * Example which publishes a locally defined event directly on an AWS Simple
 * Queue Service, reading from the same immediately.
 *
 * Parameters in environment variables:
 *  * SQS_QUEUE_URL: The queue url to send and read from
 */

declare(strict_types=1);

use Awardit\Aws\Logger;
use Awardit\Aws\Util;
use Awardit\Aws\Env;
use Awardit\SimpleEvent\Aws\SqsEventListener;
use Awardit\SimpleEvent\Aws\SqsEventEmitter;
use Awardit\SimpleEvent\EventHandlerInterface;
use Awardit\SimpleEvent\EventInterface;
use Awardit\SimpleEvent\EventRegistry;
use Awardit\SimpleEvent\Event\AbstractInternalEvent;
use Awardit\SimpleEvent\Metadata;
use Awardit\SimpleEvent\ReceivedMessageInterface;
use Aws\Sdk;

require_once dirname(__DIR__) . "/vendor/autoload.php";

// Create our internal event, extending AbstractInternalEvent to get some things for free.
/**
 * @psalm-type Payload array{value:string,another:int}
 * @psalm-type Data array{id:string,value:string,another:int}
 */
class MyInternalEvent extends AbstractInternalEvent
{
    public static function fromReceivedMessage(ReceivedMessageInterface $message): self
    {
        /** @var Payload */
        $payload = $message->getEventPayload();

        return new self([
            "id" => $message->getEventEntityId(),
            "value" => $payload["value"],
            "another" => $payload["another"],
        ]);
    }

    private string $id;
    private string $value;
    private int $another;

    /**
     * @param Data $data
     */
    public function __construct(
        array $data
    ) {
        $this->id = $data["id"];
        $this->value = $data["value"];
        $this->another = $data["another"];

        parent::__construct();
    }

    public function getEntityId(): string
    {
        return $this->id;
    }

    /**
     * @return Payload
     */
    public function formatMessagePayload(): array
    {
        return [
            "value" => $this->value,
            "another" => $this->another,
        ];
    }
}

Logger::registerErrorHandler();
Env::loadDotenv(dirname(__DIR__));

$sqsQueue = Env::mustGetValue("SQS_QUEUE_URL");
$log = Logger::getLogger("simple-event");
$sdk = new Sdk(Util::getAwsSdkCommonConfig());

$emitter = new SqsEventEmitter(
    log: $log,
    client: $sdk->createSqs(),
    queueUrl: $sqsQueue,
    sender: "example-simple-event-internal",
    correlationId: bin2hex(random_bytes(32)),
);
$listener = new SqsEventListener(
    log: $log,
    client:$sdk->createSqs(),
    queueUrl: $sqsQueue,
    registry: new EventRegistry([...EventRegistry::DEFAULT_TYPES, MyInternalEvent::class]),
);

$emitter->emit(new MyInternalEvent([
    "id" => "a-test-event-id",
    "value" => "aaa",
    "another" => 3,
]));

$listener->receiveEvents(new class implements EventHandlerInterface {
    public function handle(EventInterface $event, Metadata $meta): bool
    {
        var_dump($event);

        // True indicates that we are done processing the message and want to delete it
        return true;
    }
});
