<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Aws;

use Awardit\Aws\Json;
use Awardit\SimpleEvent\EventEmitterInterface;
use Awardit\SimpleEvent\EventInterface;
use Awardit\SimpleEvent\V1\EventMessage;
use Aws\Sqs\SqsClient;
use Psr\Log\LoggerInterface;

/**
 * @psalm-import-type Options from EventEmitterInterface
 */
class SqsEventEmitter implements EventEmitterInterface
{
    public function __construct(
        private LoggerInterface $log,
        private SqsClient $client,
        /**
         * The URL of the AWS Simple Queue Service queue to send messages on.
         */
        private string $queueUrl,
        /**
         * The name of the service emitting events.
         */
        private string $sender,
        /**
         * Default tracking id of the service this emitter is created in.
         */
        private string $correlationId,
        /**
         * Attributes to set on all emitted events.
         *
         * @var Array<string, string>
         */
        private array $attributes = [],
    ) {
    }

    public function setCorrelationId(string $correlationId): void
    {
        $this->correlationId = $correlationId;
    }

    /**
     * @param Options|array{delaySeconds:int} $options
     */
    public function emit(EventInterface $event, array $options = []): void
    {
        $message = new EventMessage(
            event: $event,
            sender: $this->sender,
            correlationId: $options["correlationId"] ?? $this->correlationId,
            attributes: $this->attributes
        );
        $jsonBody = Json::encode($message->formatMessage());

        // TODO: Detect FIFO
        $delaySeconds = max(0, $options["delaySeconds"] ?? 0);

        $this->log->debug("Sending message {subject} to AWS SQS queue {queueUrl}", [
            "subject" => $message->getMessageSubject(),
            "method" => __METHOD__,
            "queueUrl" => $this->queueUrl,
        ]);

        $result = $this->client->sendMessage([
            "DelaySeconds" => $delaySeconds,
            "MessageBody" => $jsonBody,
            "QueueUrl" => $this->queueUrl,
            "MessageAttributes" => Helper::buildAttributes($message->getMessageAttributes()),
        ]);

        $this->log->info("Emitted event {messageId} on queue {queueUrl}", [
            "subject" => $message->getMessageSubject(),
            "method" => __METHOD__,
            "messageId" => $result->get("MessageId"),
            "queueUrl" => $this->queueUrl,
            "sequenceNumber" => $result->get("SequenceNumber"),
        ]);
    }
}
