<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\V1;

use Awardit\SimpleEvent\Attributes;
use Awardit\SimpleEvent\MessageInterface;
use Awardit\SimpleEvent\EventInterface;

/**
 * Concrete message implementation for version 1 of event message construction
 * from EventInterface implementations.
 */
class EventMessage implements MessageInterface
{
    public const VERSION = 1;

    public function __construct(
        private EventInterface $event,
        /**
         * Tracking id of the origin of this event.
         */
        private string $correlationId,
        /**
         * Sending service name.
         */
        private string $sender,
        /**
         * @var Array<string, string>
         */
        private array $attributes,
    ) {
        assert($event->getRevision() >= 1);
    }

    public function getMessageAttributes(): array
    {
        return [
            ...$this->attributes,
            Attributes::SENDER => $this->sender,
            Attributes::DELETED => $this->event->getIsDeleted() ? "true" : "false",
            // We want to be able to filter by EventType
            Attributes::EVENT_TYPE => $this->getMessageEventType(),
        ];
    }

    public function getMessageEventType(): string
    {
        return $this->event::class::getEventType();
    }

    public function getMessageSubject(): string
    {
        return $this->getMessageEventType() . ":" . $this->event->getEntityId();
    }

    public function getMessageDeduplicationId(): string
    {
        return $this->getMessageEventType() . ":" . $this->event->getEntityId() . ":" . $this->event->getRevision();
    }

    public function formatMessage(): array
    {
        return [
            // The event structure version
            "version" => self::VERSION,
            "meta" => [
                // Event related things
                "id" => $this->event->getEntityId(),
                "revision" => $this->event->getRevision(),
                "deleted" => $this->event->getIsDeleted(),
                "eventType" => $this->getMessageEventType(),
                "correlationId" => $this->correlationId,
                "sender" => $this->sender,
            ],
            "payload" => $this->event->getPayload(),
        ];
    }
}
