<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent;

use Awardit\SimpleEvent\Event\ErpProduct;

/**
 * Registry which converts ReceivedMessageInterface instances to Event instances.
 *
 * Example:
 *
 *   new EventRegistry([
 *     ...EventRegistry::DEFAULT_TYPES,
 *     MyAppEvent::class
 *   ]);
 */
class EventRegistry
{
    /**
     * @var list<class-string<EventInterface>>
     */
    public const DEFAULT_TYPES = [
        ErpProduct::class,
    ];

    /**
     * Mapping of event-type name to class instance.
     *
     * @var Array<string, class-string<EventInterface>>
     */
    private array $knownEventTypes = [];

    public static function withDefaults(): self
    {
        return new self(self::DEFAULT_TYPES);
    }

    /**
     * @param list<class-string<EventInterface>> $eventTypes
     */
    public function __construct(
        array $eventTypes
    ) {
        foreach ($eventTypes as $class) {
            $type = $class::getEventType();

            if (array_key_exists($type, $this->knownEventTypes)) {
                throw new Exception(sprintf("Conflicting event type %s", $type));
            }

            $this->knownEventTypes[$type] = $class;
        }
    }

    /**
     * @return class-string<EventInterface>
     */
    public function getEventClassFromType(string $eventType): string
    {
        if (!array_key_exists($eventType, $this->knownEventTypes)) {
            throw new Exception(sprintf(
                "%s: Unknown common event type '%s'",
                __METHOD__,
                $eventType
            ));
        }

        return $this->knownEventTypes[$eventType];
    }

    public function createEvent(ReceivedMessageInterface $message): EventInterface
    {
        $class = $this->getEventClassFromType($message->getMessageEventType());
        $event = $class::fromReceivedMessage($message);

        // Some sanity-checking
        if ($event->getRevision() <= 0) {
            throw new Exception("Invalid non-positive event revision");
        }

        return $event;
    }
}
