<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event;

use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\EventInterface;

/**
 * @psalm-type TaxClass "default"
 * @psalm-type ProductType "physical"|"virtual"
 */
class ErpProduct implements AutoRevisionedEventInterface, EventInterface
{
    use AutoRevisionedEventTrait;

    public static function getEventType(): string
    {
        return "AwarditErpProductV1";
    }

    public static function getEventTopic(): string
    {
        return "awardit-erp-product";
    }

    public static function fromReceivedMessage(ReceivedMessageInterface $message): self
    {
        // TODO: Assert structure and populate
        $payload = $message->getEventPayload();

        $event = new self(
            sku: $message->getEventEntityId(),
            name: $payload["name"] ?? "<UNKNOWN>",
            type: $payload["type"] ?? "physical",
            weight: $payload["weight"] ?? null,
            archived: $payload["archived"] ?? false,
            sellable: $payload["sellable"],
            visible: $payload["visible"] ?? true,
            taxClass: $payload["taxClass"] ?? null,
            allowShortPick: $payload["allowShortPick"] ?? false,
            manufacturer: $payload["manufacturer"] ?? "<UNKNOWN>",
            mpn: $payload["mpn"] ?? "<UNKNOWN>",
            msrp: $payload["msrp"] ?? [],
            purchasePrice: $payload["purchasePrice"] ?? [],
        );

        $event->setRevision($message->getEventRevision());

        return $event;
    }

    public function __construct(
        /**
         * Product Stock Keeping Unit.
         */
        readonly public string $sku,
        /**
         * Product name.
         */
        readonly public string $name,
        /**
         * @var ProductType
         */
        readonly public string $type,
        /**
         * Product weight in kg, for physical products.
         */
        readonly public float|null $weight,
        /**
         * True if the product is archived.
         */
        readonly public bool $archived,
        /**
         * If we can sell the product.
         */
        readonly public bool $sellable,
        /**
         * If the product is visible to the customer.
         */
        readonly public bool $visible,
        /**
         * If we allow sales despite not having enouch in stock.
         */
        readonly public bool $allowShortPick,
        /**
         * Manufacturer name.
         */
        readonly public string $manufacturer,
        /**
         * Manufacturer Part Number.
         */
        readonly public string $mpn,
        /**
         * Tax classification of the product.
         *
         * @var TaxClass
         */
        readonly public string $taxClass,
        /**
         * Manufacturer suggested retail prices.
         *
         * The price-key is a string-formatted decimal.
         *
         * @var list<array{price:string, currency:string}>
         */
        readonly public array $msrp,
        /**
         * The purchase price.
         *
         * The price-key is a string-formatted decimal.
         *
         * @var array{price:string, currency:string}|null
         */
        readonly public array|null $purchasePrice,
    ) {
    }

    public function getEntityId(): string
    {
        return $this->sku;
    }

    public function getIsDeleted(): bool
    {
        return $this->archived;
    }

    public function getPayload(): array
    {
        return [
            "name" => $this->name,
            "type" => $this->type,
            "weight" => $this->weight,
            "archived" => $this->archived,
            "sellable" => $this->sellable,
            "visible" => $this->visible,
            "allowShortPick" => $this->allowShortPick,
            "taxClass" => $this->taxClass,
            "manufacturer" => $this->manufacturer,
            "mpn" => $this->mpn,
            "msrp" => $this->msrp,
            "purchasePrice" => $this->purchasePrice,
        ];
    }
}
