<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Aws;

use Aws\Sns\SnsClient;
use Awardit\Aws\Util;
use Awardit\SimpleEvent\Attributes;
use Awardit\SimpleEvent\EventEmitterInterface;
use Awardit\SimpleEvent\EventInterface;
use Awardit\SimpleEvent\Exception;
use Awardit\SimpleEvent\KnownTypes;
use Awardit\SimpleEvent\Metadata;
use Awardit\SimpleEvent\V1\EventMessage;
use Psr\Log\LoggerInterface;

class SnsEventEmitter implements EventEmitterInterface
{
    /**
     * @api
     */
    public function __construct(
        private LoggerInterface $log,
        private SnsClient $client,
        private TopicsInterface $topics,
        /**
         * The name of the service emitting events.
         */
        private string $sender,
        /**
         * Default tracking id of the service this emitter is created in.
         */
        private string $correlationId,
        /**
         * Attributes to set on all emitted events.
         *
         * @var Array<string, string>
         */
        private array $attributes = [],
    ) {
    }

    /**
     * Sets the correlation id for all following events emitted.
     *
     * @api
     */
    public function setCorrelationId(string $correlationId): void
    {
        $this->correlationId = $correlationId;
    }

    public function emit(EventInterface $event, array $options = []): void
    {
        $topic = $event::class::getEventTopic();
        $message = new EventMessage(
            event: $event,
            sender: $this->sender,
            correlationId: $options["correlationId"] ?? $this->correlationId,
            attributes: $this->attributes
        );

        $jsonBody = Util::jsonEncode($message->formatMessage());
        $topicArn = $this->topics->getTopicArn($topic);

        // NOTE: Very important to get the parameter spelling and structure
        // correct, anything invalid or misspelled will be silently discarded.
        $args = [
            "TopicArn" => $topicArn,
            "Subject" => $message->getMessageSubject(),
            "Message" => $jsonBody,
            "MessageAttributes" => $this->buildAttributes($message->getMessageAttributes()),
        ];
        $logMeta = [
            "method" => __METHOD__,
            "subject" => $message->getMessageSubject(),
            "topicArn" => $topicArn,
        ];

        // AWS FIFO-topics always end with .fifo
        if (str_ends_with($topicArn, ".fifo")) {
            // Only present in FIFO-topics/queues
            $args["MessageDeduplicationId"] = $message->getMessageDeduplicationId();
            // TODO: Maybe use the eventType here in addition to topic?
            $args["MessageGroupId"] = $topic;

            $logMeta["deduplicationId"] = $args["MessageDeduplicationId"];
            $logMeta["groupId"] = $args["MessageGroupId"];
        }

        $this->log->debug("Publishing message {subject} to AWS SNS {topicArn}", $logMeta);

        $result = $this->client->publish($args);

        $this->log->info("Emitted event {messageId} on {topic}", [
            "messageId" => $result->get("MessageId"),
            "sequenceNumber" => $result->get("SequenceNumber"),
            "topic" => $topic,
        ]);
    }

    /**
     * @param Array<string, string|int|float> $attributes
     */
    public function buildAttributes(array $attributes): array
    {
        $messageAttributes = [];

        foreach ($attributes as $k => $v) {
            $messageAttributes[$k] = [
                "DataType" => (is_int($v) || is_float($v)) ? "Number" : "String",
                "StringValue" => (string)$v,
            ];
        }

        return $messageAttributes;
    }
}
