<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Event;

use JsonSerializable;
use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\EventInterface;

/**
 * Minimal product PIM event, information from Visma, to be replaced by a proper PIM.
 * 
 * @psalm-type Payload array{
 *   sku: string,
 *   name: ?string,
 *   archived: ?bool,
 *   channels: string[],
 * }
 */
class MiniPimProduct implements AutoRevisionedEventInterface, EventInterface, JsonSerializable
{
    use AutoRevisionedEventTrait;

    public static function getMessageType(): string
    {
        return "AwarditMinipimProductV1";
    }

    public static function getMessageTopic(): string
    {
        return "awardit-minipim-product";
    }

    public static function fromReceivedMessage(ReceivedMessageInterface $message): self
    {
        /** @var Payload */
        $payload = $message->getEventPayload();

        $event = new self(
            sku: $message->getEventEntityId(),
            name: $payload["name"] ?? null,
            archived: $payload["archived"] ?? false,
            channels: $payload["channels"],
        );

        $event->setRevision($message->getEventRevision());

        return $event;
    }

    public function __construct(
        /**
         * Product Stock Keeping Unit.
         */
        readonly public string $sku,
        /**
         * Product name.
         */
        readonly public ?string $name = null,
        /**
         * If the product is archived.
         */
        readonly public bool $archived = false,
        /**
         * List of channels the product is included in.
         * 
         * Example:
         * ```
         * ["1432", "giftcard", "someshop_se"]
         * ```
         * 
         * @var list<string>
         */
        readonly public array $channels = [],
    ) {
    }

    public function getEntityId(): string
    {
        return $this->sku;
    }

    public function getIsDeleted(): bool
    {
        return $this->archived;
    }

    /**
     * @return Payload
     */
    public function formatMessagePayload(): array
    {
        return [
            "sku" => $this->sku,
            "name" => $this->name,
            "archived" => $this->archived,
            "channels" => $this->channels,
        ];
    }

    public function jsonSerialize(): array
    {
        return get_object_vars($this);
    }
}
