<?php

declare(strict_types=1);

namespace Awardit\SimpleEvent\Aws\Lambda;

use Awardit\Aws\Lambda\HandlerInterface;
use Awardit\Aws\Lambda\Request;
use Awardit\Aws\Json;
use Awardit\SimpleEvent\ErrorEventEmitterInterface;
use Awardit\SimpleEvent\EventHandlerInterface;
use Awardit\SimpleEvent\EventRegistry;
use Awardit\SimpleEvent\Exception;
use Awardit\SimpleEvent\Metadata;
use Awardit\SimpleEvent\ReceivedMessageInterface;
use Awardit\SimpleEvent\V1\ReceivedMessage;
use Psr\Log\LoggerInterface;
use Throwable;

// TODO: Support wrapped message delivery (ie. non-raw), preferably with auto-detect
/**
 * Handler for SQS messages consumed by in a Lambda function.
 *
 * NOTE: Requires the use of ReportBatchItemFailures setting on the
 *       SQSEventSource configuration.
 *
 * @api
 * @psalm-type SqsRecord array{messageId:string, eventSource:string, body:string}
 * @psalm-type ItemFailure array{itemIdentifier:string}
 * @psalm-type SqsHandlerResponse array{batchItemFailures:list<ItemFailure>}
 * @implements HandlerInterface<array{Records?:list<SqsRecord>}, SqsHandlerResponse>
 */
class SqsHandler implements HandlerInterface
{
    public function __construct(
        private LoggerInterface $log,
        private EventRegistry $registry,
        private EventHandlerInterface $eventHandler,
        private bool $debugLogFullPayload = false,
        private ErrorEventEmitterInterface|false $errorEventEmitter = false,
    ) {
    }

    public function handle(Request $request): array
    {
        $failed = [];
        $payload = $request->getPayload();

        if (!array_key_exists("Records", $payload)) {
            throw new Exception(sprintf("%s: Missing Records property in Lambda request", __METHOD__));
        }

        $debugContext = [
            "method" => __METHOD__,
            "count" => count($payload["Records"]),
            "invocationId" => $request->getInvocationId(),
        ];

        if ($this->debugLogFullPayload) {
                $debugContext["payload"] = $payload;
        }

        $this->log->debug("Received {count} messages from Lambda Request {invocationId}", $debugContext);

        foreach ($payload["Records"] as $record) {
            try {
                ["decoded" => $decoded, "meta" => $meta] = $this->decodeMessage($record);
                $event = $this->registry->createEvent($decoded);

                $this->log->debug("Constructed event from received message", [
                    "method" => __METHOD__,
                    "event" => $event,
                    "event.class" => $event::class,
                ]);

                if (!$this->eventHandler->handle($event, $meta)) {
                    $this->log->info("Marking record {messageId} as failed based on return value", [
                        "method" => __METHOD__,
                        "messageId" => $record["messageId"],
                        "record" => $record,
                    ]);

                    if ($this->errorEventEmitter !== false) {
                        $this->errorEventEmitter->emit(
                            $this->eventHandler->getLastError(),
                            $record["messageId"],
                            $record["body"],
                        );
                    }

                    $failed[] = [
                        "itemIdentifier" => $record["messageId"],
                    ];
                }
            } catch (Throwable $t) {
                $this->log->error("Failed to process record {messageId}: {message}", [
                    "message" => $t->getMessage(),
                    "method" => __METHOD__,
                    "messageId" => $record["messageId"],
                    "record" => $record,
                    "exception" => $t,
                ]);

                if ($this->errorEventEmitter !== false) {
                    $this->errorEventEmitter->emit(
                        $t->getMessage(),
                        $record["messageId"],
                        $record["body"],
                    );
                }

                $failed[] = [
                    "itemIdentifier" => $record["messageId"],
                ];
            }
        }

        return [
            "batchItemFailures" => $failed,
        ];
    }

    /**
     * @param SqsRecord $record
     * @return array{decoded:ReceivedMessageInterface, meta:Metadata}
     */
    private function decodeMessage(array $record): array
    {
        $body = Json::decode($record["body"]);
        $version = (int)($body["version"] ?? 1);

        switch ($version) {
            case 1:
                return [
                    "decoded" => new ReceivedMessage($body),
                    "meta" => new Metadata(
                        correlationId: (string)($body["meta"]["correlationId"] ?? "<UNKNOWN>"),
                        sender: (string)($body["meta"]["sender"] ?? "<UNKNOWN>"),
                    ),
                ];
            default:
                throw new Exception(sprintf("Unsupported message version %d", $version));
        }
    }
}
